/* mbed PT6311 Library, for Princeton PT6311 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version for VFDEM2
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6311_H
#define PT6311_H

// Select one of the testboards for Princeton PT6311 VFD controller
#include "PT6311_Config.h"

/** An interface for driving Princeton PT6311 VFD controller
 *
 * @code
 *
 * #if (PT6311_TEST == 1)  
 * // Direct driving of PT6311 Test
 *
 * #include "mbed.h"
 * #include "PT6311.h" 
 * 
 * DisplayData_t size is 24 bytes (8 Grids @ 20 Segments) ... 48 bytes (16 Grids @ 12 Segments) 
 * DisplayData_t size default is 48 bytes (16 Grids @ 12 Segments) 
 * PT6311::DisplayData_t all_str  = {0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00,
 *                                   0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00};  
 *
 * // KeyData_t size is 6 bytes  
 * PT6311::KeyData_t keydata; 
 *
 * // PT6311 declaration, Default setting 16 Grids @ 12 Segments
 * PT6311 PT6311(p5,p6,p7, p8);
 *
 * int main() {
 *   PT6311.cls(); 
 *   PT6311.writeData(all_str);
 *   wait(4);
 *   PT6311.setBrightness(PT6311_BRT0);
 *   wait(1);
 *   PT6311.setBrightness(PT6311_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (PT6311.getKeys(&keydata)) {
 *      pc.printf("Keydata 0..5 = 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n", keydata[0], keydata[1], keydata[2], keydata[3], keydata[4], keydata[5]);
 *
 *      if (keydata[0] == 0x10) { //sw2   
 *        PT6311.cls(); 
 *        wait(1);
 *        PT6311.writeData(all_str);
 *      }  
 *    } 
 *   }   
 * }
 * #endif
 *
 * @endcode
 */

//PT6311 Display and Keymatrix data
#define PT6311_MAX_NR_GRIDS   16
#define PT6311_BYTES_PER_GRID  3
//Significant bits Keymatrix data
#define PT6311_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define PT6311_DISPLAY_MEM  (PT6311_MAX_NR_GRIDS * PT6311_BYTES_PER_GRID)
#define PT6311_KEY_MEM         6

//Reserved bits for commands
#define PT6311_CMD_MSK      0xC0

//Mode setting command
#define PT6311_MODE_SET_CMD 0x00
#define PT6311_GR8_SEG20    0x00
#define PT6311_GR9_SEG19    0x08
#define PT6311_GR10_SEG18   0x09
#define PT6311_GR11_SEG17   0x0A
#define PT6311_GR12_SEG16   0x0B
#define PT6311_GR13_SEG15   0x0C
#define PT6311_GR14_SEG14   0x0D
#define PT6311_GR15_SEG13   0x0E
#define PT6311_GR16_SEG12   0x0F //default

//Data setting commands
#define PT6311_DATA_SET_CMD 0x40
#define PT6311_DATA_WR      0x00
#define PT6311_LED_WR       0x01
#define PT6311_KEY_RD       0x02
#define PT6311_SW_RD        0x03
#define PT6311_ADDR_INC     0x00
#define PT6311_ADDR_FIXED   0x04
#define PT6311_MODE_NORM    0x00
#define PT6311_MODE_TEST    0x08

//LED settings data
#define PT6311_LED_MSK      0x1F
#define PT6311_LED1         0x01
#define PT6311_LED2         0x02
#define PT6311_LED3         0x04
#define PT6311_LED4         0x08
#define PT6311_LED5         0x10

//Switch settings data
#define PT6311_SW_MSK       0x0F
#define PT6311_SW1          0x01
#define PT6311_SW2          0x02
#define PT6311_SW3          0x04
#define PT6311_SW4          0x08

//Address setting commands
#define PT6311_ADDR_SET_CMD 0xC0
#define PT6311_ADDR_MSK     0x3F

//Display control commands
#define PT6311_DSP_CTRL_CMD 0x80
#define PT6311_BRT_MSK      0x07
#define PT6311_BRT0         0x00 //Pulsewidth 1/16, Default
#define PT6311_BRT1         0x01
#define PT6311_BRT2         0x02
#define PT6311_BRT3         0x03
#define PT6311_BRT4         0x04
#define PT6311_BRT5         0x05
#define PT6311_BRT6         0x06
#define PT6311_BRT7         0x07 //Pulsewidth 14/16

#define PT6311_BRT_DEF      PT6311_BRT3

#define PT6311_DSP_OFF      0x00 //Default
#define PT6311_DSP_ON       0x08


/** A class for driving Princeton PT6311 VFD controller
 *
 * @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
 *        SPI bus interface device. 
 */
class PT6311 {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid8_Seg20  = PT6311_GR8_SEG20,
    Grid9_Seg19  = PT6311_GR9_SEG19,
    Grid10_Seg18 = PT6311_GR10_SEG18,
    Grid11_Seg17 = PT6311_GR11_SEG17,
    Grid12_Seg16 = PT6311_GR12_SEG16,
    Grid13_Seg15 = PT6311_GR13_SEG15,
    Grid14_Seg14 = PT6311_GR14_SEG14,
    Grid15_Seg13 = PT6311_GR15_SEG13,   
    Grid16_Seg12 = PT6311_GR16_SEG12   
  };
 
  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[PT6311_DISPLAY_MEM];
  typedef char KeyData_t[PT6311_KEY_MEM];
    
 /** Constructor for class for driving Princeton PT6311 VFD controller
  *
  *  @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
  *         SPI bus interface device. 
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  *  @param  Mode selects either number of Grids and Segments (default 16 Grids, 12 Segments)
  */
  PT6311(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode=Grid16_Seg12);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to PT6311
   *  @param  int address display memory location to write byte
   *  @param  char data byte written at given address
   *  @return none
   */ 
  void writeData(int address, char data); 
 
  /** Write Display datablock to PT6311
   *  @param  DisplayData_t data Array of PT6311_DISPLAY_MEM (=48) bytes for displaydata (starting at address 0)
   *  @param  length number bytes to write (valid range 0..PT6311_DISPLAY_MEM (=48), starting at address 0)   
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = PT6311_DISPLAY_MEM);


  /** Read keydata block from PT6311
   *  @param  *keydata Ptr to Array of PT6311_KEY_MEM (=6) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the PT6311 key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);


  /** Read switches from PT6311
   *
   *  @param  none
   *  @return char for switch data (4 least significant bits)
   *
   */   
  char getSwitches();

  /** Set LEDs
    *
    * @param  char leds (5 least significant bits)  
    * @return none
    */
  void setLED (char leds = 0);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = PT6311_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  char _display;
  char _bright; 
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The PT6311 expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write command and parameter to PT6311
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
  void _writeCmd(int cmd, int data);  
};



#if (VFDEM2_TEST == 1)
// Derived class for VFDEM2 front display unit
//   Grids 2-11 all display 14-Segment digits and several Icons.
//   Grid 1 and 12 display Icons. 

//
#include "Font_16Seg.h"

//VFDEM2 Display data
#define VFDEM2_NR_GRIDS  12
#define VFDEM2_NR_DIGITS 10
#define VFDEM2_NR_UDC    8

//VFDEM2 Memory size in bytes for Display
#define VFDEM2_DISPLAY_MEM (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID)


/** Constructor for class for driving Princeton PT6311 VFD controller as used in VFDEM2
  *
  *  @brief Supports 12 Grids of 16 Segments and Icons (10 digits of 14 segments plus some icons and another 2 Icon grids).
  *         Also supports a scanned keyboard of 7 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class PT6311_VFDEM2 : public PT6311, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  enum Icon {
    CIR = (1<<24) | S16_CIR,
    PIE_R = (1<<24) | S16_PIE_R,
    PIE_G = (1<<24) | S16_PIE_G,
    PIE_B = (1<<24) | S16_PIE_B,    
    STP = (1<<24) | S16_STP,
    PSE = (1<<24) | S16_PSE,
    PLY = (1<<24) | S16_PLY,
    RR  = (1<<24) | S16_RR,
    LL  = (1<<24) | S16_LL,
    PCM = (1<<24) | S16_PCM,
    DTS = (1<<24) | S16_DTS,
    MIC = (1<<24) | S16_MIC,
    DLB = (1<<24) | S16_DLB,

    REC = (2<<24) | S16_REC,

    PRG = (3<<24) | S16_PRG,    
    RND = (3<<24) | S16_RND,

    DP8 =  (4<<24) | S16_DP8,        
    COL8 = (4<<24) | S16_COL8,

    ANG = (5<<24) | S16_ANG,        
    ZM  = (5<<24) | S16_ZM,

    PBC =  (6<<24) | S16_PBC,        
    COL6 = (6<<24) | S16_COL6,

    MP3 =  (12<<24) | S16_MP3,
    CDDA = (12<<24) | S16_CDDA,
    SS  = (12<<24) | S16_SS,
    VCD = (12<<24) | S16_VCD,
    DVD = (12<<24) | S16_DVD,   
    ARW = (12<<24) | S16_ARW,
    ONE = (12<<24) | S16_ONE,    
    ALL = (12<<24) | S16_ALL,
    AA  = (12<<24) | S16_AA,    
    BB  = (12<<24) | S16_BB,
    TTL = (12<<24) | S16_TTL,
    CHP = (12<<24) | S16_CHP       
  };
  
  typedef short UDCData_t[VFDEM2_NR_UDC];


/** Constructor for class for driving Princeton PT6311 VFD controller as used in VFDEM2
  *
  *  @brief Supports 12 Grids of 16 Segments and Icons (10 digits of 14 Segments plus some icons and another 2 Icon grids).
  *         Also supports a scanned keyboard of 7 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
  PT6311_VFDEM2(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to PT6311
     *  @param  int address display memory location to write byte
     *  @param  char data byte written at given address
     *  @return none
     */ 
    void writeData(int address, char data){
      PT6311::writeData(address, data);
    }        
 
   /** Write Display datablock to PT6311
    *  @param  DisplayData_t data Array of PT6311_DISPLAY_MEM (=48) bytes for displaydata (starting at address 0)
    *  @param  length number bytes to write (valid range 0..(VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID) == 36, starting at address 0)   
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID)) {
      PT6311::writeData(data, length);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;                     // Current cursor location
    int _columns;                    // Max number of columns
    
    DisplayData_t _displaybuffer;    // Local mirror for all chars and icons
    UDCData_t _UDC_16S;              // User Defined Character pattterns (UDC)    
};
#endif

#endif
