/* mbed PT6311 Library, for Princeton PT6311 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version, for VFDEM2 code
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "PT6311.h"

/** Constructor for class for driving Princeton PT6311 VFD controller
 *
 *  @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
 *         SPI bus interface device. 
 *   
 *  @param PinName mosi, miso, sclk, cs SPI bus pins
 *  @param Mode selects either number of Digits and Segments
*/
PT6311::PT6311(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode) : _spi(mosi,miso,sclk), _cs(cs), _mode(mode) {

  _init();
}

/** Init the SPI interface and the controller
  * @param  none
  * @return none
  */ 
void PT6311::_init(){
  
//init SPI
  _cs=1;
  _spi.format(8,3); //PT6311 uses mode 3 (Clock High on Idle, Data latched on second (=rising) edge)
//  _spi.frequency(100000);   
  _spi.frequency(500000);     

//init controller  
  _writeCmd(PT6311_MODE_SET_CMD, _mode);                                               // Mode set command

  _display = PT6311_DSP_ON;
  _bright  = PT6311_BRT_DEF; 
  _writeCmd(PT6311_DSP_CTRL_CMD, _display | _bright );                                 // Display control cmd, display on/off, brightness   
  
  _writeCmd(PT6311_DATA_SET_CMD, PT6311_DATA_WR | PT6311_ADDR_INC | PT6311_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}   


/** Clear the screen and locate to 0
 */  
void PT6311::cls() {
  
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6311_ADDR_SET_CMD | 0x00)); // Address set cmd, 0
      
  for (int cnt=0; cnt<PT6311_DISPLAY_MEM; cnt++) {
    _spi.write(0x00); // data 
  }
  
  wait_us(1);
  _cs=1;      
  
}  

/** Set Brightness
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void PT6311::setBrightness(char brightness){

  _bright = brightness & PT6311_BRT_MSK; // mask invalid bits
  
  _writeCmd(PT6311_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness  

}

/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void PT6311::setDisplay(bool on) {
  
  if (on) {
    _display = PT6311_DSP_ON;
  }
  else {
   _display = PT6311_DSP_OFF;
  }
  
  _writeCmd(PT6311_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness   
}

/** Write databyte to PT6311
  *  @param  int address display memory location to write byte
  *  @param  char data byte written at given address
  *  @return none
  */ 
void PT6311::writeData(int address, char data) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6311_ADDR_SET_CMD | (address & PT6311_ADDR_MSK))); // Set Address cmd
      
  _spi.write(_flip(data)); // data 
  
  wait_us(1);
  _cs=1;         
    
}

/** Write Display datablock to PT6311
  *  @param  DisplayData_t data Array of PT6311_DISPLAY_MEM (=48) bytes for displaydata (starting at address 0)
  *  @param  length number bytes to write (valid range 0..PT6311_DISPLAY_MEM (=48), starting at address 0)   
  *  @return none
  */    
void PT6311::writeData(DisplayData_t data, int length) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6311_ADDR_SET_CMD | 0x00)); // Set Address at 0
      
// sanity check
  if (length < 0) {length = 0;}
  if (length > PT6311_DISPLAY_MEM) {length = PT6311_DISPLAY_MEM;}

//  for (int idx=0; idx<PT6311_DISPLAY_MEM; idx++) {  
  for (int idx=0; idx<length; idx++) {    
    _spi.write(_flip(data[idx])); // data 
  }
  
  wait_us(1);
  _cs=1;         
    
}

/** Read keydata block from PT6311
  *  @param  *keydata Ptr to Array of PT6311_KEY_MEM (=6) bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  *
  * Note: Due to the hardware configuration the PT6311 key matrix scanner will detect multiple keys pressed at same time,
  *       but this may also result in some spurious keys being set in keypress data array.
  *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.  
  */ 
bool PT6311::getKeys(KeyData_t *keydata) {
  int keypress = 0;
  char data;

  // Read keys
  _cs=0;
  wait_us(1);    
  
  // Enable Key Read mode
  _spi.write(_flip(PT6311_DATA_SET_CMD | PT6311_KEY_RD | PT6311_ADDR_INC | PT6311_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  for (int idx=0; idx < PT6311_KEY_MEM; idx++) {
    data = _flip(_spi.write(0xFF));    // read keys and correct bitorder
    
    data = data & PT6311_KEY_MSK; // Mask valid bits
    if (data != 0) {  // Check for any pressed key
      for (int bit=0; bit < 8; bit++) {
        if (data & (1 << bit)) {keypress++;} // Test all significant bits
      }
    }  

    (*keydata)[idx] = data;            // Store keydata after correcting bitorder
  }

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6311_DATA_SET_CMD, PT6311_DATA_WR | PT6311_ADDR_INC | PT6311_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
#if(1)
// Dismiss multiple keypresses at same time
  return (keypress == 1);    
#else
// Allow multiple keypress and accept possible spurious keys
  return (keypress > 0);
#endif  
}


/** Read switches from PT6311
  *
  *  @param  none
  *  @return char for switch data (4 least significant bits) 
  *
  */   
char PT6311::getSwitches() {
  char data;

  // Read switches
  _cs=0;
  wait_us(1);    
  
  // Enable Switch Read mode
  _spi.write(_flip(PT6311_DATA_SET_CMD | PT6311_SW_RD | PT6311_ADDR_INC | PT6311_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  data = _flip(_spi.write(0xFF)) & PT6311_SW_MSK;   // read switches and correct bitorder

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6311_DATA_SET_CMD, PT6311_DATA_WR | PT6311_ADDR_INC | PT6311_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
  return data;       
}
   

/** Set LEDs
  *
  * @param  char leds (5 least significant bits)  
  * @return none
  */
void  PT6311::setLED (char leds) {

  // Set LEDs
  _cs=0;
  wait_us(1);    
  
  // Enable LED Write mode
  _spi.write(_flip(PT6311_DATA_SET_CMD | PT6311_LED_WR | PT6311_ADDR_INC | PT6311_MODE_NORM)); // Data set cmd, normal mode, auto incr, write data

  _spi.write(_flip(leds & PT6311_LED_MSK));    // write LEDs in correct bitorder

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6311_DATA_SET_CMD, PT6311_DATA_WR | PT6311_ADDR_INC | PT6311_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}



/** Helper to reverse all command or databits. The PT6311 expects LSB first, whereas SPI is MSB first
  *  @param  char data
  *  @return bitreversed data
  */ 
char PT6311::_flip(char data) {
 char value=0;
  
 if (data & 0x01) {value |= 0x80;} ;  
 if (data & 0x02) {value |= 0x40;} ;
 if (data & 0x04) {value |= 0x20;} ;
 if (data & 0x08) {value |= 0x10;} ;
 if (data & 0x10) {value |= 0x08;} ;
 if (data & 0x20) {value |= 0x04;} ;
 if (data & 0x40) {value |= 0x02;} ;
 if (data & 0x80) {value |= 0x01;} ;
 return value;       
}


/** Write command and parameter to PT6311
  *  @param  int cmd Command byte
  *  &Param  int data Parameters for command
  *  @return none
  */  
void PT6311::_writeCmd(int cmd, int data){
    
  _cs=0;
  wait_us(1);    
 
  _spi.write(_flip( (cmd & PT6311_CMD_MSK) | (data & ~PT6311_CMD_MSK)));   
 
  wait_us(1);
  _cs=1;      
    
};  




#if (VFDEM2_TEST == 1)
/** Constructor for class for driving Princeton PT6311 VFD controller as used in VFDEM2
  *
  *  @brief Supports 12 Grids of 16 Segments and Icons (10 digits of 14 Segments plus some icons and another 2 Icon grids).  
  *         Also supports a scanned keyboard of 7 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */  
PT6311_VFDEM2::PT6311_VFDEM2(PinName mosi, PinName miso, PinName sclk, PinName cs) : PT6311(mosi, miso, sclk, cs, Grid12_Seg16) {
  _column  = 0;
  _columns = VFDEM2_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  */
void PT6311_VFDEM2::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int PT6311_VFDEM2::columns() {
    return _columns;
}

    
/** Clear the screen and locate to 0
  * @param bool clrAll Clear Icons also (default = false)
  */ 
void PT6311_VFDEM2::cls(bool clrAll) {  
  int idx;
  
  if (clrAll) {
    //clear local buffer (including Icons)
    for (idx=0; idx < (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID); idx++) {
      _displaybuffer[idx] = 0x00;  
    }
  }  
  else {
    //clear local buffer (preserving Icons)
    for (int grd=0; grd < VFDEM2_NR_GRIDS; grd++) {
      idx = grd * PT6311_BYTES_PER_GRID; // 3 bytes for every Grid
      _displaybuffer[idx    ] = _displaybuffer[idx    ] & MASK_ICON_GRID[grd][0];  
      _displaybuffer[idx + 1] = _displaybuffer[idx + 1] & MASK_ICON_GRID[grd][1];
      _displaybuffer[idx + 2] = _displaybuffer[idx + 2] & MASK_ICON_GRID[grd][2];      
    }  
  }

  writeData(_displaybuffer, (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID));

  _column = 0;   
}    

/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void PT6311_VFDEM2::setIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) * PT6311_BYTES_PER_GRID;  // 3 Bytes for every Grid
    
  //Save char...and set bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] | LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] | MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] | HI(icn);        
  writeData(_displaybuffer, (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID));
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void PT6311_VFDEM2::clrIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr -1 ) * PT6311_BYTES_PER_GRID;   // 3 Bytes for every Grid
    
  //Save char...and clr bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] & ~LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] & ~MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] & ~HI(icn);        
  writeData(_displaybuffer, (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID));
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (16 bits)       
  */
void PT6311_VFDEM2::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (VFDEM2_NR_UDC-1)) {
    return;
  }

  // Mask out Icon bits?
  _UDC_16S[udc_idx] = udc_data & 0xFFFF;
}

/** Write a single character (Stream implementation)
  */
int PT6311_VFDEM2::_putc(int value) {
    bool validChar = false;
    short pattern  = 0x0000;
    int addr;
    
    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    }
    else if ((value >= 0) && (value < VFDEM2_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_16S[value];
    }  
#if (SHOW_ASCII == 1)
    //display all ASCII characters
    else if ((value >= FONT_16S_START) && (value <= FONT_16S_END)) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[value - FONT_16S_START];
    } // else
#else
    //display only digits and hex characters
    else if (value == '-') {
      //Character to write
      validChar = true;
      pattern = C16_MIN;         
    }
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[10 + value - (int) 'a'];
    } //else
#endif
    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      //Note that the VFDEM2 has ten 14 Segment digits/grids. 
      //Some of these Grids also have icons that need to be preserved
      //_column ==  0 => Grid11 => addr = 30
      //_column ==  1 => Grid10 => addr = 27
      // ....
      //_column ==  9 => Grid2  => addr = 3            
      addr = (10 - _column) * PT6311_BYTES_PER_GRID; // 3 Bytes for every Grid;
      
      //Save icons...and set bits for character to write
      _displaybuffer[addr]   = (_displaybuffer[addr]   & MASK_ICON_GRID[10 - _column][0]) | LO(pattern);
      _displaybuffer[addr+1] = (_displaybuffer[addr+1] & MASK_ICON_GRID[10 - _column][1]) | MD(pattern);
//      _displaybuffer[addr+2] = (_displaybuffer[addr+2] & MASK_ICON_GRID[10 - _column][2]) | HI(pattern);

      writeData(_displaybuffer, (VFDEM2_NR_GRIDS * PT6311_BYTES_PER_GRID));
                                
      //Update Cursor
      _column++;
      if (_column > (VFDEM2_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}

// get a single character (Stream implementation)
int PT6311_VFDEM2::_getc() {
    return -1;
}
#endif
