/* mbed PT6301 Library, for Princeton PT6301 VFD controller
 *             The controller is used by Futaba 'Chip In Glass' (CIG) VFD tubes. 
 *
 * Copyright (c) 2021, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6301_H
#define PT6301_H

// Select one of the testboards for Princeton PT6301 VFD controller
#include "PT6301_Config.h"
#include "PT6301_UDC.h"

/** An interface for driving Princeton PT6301 VFD controller
 *
 * @code
 *
 * #if (PT6301_TEST == 1)  
 * // Direct driving of PT6301 Test
 *
 * #include "mbed.h"
 * #include "PT6301.h" 
 *
 * DigitalOut myled(LED1);
 * Serial pc(USBTX, USBRX);
 * 
 * // PT6301 declaration, Default setting 2x20 Grids @ 5x7 Segments, 2 Icon segments for each Grid
 * PT6301 PT6301(p5, p7, p8, p9); // DI, CLK, CS, RST
 *
 * int main() {
 *   pc.printf("Hello World: PT6301 test\n\r");
 *
 *   PT6301.cls(); 
 *
 * //  PT6301.writeData((char)'H',0,0);
 * //  PT6301.writeData((char)'e',0,1);
 * //  PT6301.writeData((char)'l',0,2);
 * //  PT6301.writeData((char)'l',0,3);
 * //  PT6301.writeData((char)'o',0,4);
 *   PT6301.printf("Hello World");
 *   PT6301.refresh();  
 *   wait(2);
 *
 *   PT6301.setBrightness(PT6301_BRT0);
 *   wait(2);
 *   PT6301.setBrightness(PT6301_BRT3);
 *           
 *   while(1) {
 *     myled = !myled;
 *     wait(1);
 *   }
 * }
 * #endif
 *
 * @endcode
 */


//PT6301 Display and Annunciator data
#define PT6301_MAX_NR_GRIDS   20
#define PT6301_MAX_NR_ROWS     2
#define PT6301_BYTES_PER_GRID  1


//The PT6301 has internal memory for all characters and icons. The content is automatically displayed on the tube.
//The memory consists of two banks (row A and row B) for character memory and two banks (row A and row B) for icon memory. 
//Each of those banks is accessed by separate commands.
//However, these command do not support addressing individual locations in the memory. Memory updates always start at
//address 0 in the selected row A or B. Consequently, the whole displaymemory needs to be rewritten when any 
//location (except for 0) is to be updated. 
//The library therefor uses a local mirror memory to store the display content, update one or more characters in
//the mirrormemory as needed, and rewrite the whole displaymemory from the mirrorcontent. 
//The write-back is performed by calling the 'refresh' method.
//Additional advantage of the mirror-memory is that we can also implement wrap-around and scrolling from row A to B for multi-line displays.
   
//Memory size in bytes for Display and Annunciators in total
//#define PT6301_DSP_MEM        (PT6301_MAX_NR_ROWS * PT6301_MAX_NR_GRIDS * PT6301_BYTES_PER_GRID)
//#define PT6301_ADD_MEM        (PT6301_MAX_NR_ROWS * PT6301_MAX_NR_GRIDS)
//Memory size in bytes for Display and Annunciators per Row
#define PT6301_DSP_MEM        (PT6301_MAX_NR_GRIDS * PT6301_BYTES_PER_GRID)
#define PT6301_ADD_MEM        (PT6301_MAX_NR_GRIDS * PT6301_BYTES_PER_GRID)

//User Defined Characters
//Note that the PT6301 actually has two separate UDC banks for row A and B. 
//In this lib both UDC memories are always identical.
#define PT6301_UDC_MEM        16

//SPI Serial control data consists of an 8-bit command and one or more data bytes.
//Command and data are sent LSB first and latched on rising edge of CLK. Idle CLK is high.
//Data address is auto incremented.
//The commands and data are transmitted during CE low and latched on rising CE edge.
//Multiple PT6301 devices on the same bus can only be distinguised by the CE control.
//Normal SPI methods send data MSB first. This lib uses a helper method to flip the bits before transmission.

//Command delay in us
#define PT6301_CMD_DLY         8
#define PT6301_CS_DLY         16
//Reset delay in ms
#define PT6301_RST_DLY        10

//
//Set Char data command (DCRAM_A)
// 0 0 0 1 x x x x
#define PT6301_DATA_A_REG     0x10
//Set Char data command (DCRAM_B)
// 1 0 0 1 x x x x
#define PT6301_DATA_B_REG     0x90


//Note: The PT6301 does not support address selection. The commandformat allows 
//only 4bits for the address anyhow and this would be insufficient for a display
//controller that can support upto 20 Grids.
//Some other controllers with very similar commandsets and a max of 16 Grids do use the available 4 addressbits.
//
//No DCRAM Address Support, Always starts at address 0x00
#define PT6301_DADR_MSK       0x0F

//Char data (2nd byte, 3rd byte ...)
//DA7..DA0 Character Data
#define PT6301_DATA_MSK       0xFF

//Note: The PT6301 supports separate UDCs banks for both rows A and B.
//However, this lib always keeps both UDC banks identical.

//Set UDC data command (CGRAM_A)
// 0 0 1 0 UA3 UA2 UA1 UA0
#define PT6301_UDC_A_REG      0x20
//Set UDC data command (CGRAM_B)
// 1 0 1 0 UA3 UA2 UA1 UA0
#define PT6301_UDC_B_REG      0xA0

//UA3..UA0 CGRAM Address (UDC RAM address)
#define PT6301_UADR_MSK       0x0F
#define PT6301_NR_UDC           16

//User Defined Characters (UDCs) consist of 5x7 dots and are defined by a 5 byte bitpattern.
//UDC data (2nd byte .. 6th byte)
//    D7 D6   D5   D4.. D1 D0 
// 0  *  CD30 CD25 ......  CD0
// 1  *  CD31 CD26 ......  CD1
// 2  *  CD32 CD27 ......  CD2
// 3  *  CD33 CD28 ......  CD3
// 4  *  CD34 CD29 ......  CD4
//
#define PT6301_UDC_MSK        0x7F

//CD34..CD0 UDC Data
//UDC is a 5x7 Matrix pattern that will show on the VFD as
// 0   C0  C1  C2  C3  C4
// 1   C5  C6 .......  C9
// .    .............
// .    .............
// .    .............
// 6   C30 C31 ...     C34
//

//UDCs are defined by a 5x7 matrix and stored as 5 bytes
typedef char UDCData_t[5];


//Set Additional data command (ADRAM_A), Used for annunciators etc
// 0 0 1 1 x x x x
#define PT6301_ADAT_A_REG     0x30
//Set Additional data command (ADRAM_B), Used for annunciators etc
// 1 0 1 1 x x x x
#define PT6301_ADAT_B_REG     0xB0

//Note: The PT6301 does not support address selection for icons. The commandformat allows 
//only 4bits for the address anyhow and this would be insufficient for a display with upto 20 Grids
//
//No ADRAM Address (Additional data), always starts at 0x00
#define PT6301_AADR_MSK       0x0F

//* * * * * * * AD0 Additional Data (2nd byte, 3rd byte, ..)
#define PT6301_ADAT_MSK       0x01

//
//Set Brightness command
// 0 1 0 1 * * DC1 DC0
#define PT6301_BRT_REG        0x50
#define PT6301_BRT_MSK        0x03

//Set Brightness command (2nd Byte)
// DC9 DC8 DC7 DC6 DC5 DC4 DC3 DC2


//Brightness Level (0..7), mapped onto Brightness command 2nd databyte
//Note Brightness relationship between the number of active Grids (period) and the BRT value (duty cycle)
#define PT6301_BRT_0          0x00   //Duty  0/1024 (Default)
#define PT6301_BRT_1          0x20
#define PT6301_BRT_2          0x40
#define PT6301_BRT_3          0x80
#define PT6301_BRT_4          0xA0
#define PT6301_BRT_5          0xB0
#define PT6301_BRT_6          0xD0
#define PT6301_BRT_7          0xFF   //Duty 960/1024

#define PT6301_BRT_DEF       (PT6301_BRT_2)


//
//Grid control command
// 0 1 1 0 GN3 GN2 GN1 GN0
#define PT6301_GRID_REG       0x60
#define PT6301_GRID_MSK       0x0F

//Grids. Each Grid controls 2 characters (A and B) and 2 icons (A and B)
//
// GN3 GN2 GN1 GN0
//  0   0   0   0 G1 to G20  // Default
//  0   0   0   1 G1 to G5
//  0   0   1   0 G1 to G6
//  0   0   1   1 G1 to G7
//...
//  1   0   0   0 G1 to G12
//  1   0   0   1 G1 to G13
//  1   0   1   0 G1 to G14
//...
//  1   1   1   1 G1 to G19
//
#define PT6301_GR1_GR5        0x01
#define PT6301_GR1_GR6        0x02
#define PT6301_GR1_GR7        0x03
#define PT6301_GR1_GR8        0x04
#define PT6301_GR1_GR9        0x05
#define PT6301_GR1_GR10       0x06
#define PT6301_GR1_GR11       0x07
#define PT6301_GR1_GR12       0x08
#define PT6301_GR1_GR13       0x09
#define PT6301_GR1_GR14       0x0A
#define PT6301_GR1_GR15       0x0B
#define PT6301_GR1_GR16       0x0C
#define PT6301_GR1_GR17       0x0D
#define PT6301_GR1_GR18       0x0E
#define PT6301_GR1_GR19       0x0F
#define PT6301_GR1_GR20       0x00

//
//Display On/Off command
// 0 1 1 1 * * H L
#define PT6301_DSPL_REG       0x70
#define PT6301_DSPL_MSK       0x03

//Display Mode
// H L Display operating state
// 0 0 Normal display (default)
// 0 1 Off
// 1 0 All Segments and Additional Segments On
// 1 1 All Segments and Additional Segments On
#define PT6301_DSPL_NRM       0x00
#define PT6301_DSPL_OFF       0x01
#define PT6301_DSPL_ON        0x02


/** A class for driving Princeton PT6301 VFD controller
 *
 *  @brief Supports upto 20 Grids of 5x7 matrix segments for 2 rows of characters (A and B). 
 *         Also supports               1 additional segment for 2 rows (A and B).
 *         SPI bus interface device (LSB first). 
 */
class PT6301 : public Stream {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid5  = PT6301_GR1_GR5,
    Grid6  = PT6301_GR1_GR6,
    Grid7  = PT6301_GR1_GR7,
    Grid8  = PT6301_GR1_GR8,
    Grid9  = PT6301_GR1_GR9,
    Grid10 = PT6301_GR1_GR10,
    Grid11 = PT6301_GR1_GR11,
    Grid12 = PT6301_GR1_GR12,
    Grid13 = PT6301_GR1_GR13,
    Grid14 = PT6301_GR1_GR14,
    Grid15 = PT6301_GR1_GR15,
    Grid16 = PT6301_GR1_GR16,
    Grid17 = PT6301_GR1_GR17,
    Grid18 = PT6301_GR1_GR18,
    Grid19 = PT6301_GR1_GR19,
    Grid20 = PT6301_GR1_GR20
  };
 
 /** Datatypes for display data */
typedef char DisplayData_t[PT6301_MAX_NR_ROWS][PT6301_DSP_MEM];
typedef char DisplayAdd_t[PT6301_MAX_NR_ROWS][PT6301_ADD_MEM];  
    
 /** Constructor for class for driving Princeton PT6301 VFD controller
  *
  *  @brief Supports upto 20 Grids of 5x7 matrix segments for 2 rows of characters (row A and B). 
  *         Also supports               1 additional segment for 2 rows of characters (row A and B).  
  *         SPI bus interface device. 
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  PinName rst reset pin
  *  @param  Mode selects number of Grids and Segments (default 20 Grids, 5x7 matrix segments + 1 additional segment for 2 rows)
  *  @param  bool inverted_rows selects mapping of Data onto Display layout (default false)
  *  @param  Columns selects number of characters per row (default 20, same as Grids)
  *  @param  Rows selects number of rows (default 2)
  */
  PT6301(PinName mosi, PinName sclk, PinName cs, PinName rst, Mode mode = Grid20, bool inverted_rows = false, int columns = PT6301_MAX_NR_GRIDS, int rows = PT6301_MAX_NR_ROWS);
      
  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (dutycycle linked to number of grids)  
    * @return none
    */    
  void setBrightness(char brightness = PT6301_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    * @return none    
    */
  void setDisplay(bool on);

  /** Set the Display test mode On/off
    *
    * @param bool display test mode
    * @return none    
    */
  void setDisplayTest(bool on);

  /** Set User Defined Characters (UDC) for A and B row
    *
    * @param unsigned char udc_idx   The Index of the UDC (0..15)
    * @param UDCData_t udc_data      The bitpattern for the UDC (5 bytes)
    * @return none    
    */
  void setUDC(unsigned char udc_idx, UDCData_t udc_data);

  /** Set Icon
    *
    * @param int row    The row of the icon (0..(rows-1))
    * @param int column The column of the icon (0..(cols-1))    
    * @return none    
    */
  void setIcon(int row, int column);

  /** Clr Icon
    *
    * @param int row    The row of the icon (0..(rows-1))
    * @param int column The column of the icon (0..(cols-1))    
    * @return none    
    */
  void clrIcon(int row, int column);

  /** Locate cursor to a screen row, column
    *
    * @param row     The vertical position from the top, indexed from 0     
    * @param column  The horizontal position from the left, indexed from 0
    * @return none     
    */
  void locate(int row, int column);
    
  /** Clear the screen and locate to (0,0)
    *
    * @param bool clrAll Clear Icons also (default = true)
    */
  void cls(bool clrAll = true);
  
  /** Refresh screen and show data in local mirrors on the display
    *
    * @param bool copyAll Copy Icons in Adat local mirror also (default = true)       
    * @return none
    */
  void refresh(bool copyAll = true);  
  

  /** Number of screen columns
    *
    * @param none
    * @return columns
    */
  int columns();  

  /** Number of screen rows
    *
    * @param none
    * @return rows
    */
  int rows();  

  /** Write Data to local mirror
    *
    * @param char data The databyte        
    * @param row       The vertical position from the top, indexed from 0     
    * @param column    The horizontal position from the left, indexed from 0
    * @return none     
    */
  void setData(char data, int row, int column);

  /** Read Data from local mirror
    *
    * @param row       The vertical position from the top, indexed from 0     
    * @param column    The horizontal position from the left, indexed from 0
    * @return char     The databyte        
    */
  char getData(int row, int column);

  /** Write AData to local mirror
    *
    * @param char data The symbol databyte        
    * @param row       The vertical position from the top, indexed from 0     
    * @param column    The horizontal position from the left, indexed from 0
    * @return none     
    */
  void setAData(char data, int row, int column);

  /** Read AData from local mirror
    *
    * @param row       The vertical position from the top, indexed from 0     
    * @param column    The horizontal position from the left, indexed from 0
    * @return char     The symbol databyte        
    */
  char getAData(int row, int column);

  
 protected:  

  /** Write Command byte to PT6301
    *
    *  @param char cmd Command byte
    *  @return none
    */  
  void _writeCmd(char cmd);

  /** Write Command and Data byte to PT6301
    *
    *  @param char cmd Command byte
    *  @param char data Parameter for Command byte    
    *  @return none
    */  
  void _writeCmd(char cmd, char data);
  

  /** Helper to reverse all command or databits. The PT6301 expects LSB first, whereas SPI is MSB first
    *
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

/** Helper to reverse row idx depending on VFD layout
  *
  *  @param  int row_idx
  *  @return adjusted row_idx
  */ 
int _row_flip(int row_idx);


#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display RAM
     * @return char written
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();


 private:  
  SPI _spi;
  DigitalOut _cs;
  DigitalOut _rst;  
  Mode _mode;                      // Number of Grids
  bool _inverted_rows;             // Mapping of Rows to VFD layout

  int _column;                     // Current cursor location
  int _columns;                    // Max number of columns
  int _row;                        // Current cursor location
  int _rows;                       // Max number of rows
    
  DisplayData_t _displaybuffer;    // Local mirror for all chars
  DisplayAdd_t _addbuffer;         // Local mirror for all icons    

  /** Init the SPI interface and the controller
    *
    * @param  none
    * @return none
    */ 
  void _init();

};


#if (SMTG7400_TEST == 1)
// Derived class for SMTG7400 display unit
//   Grids 1-16, 1 row of 16 matrix characters (5x7 segments), 4 Additional segments in use.
//

//SMTG7400 Display data
#define SMTG7400_NR_GRIDS 16
#define SMTG7400_NR_COLS  16
#define SMTG7400_NR_ROWS   1

//#define SMTG7400_NR_UDC   16

//SMTG7400 Icon data,
//#defines encode the row and position in the row for each icon
#define SMTG7400_ICON_ROW_SHFT  8
#define SMTG7400_ICON_ROW_0    (0x00 << SMTG7400_ICON_ROW_SHFT)
#define SMTG7400_ICON_ROW_1    (0x01 << SMTG7400_ICON_ROW_SHFT)
#define SMTG7400_ICON_COL_MSK  (0x00FF)
//
#define SMTG7400_ICON_OFFLINE  (SMTG7400_ICON_ROW_0 | 1)
#define SMTG7400_ICON_WIFI     (SMTG7400_ICON_ROW_0 | 2)
#define SMTG7400_ICON_PHONE    (SMTG7400_ICON_ROW_0 | 3)
#define SMTG7400_ICON_REC      (SMTG7400_ICON_ROW_0 | 8)

/** Constructor for class for driving Princeton PT6301 VFD controller as used in SMTG7400
  *
  *  @brief Supports 16 Grids of 5x7 Segments with 4 additional Segments in use.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  PinName rst reset pin  
  */
class PT6301_SMTG7400 : public PT6301{
 public:

/** Constructor for class for driving Princeton PT6301 VFD controller as used in SMTG7400
  *
  *  @brief Supports 16 Grids of 5x7 Segments with 4 additional Segments in use.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  PinName rst reset pin  
  */
  PT6301_SMTG7400(PinName mosi, PinName sclk, PinName cs, PinName rst);

 /** Set Icon
    *
    * @param int icon   The icon ID
    * @return none    
    */
  void setIcon(int icon);

  /** Clr Icon
    *
    * @param int icon   The icon ID
    * @return none    
    */
  void clrIcon(int icon);

protected:  

private:

};
#endif

#if (SMTC7140_TEST == 1)
// Derived class for SMTC7140 display unit
//   Grids 1-12, with 2 rows of 12 matrix characters (35 segments), no Additional segments.
//   Grid    13 is used for icons displayed as a UDC symbol.
//

//SMTC7140 Display data
#define SMTC7140_NR_GRIDS  13
#define SMTC7140_NR_COLS   12
#define SMTC7140_NR_ROWS    2

//#define SMTC7140_NR_UDC    16
#define SMTC7140_ICON_DOLBY     (1)
#define SMTC7140_ICON_PLUS      (2)
#define SMTC7140_ICON_HD        (3)
#define SMTC7140_ICON_CLOCK     (4)
#define SMTC7140_ICON_REC       (5)
#define SMTC7140_ICON_MAIL      (6)

const char SMTC7140_ICONS[][5] = {{0x7F, 0x7F, 0x7F, 0x7F, 0x7F},   // All On
                                  {0x00, 0x00, 0x08, 0x00, 0x00},   // Dolby
                                  {0x00, 0x04, 0x00, 0x00, 0x00},   // +
                                  {0x00, 0x00, 0x00, 0x08, 0x00},   // HD
                                  {0x00, 0x00, 0x00, 0x00, 0x10},   // Clock
                                  {0x20, 0x00, 0x00, 0x00, 0x00},   // Rec
                                  {0x00, 0x00, 0x00, 0x00, 0x40},   // Mail
                                  {0x00, 0x00, 0x00, 0x00, 0x00},   // 
                                }; 
                                
/** Constructor for class for driving Princeton PT6301 VFD controller as used in SMTC7140
  *
  *  @brief Supports 12 Grids of 5x7 Segments without additional Icon Segments, for 2 Rows.
  *                  Grid13 is used for icons displayed by a UDC symbol. 
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  PinName rst reset pin  
  */
class PT6301_SMTC7140 : public PT6301 {
 public:

/** Constructor for class for driving Princeton PT6301 VFD controller as used in SMTC7140
  *
  *  @brief Supports 12 Grids with 2 rows of 12 matrix characters (35 Segments), without additional Segments.
  *                  Grid 13 is used for icons displayed by a UDC symbol.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  PinName rst reset pin  
  */
  PT6301_SMTC7140(PinName mosi, PinName sclk, PinName cs, PinName rst);


/** Set VFD VGen
  *
  * @param  bool on
  * @return none
  */
  void setVGen (bool on = true);     

/** Set IconGrid13
  * Icons are shown on Grid13 using the UDC at index=0. The UDC char=0 is stored in _displaybuffer[0][12] at reset.
  * This method will set the correct segment in the UDC for each icon.
  *
  * @param int icon   The icon ID
  * @return none    
  */
  void setIconGrid13(int icon);

/** Clr IconGrid13
  * Icons are shown on Grid13 using the UDC at index=0. The UDC char=0 is stored in _displaybuffer[0][12] at reset.
  * This method will clr the correct segment in the UDC for each icon. 
  *
  * @param int icon   The icon ID
  * @return none    
  */
  void clrIconGrid13(int icon);

protected:  

private:
  UDCData_t _icon_data; // Icon data is stored as UDC_idx=0 and displayed on position (0, 12)

};
#endif

#endif