/* mbed PT6302 Library, for Princeton PT6302 VFD controller
 * Note the PT6302 is identical to the OKI ML9208 
 *
 * Copyright (c) 2017, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6302_H
#define PT6302_H

// Select one of the testboards for Princeton PT6302 VFD controller
#include "PT6302_Config.h"
#include "PT6302_UDC.h"

/** An interface for driving Princeton PT6302 VFD controller
 *
 * @code
 *
 * #if (PT6302_TEST == 1)  
 * // Direct driving of PT6302 Test
 *
 * #include "mbed.h"
 * #include "PT6302.h" 
 *
 * DigitalOut myled(LED1);
 * Serial pc(USBTX, USBRX);
 * 
 * // PT6302 declaration, Default setting 16 Grids @ 35 Segments
 * PT6302 PT6302(p5, p7, p8); // DI, CLK, CS
 *
 * int main() {
 *   pc.printf("Hello World: PT6302 test\n\r");
 *
 *   PT6302.cls(); 
 *
 *   PT6302.writeData((char)'H', 9);
 *   PT6302.writeData((char)'e', 8);
 *   PT6302.writeData((char)'l', 7);
 *   PT6302.writeData((char)'l', 6);
 *   PT6302.writeData((char)'o', 5);
 *   wait(2);
 *   PT6302.setBrightness(PT6302_BRT0);
 *   wait(2);
 *   PT6302.setBrightness(PT6302_BRT3);
 *           
 *   while(1) {
 *     myled = !myled;
 *     wait(1);
 *   }
 * }
 * #endif
 *
 * @endcode
 */


//PT6302 Display and Annunciator data
#define PT6302_MAX_NR_GRIDS   16
#define PT6302_BYTES_PER_GRID  1

//Memory size in bytes for Display and Annunciators
#define PT6302_DSP_MEM        16
#define PT6302_ADD_MEM        16
//#define PT6302_UDC_MEM         8

//SPI Serial control data consists of an 8-bit command and one or more data bytes.
//Command and data are sent LSB first and latched on rising edge of CLK. Idle CLK is high.
//Data address is auto incremented.
//The commands and data are transmitted during CE low and latched on rising CE edge.
//Multiple PT6302 devices on the same bus can only be distinguised by the CE control.

//Command delay
#define PT6302_CMD_DLY         8
#define PT6302_CS_DLY         16

//
//Set Char data command (DCRAM)
// 0 0 0 1 DA3 DA2 DA1 DA0
#define PT6302_DATA_REG       0x10

//DA3..DA0 DCRAM Address
#define PT6302_DADR_MSK       0x0F

//Char data (2nd byte, 3rd byte ...)
//DA7..DA0 Character Data
#define PT6302_DATA_MSK       0xFF


//
//Set UDC data command (CGRAM)
// 0 0 1 0 * UA2 UA1 UA0
#define PT6302_UDC_REG        0x20

//UA2..UA0 CGRAM Address (UDC RAM address)
#define PT6302_UADR_MSK       0x07
#define PT6302_NR_UDC            8

//User Defined Characters (UDCs) consist of 5x7 dots and are defined by a 5 byte bitpattern.
//UDC data (2nd byte .. 6th byte)
//    D7 D6   D5   D4.. D1 D0 
// 0  *  CD30 CD25 ......  CD0
// 1  *  CD31 CD26 ......  CD1
// 2  *  CD32 CD27 ......  CD2
// 3  *  CD33 CD28 ......  CD3
// 4  *  CD34 CD29 ......  CD4
//
#define PT6302_UDC_MSK        0x7F

//CD34..CD0 UDC Data
//UDC is a 5x7 Matrix pattern that will show on the VFD as
// 0   C0  C1  C2  C3  C4
// 1   C5  C6 .....    C9
// .    .............
// .    .............
// .    .............
// 6   C30 C31 ...     C34
//

//UDCs are defined by a 5x7 matrix and stored as 5 bytes
typedef char UDCData_t[5];


//
//Set Additional data command (ADRAM), Used for annunciators etc
// 0 0 1 1 AA3 AA2 AA1 AA0
#define PT6302_ADAT_REG      0x30

//AA3..AA0 ADRAM Address (Additional data)
#define PT6302_AADR_MSK      0x0F

//* * * * * * AD1 AD0 Additional Data (2nd byte, 3rd byte, ..)
#define PT6302_ADAT_MSK      0x03


//
//Set Port data command (General output)
// 0 1 0 0 * * P1 P0
#define PT6302_PDAT_REG      0x40

//P1 P0 Port data
#define PT6302_PDAT_MSK      0x03



//
//Set Brightness command
// 0 1 0 1 * DC2 DC1 DC0
#define PT6302_BRT_REG      0x50
#define PT6302_BRT_MSK      0x07

//DC2..DC0 Brightness Level (0..7)
//Note Brightness relationship between the number of active Grids (period) and the BRT value (duty cycle)
#define PT6302_BRT_0        0x00   //Duty  8/16 (Default)
#define PT6302_BRT_1        0x01
#define PT6302_BRT_2        0x02
#define PT6302_BRT_3        0x03
#define PT6302_BRT_4        0x04
#define PT6302_BRT_5        0x05
#define PT6302_BRT_6        0x06
#define PT6302_BRT_7        0x07   //Duty 15/16

#define PT6302_BRT_DEF      (PT6302_BRT_2)


//
//Grid control command
// 0 1 1 0 * GN2 GN1 GN0
#define PT6302_GRID_REG     0x60
#define PT6302_GRID_MSK     0x07

//Grids
//
// GN2 GN1 GN0
//  0   0   0   G1 to G16  // Default
//  0   0   1   G1 to G9
//  0   1   0   G1 to G10
//  0   1   1   G1 to G11
//  1   0   0   G1 to G12
//  1   0   1   G1 to G13
//  1   1   0   G1 to G14
//  1   1   1   G1 to G15
#define PT6302_GR1_GR9      0x01
#define PT6302_GR1_GR10     0x02
#define PT6302_GR1_GR11     0x03
#define PT6302_GR1_GR12     0x04
#define PT6302_GR1_GR13     0x05
#define PT6302_GR1_GR14     0x06
#define PT6302_GR1_GR15     0x07
#define PT6302_GR1_GR16     0x00


//
//Display On/Off command
// 0 1 1 1 * * H L
#define PT6302_DSPL_REG     0x70
#define PT6302_DSPL_MSK     0x03

//Display Mode
// H L Display operating state
// 0 0 Normal (default)
// 0 1 Off
// 1 0 All Segments and Additional Segments On
// 1 1 All Segments and Additional Segments On
#define PT6302_DSPL_NRM     0x00
#define PT6302_DSPL_OFF     0x01
#define PT6302_DSPL_ON      0x02


/** A class for driving Princeton PT6302 VFD controller
 *
 *  @brief Supports upto 16 Grids of 35 matrix segments. 
 *         Also supports 2 additional segments and 2 port pins. 
 *         SPI bus interface device. 
 */
class PT6302 {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid9_Add2  = PT6302_GR1_GR9,
    Grid10_Add2 = PT6302_GR1_GR10,
    Grid11_Add2 = PT6302_GR1_GR11,
    Grid12_Add2 = PT6302_GR1_GR12,
    Grid13_Add2 = PT6302_GR1_GR13,
    Grid14_Add2 = PT6302_GR1_GR14,
    Grid15_Add2 = PT6302_GR1_GR15,
    Grid16_Add2 = PT6302_GR1_GR16
  };
 
 /** Datatypes for display data */
//  typedef char DisplayData_t[PT6302_DISPLAY_MEM];
//  typedef char DisplayAdd_t[PT6302_ADD_MEM];  
    
 /** Constructor for class for driving Princeton PT6302 VFD controller
  *
  *  @brief Supports upto 16 Grids of 35 matrix segments. 
  *         Also supports 2 additional segments and 2 port pins.  
  *         SPI bus interface device. 
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  Mode selects number of Grids and Segments (default 16 Grids, 2 additional segments)
  */
  PT6302(PinName mosi, PinName sclk, PinName cs, Mode mode = Grid16_Add2);
      
  /** Clear the screen
    *
    * @param none
    * @return none
    */ 
  void cls();  

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (dutycycle linked to number of grids)  
    * @return none
    */    
  void setBrightness(char brightness = PT6302_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    * @return none    
    */
  void setDisplay(bool on);


  /** Set Port
    *
    * @param  char port (2 least significant bits)  
    * @return none
    */
  void setPort (char port = 0);

  /** Set User Defined Characters (UDC)
    *
    * @param unsigned char udc_idx   The Index of the UDC (0..7)
    * @param UDCData_t udc_data      The bitpattern for the UDC (5 bytes)
    * @return none    
    */
  void setUDC(unsigned char udc_idx, UDCData_t udc_data);

 
  /** Write Data to PT6302
    *
    *  @param char data Character code
    *  @param char address Parameter for data
    *  @return none
    */  
  void writeData(char data, char address);

  /** Write Additional Data to PT6302
    *
    *  @param char adata Additional code (annunciator)
    *  @param char address Parameter for data
    *  @return none
    */  
  void writeAData(char adata, char address);
 
 protected:  
  /** Write command and parameters to PT6302
    *
    *  @param char cmd Command byte
    *  @param char data Parameter for command
    *  @return none
    */  
  void _writeCmd(char cmd, char data);
 
  /** Write command to PT6302
    *
    *  @param char cmd Command byte
    *  @return none
    */  
  void _writeCmd(char cmd);

  char _port; 

 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  
  /** Init the SPI interface and the controller
    *
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The PT6302 expects LSB first, whereas SPI is MSB first
    *
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

};


#if (HANNSTAR_TEST == 1)
// Derived class for HANNSTAR display unit
//   Grids 1-16 all display 35 segment matrix characters and no Additional segments.
//

//HANNSTAR Display data
#define HANNSTAR_NR_GRIDS  16
#define HANNSTAR_NR_DIGITS 16
//#define HANNSTAR_NR_UDC    8

//HANNSTAR Memory size in bytes for Display
//#define HANNSTAR_DISPLAY_MEM (HANNSTAR_NR_GRIDS * PT6302_BYTES_PER_GRID)


//
//Set Port data command (General output)
// 0 1 0 0 * * P1 P0

//P0 P1 Port data, P0 is used for VGen control
#define PT6302_HANN_PDAT_MSK  0x02

//P0 is used to control the VFD Voltage generator (-24V and Filament)
#define PT6302_HANN_VGEN      0x01



/** Constructor for class for driving Princeton PT6302 VFD controller as used in HANNSTAR
  *
  *  @brief Supports 16 Grids of 35 Segments without additional Segments.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  */
class PT6302_HANNSTAR : public PT6302, public Stream {
 public:

/** Constructor for class for driving Princeton PT6302 VFD controller as used in HANNSTAR
  *
  *  @brief Supports 16 Grids of 35 Segments without additional Segments.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  */
  PT6302_HANNSTAR(PinName mosi, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     * @return char written
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @return none     
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     *
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();  

  /** Set Port
    *
    * @param  char port (Only Bit 1 is used)
    * @return none
    */
  void setPort (char port = 0);     

  /** Set VFD VGen
    *
    * @param  bool on
    * @return none
    */
  void setVGen (bool on = true);     

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;                     // Current cursor location
    int _columns;                    // Max number of columns
    
//    DisplayData_t _displaybuffer;    // Local mirror for all chars and icons
};
#endif

#endif