/* PCF8574_EnableBus - Use the PCF8574 I2C Port Extender for controlling the Chip Enable Bus
 * Copyright (c) 2011 Wim Huiskamp
 *
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * version 0.2 Initial Release
*/
#include "mbed.h"
#include "PCF8574_EnableBus.h"


/** Create an PCF8574_EnableBus object connected to the specified I2C object and using the specified deviceAddress
 *
 * @param I2C &i2c the I2C port to connect to 
 * @param char deviceAddress the address of the PCF8574
*/
PCF8574_EnableBus::PCF8574_EnableBus(I2C &i2c, char deviceAddress) : _i2c(i2c) {
   _writeOpcode = deviceAddress & 0xFE; // low order bit = 0 for write
   _readOpcode  = deviceAddress | 0x01; // low order bit = 1 for read
   _init();
}


/** Set or Reset Chip Select pins on Enable Bus
 *
 * @param CS_Pin cs_pin 
 * @param Bit_Level cs_level
*/
void PCF8574_EnableBus::chipselect (CS_Pin cs_pin, Bit_Level cs_level) {
    int result = 1;
      
    switch (cs_pin) {
     case CS_SWITCH : if (cs_level == LOW)
                         _enable_bus = ~D_CS_SWITCH; // CS Pin Low, make sure that only one CS is active
                      else
                         _enable_bus |= D_CS_SWITCH; // CS Pin High
                      break;
     case LATCHEN_1 : if (cs_level == LOW)
                         _enable_bus = ~D_LATCHEN_1; // CS Pin Low, make sure that only one CS is active
                      else
                         _enable_bus |= D_LATCHEN_1; // CS Pin High
                      break;
     case LATCHEN_2 : if (cs_level == LOW)
                         _enable_bus = ~D_LATCHEN_2; // CS Pin Low, make sure that only one CS is active
                      else
                         _enable_bus |= D_LATCHEN_2; // CS Pin High
                      break;
     case CS_BRIGHT : if (cs_level == LOW)
                         _enable_bus = ~D_CS_BRIGHT; // CS Pin Low, make sure that only one CS is active
                      else
                         _enable_bus |= D_CS_BRIGHT; // CS Pin High
                      break;
     case CS_DISP : if (cs_level == LOW)
                         _enable_bus = ~D_CS_DISP;   // CS Pin Low, make sure that only one CS is active
                      else
                         _enable_bus |= D_CS_DISP;   // CS Pin High
                      break;
     
     default:          // Oops, we should never end up here....
                      result = -1;
    }

    _write();                              // Write chip enable bits to bus
}

/** Set or Clear the Reset pin on Enable Bus
 *
 * @param Bit_Level rst_level
*/
void PCF8574_EnableBus::reset (Bit_Level rst_level) {

    if (rst_level == LOW) {
        _reset_pin = 0x00;                 // Reset Pin Low
      }               
    else {
        _reset_pin = D_RESET;              // Reset Pin High
    }     
    
    _write();                              // Write RST bit to bus
}
 
               
/** Set or Clear the NoGo pin on Enable Bus
 *
 * @param Bit_Level nogo_level
*/
void PCF8574_EnableBus::nogo (Bit_Level nogo_level) {

    if (nogo_level == LOW) {
        _nogo_pin = 0x00;                  // NOGO Pin Low
      }               
    else {
        _nogo_pin = D_NOGO;                // NOGO Pin High
    }     
    
    _write();                              // Write NoGo bit to bus
}
 

/** Optimised EnableBus write operation.
 * @param byte the value to output on the bus
*/
void PCF8574_EnableBus::_write(char byte) {
    char data[1];
    
    data[0] = byte;
    _i2c.write(_writeOpcode, data, 1);    // Write value to bus   
}

/** Optimised EnableBus write operation.
 * @param 
*/
void PCF8574_EnableBus::_write() {
    char data[1];
    
    data[0] = (_enable_bus & D_ENABLE_MSK) | _reset_pin | _nogo_pin;      // Combine enable bits and control bits
    _i2c.write(_writeOpcode, data, 1);    // Write value to bus   
}


/** Init PCF8574_EnableBus
 * @param
 * @returns 
 */
void PCF8574_EnableBus::_init() {
    _enable_bus = 0xFF;                    // Make sure that all CS pins are disabled
    _reset_pin  = D_RESET;                 // Make sure that Reset pin is disabled
    _nogo_pin   = D_NOGO;                  // Make sure that NoGo pin is disabled    
    _write();                              // Write value to bus    
}