/* mbed LC75710 Library, for Sanyo LC7571X VFD controller
 * Note: The LC75710, LC75711 and LC75712 differ only in the built-in character ROM
 *
 * Copyright (c) 2017, v01: WH, Initial version
 *               2017, v02: WH, Cleaned up docs 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef LC75711_H
#define LC75711_H

// Select one of the testboards for Sanyo LC75711 VFD controller
#include "LC75711_Config.h"
#include "LC75711_UDC.h"

/** An interface for driving Sanyo LC75711 VFD controller
 *
 * @code
 *
 * #if (LC75711_TEST == 1)  
 * // Direct driving of LC75711 Test
 *
 * #include "mbed.h"
 * #include "LC75711.h" 
 *
 * DigitalOut myled(LED1);
 * Serial pc(USBTX, USBRX);
 * 
 * // LC75711 declaration, Default setting 16 Grids @ 35 Segments
 * LC75711 LC75711(p5, p7, p8); // DI, CLK, CS
 *
 * int main() {
 *   pc.printf("Hello World: LC75711 test\n\r");
 *
 *   LC75711.cls(); 
 *   LC75711.writeData(all_str);
 *   wait(4);
 *   LC75711.setBrightness(LC75711_BRT0);
 *   wait(1);
 *   LC75711.setBrightness(LC75711_BRT3);
 *
 *   LC75711.writeData((char)'H', 9);
 *   LC75711.writeData((char)'e', 8);
 *   LC75711.writeData((char)'l', 7);
 *   LC75711.writeData((char)'l', 6);
 *   LC75711.writeData((char)'o', 5);
 *           
 *   while(1) {
 *     myled = !myled;
 *     wait(1);
 *   }
 * }
 * #endif
 *
 * @endcode
 */


//LC75711 Display and Annunciator data
#define LC75711_MAX_NR_GRIDS   16
#define LC75711_BYTES_PER_GRID  1

//Memory size in bytes for Display and Annunciators
#define LC75711_DISPLAY_MEM    64
#define LC75711_ADD_MEM        16
//#define LC75711_UDC_MEM       8

//Serial control data consists of an 8-bit address and a 24-bit instruction. The address is used as a chip select function 
//when multiple ICs are connected to the same bus. The address for the LC75710NE series is only used to distinguish the device
//from different types of devices. Multiple LC75711 devices on the same bus can only be distinguised by the CE control.
//Note that the serial control is similar but not identical to SPI behaviour: 
// The address is transmitted during CE low, the command & data is latched on falling CE edge.
// A wait time must be observed after each command. Typical delay is 18 us.
//Address (LSB sent first)
#define LC75711_ADDRESS      0x67


//
//Blink command, allows individual digit control
// 1 0 1 M A BC2 BC1 BC0 GR16 ... GR1
#define LC75711_BLNK_REG     0xA0
#define LC75711_BLNK_MA_MSK  0x18
#define LC75711_BLNK_BC_MSK  0x07

//Blinking Mode
// M A Display operating state
// 0 0 Neither MDATA nor ADATA blinks.
// 0 1 Only ADATA blinks.
// 1 0 Only MDATA blinks.
// 1 1 Both ADATA and MDATA blink.

//Blinking Period in sec when fOSC is 2.7 MHz
// BC2 BC1 BC0 HEX
// 0   0   0   0   Blink operation is stopped.
// 0   0   1   1   0.1
// 0   1   0   2   0.2
// 0   1   1   3   0.3
// 1   0   0   4   0.4
// 1   0   1   5   0.5
// 1   1   0   6   0.8
// 1   1   1   7   1.0
#define LC75711_BLNK_00      0x00
#define LC75711_BLNK_01      0x01
#define LC75711_BLNK_02      0x02
#define LC75711_BLNK_03      0x03
#define LC75711_BLNK_04      0x04
#define LC75711_BLNK_05      0x05
#define LC75711_BLNK_08      0x06
#define LC75711_BLNK_10      0x07

#define LC75711_BLNK_ON      (LC75711_BLNK_MA_MSK | LC75711_BLNK_08)
#define LC75711_BLNK_OFF     (LC75711_BLNK_MA_MSK | LC75711_BLNK_00)

//Grid selectors
#define LC75711_GR1     (1 <<  0)
#define LC75711_GR2     (1 <<  1)
#define LC75711_GR3     (1 <<  2)
#define LC75711_GR4     (1 <<  3)
#define LC75711_GR5     (1 <<  4)
#define LC75711_GR6     (1 <<  5)
#define LC75711_GR7     (1 <<  6)
#define LC75711_GR8     (1 <<  7)
#define LC75711_GR9     (1 <<  8)
#define LC75711_GR10    (1 <<  9)
#define LC75711_GR11    (1 << 10)
#define LC75711_GR12    (1 << 11)
#define LC75711_GR13    (1 << 12)
#define LC75711_GR14    (1 << 13)
#define LC75711_GR15    (1 << 14)
#define LC75711_GR16    (1 << 15)

#define LC75711_GR_ALL   (0xFFFF)
#define LC75711_GR_MSK   (0xFFFF)

//Blinking Command delay
#define LC75711_BLNK_DLY       18

//
//Display On/Off command, allows individual digit control
// 0 0 0 1 * M A O GR16 ... GRD1
#define LC75711_DSPL_REG     0x10
#define LC75711_DSPL_MA_MSK  0x06
#define LC75711_DSPL_O_MSK   0x01

//On/Off Mode
// M A Display operating state
// 0 0 Both MDATA and ADATA off
// 0 1 Only ADATA on
// 1 0 Only MDATA on
// 1 1 Both ADATA and MDATA on

//On/Off
// O Display state
// 0 Off
// 1 On

#define LC75711_DSPL_ON      (LC75711_DSPL_MA_MSK | LC75711_DSPL_O_MSK)
#define LC75711_DSPL_OFF     (LC75711_DSPL_MA_MSK)

//Display Command delay
#define LC75711_DSPL_DLY       18


//Display shift is NOT USED
// This would screw up the correlation between column index and character position.
// It also screws up the correlation between character data and additional data.
// Note that chardata has 64 positions whereas adddata has 16 positions

//
//Display shift command
// 0 0 1 0 * M A R/L ...
#define LC75711_SHFT_REG     0x20
#define LC75711_SHFT_MA_MSK  0x06
#define LC75711_SHFT_RL_MSK  0x01

// Shift Mode
// M A Display operating state
// 0 0 Neither MDATA and ADATA shift
// 0 1 Only ADATA 
// 1 0 Only MDATA 
// 1 1 Both ADATA and MDATA

//Shift direction
// RL Display shift
// 0 Right 
// 1 Left

//Shift Command delay
#define LC75711_SHFT_DLY       18

//
//Grid control command
// 0 0 1 1 GN3 GN2 GN1 GN0 ...
#define LC75711_GRID_REG     0x30
#define LC75711_GRID_MSK     0x0F

//Grids
//
// GN3 GN2 GN1 GN0
//  0   0   0   0   G1 to G16
//  0   0   0   1   G1
//  0   0   1   0   G1 to G2
//  0   0   1   1   G1 to G3
//  0   1   0   0   G1 to G4
//  0   1   0   1   G1 to G5
//  0   1   1   0   G1 to G6
//  0   1   1   1   G1 to G7
//  1   0   0   0   G1 to G8
//  1   0   0   1   G1 to G9
//  1   0   1   0   G1 to G10
//  1   0   1   1   G1 to G11
//  1   1   0   0   G1 to G12
//  1   1   0   1   G1 to G13
//  1   1   1   0   G1 to G14
//  1   1   1   1   G1 to G15
#define LC75711_GR1_GR1   0x01
#define LC75711_GR1_GR2   0x02
#define LC75711_GR1_GR3   0x03
#define LC75711_GR1_GR4   0x04
#define LC75711_GR1_GR5   0x05
#define LC75711_GR1_GR6   0x06
#define LC75711_GR1_GR7   0x07
#define LC75711_GR1_GR8   0x08
#define LC75711_GR1_GR9   0x09
#define LC75711_GR1_GR10  0x0A
#define LC75711_GR1_GR11  0x0B
#define LC75711_GR1_GR12  0x0C
#define LC75711_GR1_GR13  0x0D
#define LC75711_GR1_GR14  0x0E
#define LC75711_GR1_GR15  0x0F
#define LC75711_GR1_GR16  0x00

//Grid Command delay
#define LC75711_GRID_DLY        1

//
//Set AC Address command
// 0 1 0 0 RA3 RA2 RA1 RA0 * * DA5 DA4 DA3 DA2 DA1 DA0 * * * * * * * *
#define LC75711_AC_REG       0x40
#define LC75711_AADR_MSK     0x0F
#define LC75711_DADR_MSK     0x3F

//RA3..RA0 ADRAM Address (Additional data)
//DA5..DA0 DCRAM Address (Character data)

//AC Command delay
#define LC75711_AC_DLY         18

//
//Set Brightness command
// 0 1 0 1 * * * * DC7 DC6 DC5 DC4 DC3 DC2 DC1 DC0 * * * * * * * *
#define LC75711_BRT_REG      0x50
#define LC75711_BRT_MSK      0xFF

//DC7..DC0 Brightness Level (0..239)
//Note Brightness relationship between the number of active Grids (period) and the BRT value (duty cycle)
#define LC75711_BRT_0        0x00
#define LC75711_BRT_1        0x20
#define LC75711_BRT_2        0x40
#define LC75711_BRT_3        0x80
#define LC75711_BRT_4        0xA0
#define LC75711_BRT_5        0xC0
#define LC75711_BRT_6        0xD0
#define LC75711_BRT_7        0xF0

#define LC75711_BRT_DEF      (LC75711_BRT_3)

//Brightness Command delay
#define LC75711_BRT_DLY         1

//
//Set Char data command (DCRAM)
// 0 1 1 0 * * * * * * DA5 DA4 DA3 DA2 DA1 DA0 D7...D0
#define LC75711_DATA_REG     0x60
//#define LC75711_DADR_MSK     0x3F
//#define LC75711_DATA_MSK     0xFF

//AA5..DA0 DCRAM Address (Character data)
//DA7..DA0 Character Data

//Set Data Command delay
#define LC75711_DATA_DLY        18

//
//Set Additional data command (ADRAM), Used for annunciators etc
// 0 1 1 1 RA3 RA2 RA1 RA0 AD8 AD7 AD6 AD5 AD4 AD3 AD2 AD1 * * * * * * * *
#define LC75711_ADAT_REG      0x70

//RA3..RA0 ADRAM Address (Additional data)
//#define LC75711_AADR_MSK       0x0F

//AD8..AD1 Additional Data
#define LC75711_ADAT_MSK      0xFF

//Set AData Command delay
#define LC75711_ADAT_DLY        18

//
//Set UDC data command (CGRAM)
// 1 0 0 0 * * * * CA7 CA6 ... CA0
//
// * * * * * CD35 CD34 ...    CD25
//
// CD24 CD23 ...               CD9
//
// CD8 ...      CD1
#define LC75711_UDC_REG        0x80
#define LC75711_UDC_MSK        0x0F
#define LC75711_NR_UDC            8

//CA7..CA0 CGRAM Address (UDC RAM address)
//CD35..CD0 UDC Data
//UDC is a 5x7 Matrix pattern
// CD1  .. CD5
// CD6  .. CD10
// CD11 .. CD15
// CD16 .. CD20
// CD21 .. CD25
// CD26 .. CD30
// CD31 .. CD35

//Set UDC Data Command delay
#define LC75711_UDC_DLY          18

//UDCs are defined by a 5x7 matrix and stored as 7 bytes
typedef char UDCData_t[7];


/** A class for driving Sanyo LC75711 VFD controller
 *
 * @brief Supports upto 16 Grids of 35 matrix segments. Also supports 3-8 additional segments (depending on number of grids).
 *        SPI bus interface device. 
 */
class LC75711 {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid1_Add8  = LC75711_GR1_GR1,
    Grid2_Add8  = LC75711_GR1_GR2,
    Grid3_Add8  = LC75711_GR1_GR3,
    Grid4_Add8  = LC75711_GR1_GR4,
    Grid5_Add8  = LC75711_GR1_GR5,
    Grid6_Add8  = LC75711_GR1_GR6,
    Grid7_Add8  = LC75711_GR1_GR7,
    Grid8_Add8  = LC75711_GR1_GR8,
    Grid9_Add8  = LC75711_GR1_GR9,
    Grid10_Add8 = LC75711_GR1_GR10,
    Grid11_Add8 = LC75711_GR1_GR11,
    Grid12_Add7 = LC75711_GR1_GR12,
    Grid13_Add6 = LC75711_GR1_GR13,
    Grid14_Add5 = LC75711_GR1_GR14,
    Grid15_Add4 = LC75711_GR1_GR15,
    Grid16_Add3 = LC75711_GR1_GR16
  };
 
 /** Datatypes for display data */
//  typedef char DisplayData_t[LC75711_DISPLAY_MEM];
//  typedef char DisplayAdd_t[LC75711_ADD_MEM];  
    
 /** Constructor for class for driving Sanyo LC75711 VFD controller
  *
  *  @brief Supports upto 16 Grids of 35 matrix segments. Also supports 3-8 additional segments (depending on number of grids).
  *         SPI bus interface device. 
  *  @param  PinName mosi, sclk, cs SPI bus pins
  *  @param  Mode selects number of Grids and Segments (default 11 Grids, 35 matrix segments, 8 additional segments)
  */
  LC75711(PinName mosi, PinName sclk, PinName cs, Mode mode = Grid11_Add8);
      
  /** Clear the screen and locate to 0
    *
    * @param none
    * @return none
    */ 
  void cls();  

  /** Set the Blink mode
    *
    * @param bool blink mode
    * @param int  grids selected grids for Blinking enable/disable (default = all)
    * @return none    
    */
  void setBlink(bool on, int grids = LC75711_GR_ALL);
  
  /** Set Brightness
    *
    * @param  char brightness (8 significant bits, valid range 0..239 (dutycycle linked to number of grids)  
    * @return none
    */    
  void setBrightness(char brightness = LC75711_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    * @return none    
    */
  void setDisplay(bool on);


  /** Set User Defined Characters (UDC)
    *
    * @param unsigned char udc_idx   The Index of the UDC (0..7)
    * @param UDCData_t udc_data      The bitpattern for the UDC (7 bytes)
    * @return none    
    */
  void setUDC(unsigned char udc_idx, UDCData_t udc_data);

 
  /** Write Data to LC75711
    *
    *  @param char data Character code
    *  @param char address Parameter for data
    *  @return none
    */  
  void writeData(char data, char address);

  /** Write Additional Data to LC75711
    *
    *  @param char adata Additional code (annunciator)
    *  @param char address Parameter for data
    *  @return none
    */  
  void writeAData(char adata, char address);
 
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  int _blink; // Local shadow
    
  /** Init the SPI interface and the controller
    *
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The LC75711 expects LSB first, whereas SPI is MSB first
    *
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);


  /** Set Address
    *
    *  @param char RAM address for data displayed at Grid1 (0..63)
    *  @param char RAM address for adata displayed at Grid1 (0..15) 
    *  @return none
    *
    *  Note that a Shift (L/R) command will change the Address of data displayed at Grid1
    */  
  void _setAddress(char data_addr=0, char adata_addr=0);


  /** Write command and parameters to LC75711
    *
    *  @param char cmd Command byte
    *  @param char data1 Parameters for command
    *  @param char data0 Parameters for command  
    *  @param char delay Delay for command execution
    *  @return none
    */  
  void _writeCmd(char cmd, char data1, char data0, char delay);
 
};


#if (ASTON_TEST == 1)
// Derived class for ASTON display unit
//   Grids 1-10 all display 35 segment matrix characters and no Additional segments.
//   Grid  11 uses a number of Segments to display Icons. 

//ASTON Display data
#define ASTON_NR_GRIDS  10
#define ASTON_NR_DIGITS 10
//#define ASTON_NR_UDC    8

//ASTON Memory size in bytes for Display
//#define ASTON_DISPLAY_MEM (ASTON_NR_GRIDS * LC75711_BYTES_PER_GRID)


/** Constructor for class for driving Sanyo LC75711 VFD controller as used in ASTON
  *
  *  @brief Supports 10 Grids of 35 Segments without additional Segments and uses Grid 11 for Icon segments.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  */
class LC75711_ASTON : public LC75711, public Stream {
 public:

  /** Enums for Icons
   *
   *  @brief Aston display uses Grid 11 to show Icons.
   *         The Icons are each connnected to one of the 35 segments. 
   *         Icons are controlled by redefining UDC_0. 
   *  Icon Enums encode UDC_0 byte index in 8 MSBs and encode Icon bit/segment in 8 LSBs
   */
  enum Icon {
    R0    = (6<<8) | 4,
    R1    = (6<<8) | 3,
    R2    = (6<<8) | 2,
    R3    = (6<<8) | 1,
    CRD1  = (5<<8) | 4,
    CRD2  = (5<<8) | 3,
    CARD  = (5<<8) | 2,
    KEY   = (4<<8) | 1,
    VDCRP = (4<<8) | 2,
    D     = (4<<8) | 0,
    D2    = (3<<8) | 4,
    MAC   = (3<<8) | 3,
    R16_9 = (0<<8) | 0,
    DISH  = (5<<8) | 0,     
    DSH1  = (4<<8) | 3,
    DSH2  = (5<<8) | 1,
    TMR   = (1<<8) | 3,
    CBND  = (2<<8) | 1,
    KBND  = (2<<8) | 4,
    AFC   = (3<<8) | 0
  };
  

/** Constructor for class for driving Sanyo LC75711 VFD controller as used in ASTON
  *
  *  @brief Supports 10 Grids of 35 Segments without additional Segments and uses Grid 11 for Icon segments.
  *  
  *  @param  PinName mosi, sclk, cs SPI bus pins
  */
  LC75711_ASTON(PinName mosi, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     * @return char written
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @return none     
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     *
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon Enums Icon Encodes UDC_0 byte index in 8 MSBs and encodes Icon bit/segment in 8 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon Enums Icon Encodes UDC_0 byte index in 8 MSBs and encodes Icon bit/segment in 8 LSBs     
     * @return none
     */
    void clrIcon(Icon icon);

   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;                     // Current cursor location
    int _columns;                    // Max number of columns
    
//    DisplayData_t _displaybuffer;    // Local mirror for all chars and icons
//    UDCData_t _UDC_16S;              // User Defined Character pattterns (UDC)
    UDCData_t _udc_icon;             // The UDC_0 bitpattern for the Icons shown at Grid 11 (7 bytes)    
};
#endif

#endif