/* mbed HP03SA Library, for an I2C Pressure and Temperature sensor which provides derived Altitude values
 * Copyright (c) 2015, v01: WH, Initial version, ported in part from Elektor weatherstation (Sept 2011), Author: W. Waetzig
 *                          See http://www.elektor-labs.com/project/usb-long-term-weather-logger-100888.12037.html
 *                          Code also based on several other public sources: 
 *                          See http://en.wikipedia.org/wiki/Atmospheric_pressure
 *                          See https://learn.sparkfun.com/tutorials/bmp180-barometric-pressure-sensor-hookup-?_ga=1.94307604.888266135.1310146152
 *
 * The HP03S pressure sensor from Hope Microelectronics contains a piezoresistive transducer and
 * integrated 16-bit A/D converter (ADC), along with control logic and an I2C interface. 
 * The transducer outputs one voltage that depends on pressure and one that depends on temperature.
 * These analogue values are alternately converted by the ADC and the results made available on the I2C interface. 
 * During the manufacturing process eleven sensor-specific calibration values with a length of two bytes are stored 
 * in the device’s EEPROM, and these can also be retrieved by the microcontroller.
 * The correction-parameters are read out from the device in used in the calculation.
 * A 32 kHz clock with an amplitude of 3 V is required to drive the ADC. This could be derived from the mbed PwmOut.
 * The HP03SA has an absolute pressure range between 300-1100 hPa. Operating temp is -20 to +60 degrees Celsius.
 * Note: 1 atm = 1013.25 hPa = 1013.25 mbar
 *
 * Unit conversions see http://www.hoperf.com/docs/guide/160.htm
 *
 *   bar Bar                                          convert to 100,000 Pascals exactly
 *   psi Pounds per Square Inch                       convert to 6894.76 Pascals
 *   mb or mbar Millibar                              convert to 100 Pascals (1 hPa) exactly
 *   hPa Hectopascal                                  convert to 100 Pascals exactly
 *   kPa Kilopascal                                   convert to 1,000 Pascals
 *   MPa Megapascal                                   convert to 1,000,000 Pascals
 *   Kgf/cm² or kg/cm² Kilogram force per Square Centimetre convert to 98,066.5 Pascals
 *   N/m² Newton per Square Meter                     convert to 1 Pascal (SI unit for pressure)
 *   torr Torr                                        convert to 133.322 Pascals
 *   mtorr Millitorr                                  convert to 0.133322 Pascals
 *   mm Hg                                            convert to 133.322387415 Pascals
 *   1 atm 1 Atmosphere                               convert to 101325 Pascals exactly (760 mm Hg)
 *   1 atm 1 Atmosphere                               convert to 1013.25 hPa exactly 
 *   Dynes/cm² or D/cm² Dynes per Square Centimetre   convert to 0.1 Pascals exactly
 *   oz/in² Ounces per Square Inch                    convert to 430.922 Pascals
 *   tsi (UK) or tfsi (UK) Tonnes Force per Square Inch (Long, UK) convert to 15444300 Pascals
 *   tsi (USA) or tfsi (USA) Ton Force per Square Inch (Short, USA) convert to 13789500 Pascals
 *   tsf (USA) or tfsf (USA) Tons Force per Square Foot (Short, USA) convert to 95760.5 Pascals
 *   psf Pounds per Square Foot                       convert to 47.8803 Pascals
 *   g/cm² or gf/cm² Grammes Force per Square Centimetre convert to 98.0665 Pascals
 *
 * The Altitude calculation is based on the ICAO Standard Atmosphere model ISA. The code has 
 * certainly not been thoroughly validated! Use at own risk.
 *
 * QNH is the barometric altimeter setting that causes an altimeter to read 
 * airfield elevation above mean sea level when on the airfield. In ISA temperature
 * conditions the altimeter will read altitude above mean sea level in the vicinity of the airfield.
 * Note: Remember QNH as mnemonic for "nautical height"
 * Average sea-level pressure is 101.325 kPa (1013.25 hPa or mbar) or 29.92 inches (inHg) or 
 * 760 millimetres of mercury (mmHg). In aviation weather reports (METAR), QNH is transmitted 
 * around the world in hectopascals or millibars (1 hectopascal = 1 millibar), except in the 
 * United States, Canada, and Colombia where it is reported in inches (to two decimal places) of mercury. 
 *
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef MBED_HP03SA_H
#define MBED_HP03SA_H

#include "mbed.h"

/** An interface for the HP03S I2C pressure and temperature sensor
 *
 * @code
 * #include "mbed.h"
 * #include "HP03S.h"
 * 
 * // I2C Communication
 * I2C i2c(p28,p27); // SDA, SCL
 *
 * // Serial Communication*
 * Serial pc(USBTX,USBRX);
 *
 * // Instantiate HP03SA
 * HP03SA hp03sa(&i2c, D_XCLR); // I2C bus, XCLK Pin 
 * // Generate the 32kHz master clock for HP03SA
 * PwmOut mclk(p21);
 *
 * int main() {
 *   pc.printf("Hello World!\n");
 * 
 *   //Init MCLK for HP03SA
 *   mclk.period_us(30);       //32768 KHz
 *   mclk.pulsewidth_us(15);   //32768 KHz
 *
 *   while(1) {
 *     // Take reading from sensor            
 *     hp03sa.sample();
 *    
 *     // Show Pressure    
 *     pc.printf("Absolute atmospheric pressure is %.1f [hPa]\r\n", (float) hp03sa.getAbsPressure() / 10.0f);
 *     pc.printf("Sealevel atmospheric pressure is %.1f [hPa]\r\n", (float) hp03sa.getSeaPressure(20) / 10.0f); // About right where I live..
 *   
 *     // Show Temperature
 *     pc.printf("Ambient temperature is %.1f [C]\r\n", (float) hp03sa.getTemperature() / 10.0f);       
 *
 *     // Show Altitude meters
 *     pc.printf("Altitude estimated at %d [m]\r\n", hp03sa.getAltitude());       
 * 
 *     // Show Altitude Ft
 *     pc.printf("Altitude estimated at %d [ft]\r\n", hp03sa.getAltitudeFt());
 *
 *     wait(1.0);
 *   }
 * }
 * @endcode
 */

// Device I2C Slave addresses
#define HP03SA_BARO_SA   0xEE 
#define HP03SA_EEPROM_SA 0xA0

//ISA computation constants
#define RLGM        0.19026674f          // ISA Exponent (R * L) / (g * M)
#define GMRL        5.25578009f          // ISA Exponent (g * M) / (R * L) 
#define LBASE_X10   0.065f               // ISA Temp Lapse (i.e. drop) in K per meter x10
#define TBASE_X10   (150.0f + 2731.5f)   // ISA Ref Temp in K x10
#define PBASE_X10   (1013.25f * 10.0f)   // ISA Ref Pres in hPa x10
  
/** Create an HP03SA Class instance
  *
  */ 
class HP03SA {

public:

  /** Create an HP03SA device instance
    *
    * @param i2c    I2C Bus 
    * @param XCLR   Clock enable control line
    */ 
   HP03SA(I2C *i2c, PinName XCLR);

  /** Get Status
    *
    * @return bool Sensor ready 
    */ 
   bool getStatus(void);

  /** Get Pressure and Temperature sample
    *
    * @return none 
    */ 
   void sample(void);

  /** Get Absolute Atmospheric Pressure in hPa x 10
    * Note: sample() must be called before getAbsPressure()
    *
    * @return int Pressure in hPa x 10
    */ 
   int getAbsPressure(void);

  /** Get Sealevel Atmospheric Pressure in hPa x 10
    * Note: sample() must be called before getSeaPressure()
    *
    * @param  int alt_meter Altitude above Mean Sea Level where measurement is taken
    * @return int Pressure at sea level in hPa x 10   
    */ 
   int getSeaPressure(int alt_meter);

  /** Get Temperature as int in °Celsius x 10
    * Note: sample() must be called before getTemperature()  
    *
    * @return int Temperature in °Celsius x 10
    */ 
   int getTemperatureInt(void);

  /** Get Temperature as float in °Celsius
    * Note: sample() must be called before getTemperatureFl()  
    *
    * @return float Temperature in °Celsius
    */ 
   float getTemperature(void);

  /** Get Altitude in meters using ISA
    * Note: sample() must be called before getAltitude()  
    *
    * @return int Altitude in meters above Mean Sea Level (MSL)
    */ 
   int getAltitude(void);

  /** Get Altitude in feet using ISA
    * Note: sample() must be called before getAltitudeFt()  
    *
    * @return int Altitude in feet above Mean Sea Level (MSL)
    */ 
   int getAltitudeFt();

  /** Set QNH Pressure to calibrate sensor
    *
    * @param int pressure_x10 at Mean Sea Level in hPa x 10
    *            The getAltitude() reading will be 0 m for the set pressure.
    */ 
   void setPressure(int pressure_x10 = PBASE_X10);

  /** Set QNH Altitude in meter to calibrate sensor
    *
    * @param int alt_meter Altitude in meters above Mean Sea Level (MSL) for current pressure
    *            The getAltitude() reading will be 'alt_meter' m for the current pressure.   
    */ 
   void setAltitude(int alt_meter = 0);


  /** Set QNH Altitude in feet to calibrate sensor
    *
    * @param int alt_feet Altitude in meters above Mean Sea Level (MSL) for current pressure
    *            The getAltitudeFt() reading will be 'alt_feet' ft for the current pressure.   
    */ 
   void setAltitudeFt(int alt_feet = 0);


   /** Convert Temperature from °Celsius into °Fahrenheit
     *
     * @param  float celsius in °Celsius  
     * @return float temperature in °Fahrenheit
     */ 
    float celsiusToFahrenheit(float celsius);
 
private: 

  /** Init HP03SA device
    *
    * @return none    
    */ 
   void _init();

  /** Read ADC value (either pressure or temperature depending on Control Register setting)
    *
    * @return int   pressure or temperature 
    */ 
   int _readADC (void);

  /** Read Pressure and Temperature
    *
    * @return none 
    */ 
   void _readTempPres(void);
  
  /** Calculate Pressure and Temperature
    *
    * @return none 
    */ 
   void _calcTempPres(void);
 
  /** Calculate Altitude
    *
    * @return none 
    */ 
   void _calcAlt(void);

  //I2C interface and ADC control
  I2C *_i2c;
  DigitalOut _xclr;

  //Raw values, computed values
  int  _D1,  _D2;        // measured values: pressure,temperature.  
  int  _dUT, _off;       // auxiliary variables
  int  _sens, _x;        // auxiliary variables  
  int  _temp_x10;        // temperature in Celsius x 10
  int  _pres_x10;        // pressure in hPa x 10  
  int  _alt;             // altitude in m  
  float _pres_base_x10;
  float _temp_base_x10;
    
  //Calibration coefficients
  int _C1,  //Sensitivity coefficient  
      _C2,  //Offset coefficient
      _C3,  //Temp coefficient of sensitivity
      _C4,  //Temp coefficient of offset
      _C5,  //Reference temperature
      _C6,  //Temp coefficient of temperrature
      _C7;  //Offset fine tuning
  //Sensor correction-parameters A, B, C, D    
  int _A, _B, _C, _D;  
};
   
#endif