/** 
 * @file HCMS2975.h 
 * @brief  mbed Avago/HP HCSM2975 LED matrix display Library. 
 * @author WH
 * @date   Copyright (c) 2014
 *         v01: WH, Initial release, 
 *              Info available at http://playground.arduino.cc/Main/LedDisplay and http://www.pjrc.com/teensy/td_libs_LedDisplay.html
 *         v02: WH, added getVersion(), 
 *         v03: WH, fixed setBrightness at init() when there is no HW reset.    
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef _HCMS2975_H
#define _HCMS2975_H

#include "mbed.h"

//Enable Stream printf or minimalistic printf
#define HCMS2975_PRINTF     1

#define HCMS2975_VERSION    2

//Max supported display is 16x2
#define HCMS2975_BUFFER_SIZE         32
//#define HCMS2975_BUFFER_SIZE         8

/* LED Type information on Rows and Columns. This information is encoded in
 * an int and used for the LEDType enumerators in order to simplify code maintenance */
// Columns encoded in b7..b0
#define LED_T_COL_MSK  0x000000FF
#define LED_T_C8       0x00000008
#define LED_T_C16      0x00000010

// Rows encoded in b15..b8  
#define LED_T_ROW_MSK  0x0000FF00
#define LED_T_R1       0x00000100
#define LED_T_R2       0x00000200
#define LED_T_R3       0x00000300
#define LED_T_R4       0x00000400

/* LED number of devices. This information is encoded in
 * an int and used for the command and databyte transmission counter in order to simplify code maintenance */
// Number of Devices encoded in b23..16
#define LED_T_CNT_MSK  0x00FF0000
#define LED_T_N1       0x00010000 
#define LED_T_N2       0x00020000 
#define LED_T_N4       0x00040000 


// Controlword 0
#define HCMS2975_CONTROL0          0x00

// D6 is /Sleep
#define HCMS2975_SLP               0x00
#define HCMS2975_NORMAL            0x40

// D4.D5 selects max peak current
#define HCMS2975_PEAK_12_8         0x30
#define HCMS2975_PEAK_9_3          0x00
#define HCMS2975_PEAK_6_4          0x10
#define HCMS2975_PEAK_4_0          0x20

// display brightness definitions, indicating percentage brightness
// D3..D0 select brightness
#define HCMS2975_BRIGHT_100        0x0F
#define HCMS2975_BRIGHT_80         0x0E
#define HCMS2975_BRIGHT_60         0x0D
#define HCMS2975_BRIGHT_47         0x0C
#define HCMS2975_BRIGHT_37         0x0B
#define HCMS2975_BRIGHT_30         0x0A
#define HCMS2975_BRIGHT_23         0x09
#define HCMS2975_BRIGHT_18         0x08
#define HCMS2975_BRIGHT_15         0x07
#define HCMS2975_BRIGHT_11_7       0x06
#define HCMS2975_BRIGHT_8_3        0x05
#define HCMS2975_BRIGHT_6_7        0x04
#define HCMS2975_BRIGHT_5          0x03
#define HCMS2975_BRIGHT_3_3        0x02
#define HCMS2975_BRIGHT_1_7        0x01
#define HCMS2975_BRIGHT_0          0x00

// default display brightness
#define HCMS2975_DEF_BRIGHT        HCMS2975_BRIGHT_30
#define HCMS2975_DEF_PEAK          HCMS2975_PEAK_6_4

// Controlword 1
#define HCMS2975_CONTROL1          0x80

//Select Prescaler mode
#define HCMS2975_PRE_1             0x00
#define HCMS2975_PRE_8             0x02

//Select Serial/Simultaneous mode
#define HCMS2975_SERIAL            0x00
#define HCMS2975_SIMUL             0x01

// Write Digit columns data as 5 sequential bytes. 
// First byte defines left column. D7 always 0, D6..D0 define bitpattern. D6 is downmost, D0 is topmost pixel.
//
extern const char udc_0[];       /* |> */
extern const char udc_1[];       /* <| */
extern const char udc_2[];       /* | */
extern const char udc_3[];       /* || */
extern const char udc_4[];       /* ||| */
extern const char udc_5[];       /* = */
extern const char udc_6[];       /* checkerboard */
extern const char udc_7[];       /* \ */

extern const char udc_Bat_Hi[];  /* Battery Full */
extern const char udc_Bat_Ha[];  /* Battery Half */
extern const char udc_Bat_Lo[];  /* Battery Low */
extern const char udc_AC[];      /* AC Power */
extern const char udc_smiley[];  /* Smiley */

/** A library for driving HCMS2975 LED matrix displays
 * @brief Currently supports 8x1, 8x2, 16x1, 16x2
 *
 *
 * @code
 * #include "mbed.h"
 * #include "HCMS2975.h"
 *
 * // SPI Communication
 * SPI spi_led(p5, NC, p7); // MOSI, MISO, SCLK
 *
 * //Display
 * HCMS2975 led(&spi_led, p8, p9, p10); // SPI bus, CS pin, RS pin, Rst pin, LEDType = default
 *
 * int main() {
 *   int cnt;
 *
 *   led.locate(0, 0);                               
 *
 *   //          12345678 
 *   led.printf("Hi mbed ");  
 *   wait(2);
 * 
 *   cnt=0x20;  
 *   while(1) {
 *     wait(0.5);
 *                       
 *     led.putc(cnt);
 *     cnt++;
 *     if (cnt == 0x80) cnt=0x20;
 *   } 
 * }
 * @endcode
 */


/** Create an HCMS2975 Display object connected to the proper pins
 *
 * @param  *spi SPI port
 * @param  cs   PinName for Chip Select (active low)
 * @param  rs   PinName for RS (Data = low, Command = High)
 * @param  rst  PinName for Rst (active low, optional, default=NC) 
 * @param type  Sets the panel size (default = LED8x1)      
*/
class HCMS2975 : public Stream {  
//class HCMS2975 {
    
//Unfortunately this #define selection breaks Doxygen !!!
//#if (HCMS2975_PRINTF == 1)
//class HCMS2975 : public Stream {  
//#else    
//class HCMS2975 {
//#endif

public:
    /** LED panel format */
    enum LEDType {
        LED8x1     = (LED_T_N1 | LED_T_C8  | LED_T_R1),    /**<  8x1 LED panel (default) */    
        LED8x2     = (LED_T_N2 | LED_T_C8  | LED_T_R2),    /**<  8x2 LED panel */          
        LED16x1    = (LED_T_N2 | LED_T_C16 | LED_T_R1),    /**< 16x1 LED panel */                  
        LED16x2    = (LED_T_N4 | LED_T_C16 | LED_T_R2),    /**< 16x2 LED panel */
    };
    
    /** Display control */
    enum DisplayMode {
        DispOff,  /**<  Display Off */    
        DispOn    /**<  Display On */            
    };

   /** Create an HCMS2975 Display object connected to the proper pins
     *
     * @param  *spi SPI port
     * @param  cs   PinName for Chip Select (active low)
     * @param  rs   PinName for RS ()
     * @param  rst  PinName for Rst (active low, optional, default = NC)      
     * @param type  Sets the panel size (default = LED8x1)     
     */
   HCMS2975(SPI *spi, PinName cs, PinName rs, PinName rst = NC, LEDType type = LED8x1);


   /** Destructor for HCMS2975 Display object
     *
     * @param  none
     * @return none
     */ 
    virtual ~HCMS2975();
     
#if (HCMS2975_PRINTF != 1)
   /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
   int putc(int c);

    /** Write a raw string to the Display
     *
     * @param string text, may be followed by variables to emulate formatting the string.
     *                     However, when (HCMS2975_PRINTF != 1) then printf formatting is NOT supported and variables will be ignored!           
     */
    int printf(const char* text, ...);
#else    
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
    
#endif    

    /** Clear the screen and locate to 0,0
     */
    void cls();                              

    /** Locate cursor to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column=0, int row=0);
    
    /** Return the number of columns
      *
      * @return int The number of columns
      */   
    int columns();
    
    /** Return the number of rows
      *
      * @return int The number of rows
      */
    int rows();
    
    /** Set the Displaymode
     *
     * @param displayMode The Display mode (DispOff, DispOn)
     */
    void setMode(DisplayMode displayMode);     

    /** Set Brightness
     *
     * @param brightness The brightness level (valid range 0..15)
     */
    void setBrightness(uint8_t brightness = HCMS2975_DEF_BRIGHT);  
    
    /** Set User Defined Characters (UDC)
     *
     * @param unsigned char c   The Index of the UDC (0..7)
     * @param char *udc_data    The bitpatterns for the UDC (5 bytes of 7 significant bits for bitpattern)       
     */
    void setUDC(unsigned char c, char *udc_data);   


   /** Returns the version number of the library
     * @return int version number
     */
    int getVersion();
    
protected:
    /** Low level Reset method for controller
      */  
    void _reset();

   /** Low level Init method for LED controller
     */  
    void _init(); 
   
   /** Low level command byte write operation.
     * @param command commandbyte to write
     */
    void _writeCommand(uint8_t command);

   /** Low level _dataBuffer write operation.   
     */
    void _pushBuffer();   

    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

    //SPI bus
    SPI *_spi;
    DigitalOut _cs;         

    DigitalOut _rs;
    DigitalOut *_rst;
    
    //Display type
    LEDType _type;
    int _nr_cols;    
    int _nr_rows;    

    int _row;
    int _column;
    int _peak, _brightness;
 
    // _displayBuffer
    char _displayBuffer[HCMS2975_BUFFER_SIZE];
    int _displaySize;
    int _deviceCount;
    
    // Local UDC memory
    char _udc[8][5];   
};


#endif
