/* mbed Library for Noritake Itron CU209SCPB-T20 VFD module
 * 
 * Copyright (c) 2017, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h" 
#include "CU209SCPB.h"
#include "CU209SCPB_UDC.inc"


 /** Constructor for class for driving Noritake Itron CU209SCPB VFD module
  *
  *  @brief Supports 1 line of 20 chars (5x7 matrix segments).
  *         Serial bus interface device. 
  *  @param  PinName TX Serial bus pin
  *  @param  Baud baud selects baudrate (default 19200)  
  *  @param  Int Parity selects paritybits (default Even)  
  */
CU209SCPB::CU209SCPB(PinName TXD, Baud baud, SerialBase::Parity parity) : _serial(TXD,NC), _baud(baud), _parity(parity) {

  _init();

  _column   = 0;
  _columns  = CU209SCPB_NR_COLS; 
}

/** Init the CU209SCPB interface and the module
  *
  * @param  none
  * @return none
  */ 
void CU209SCPB::_init(){
  
//init Serial
  _serial.baud(_baud);
  _serial.format(8, _parity, 1); // CU209SCPB uses 8 databits, parity, 1 stopbit  

//init module  
//  _serial.putc(D_ESC);
//  _serial.putc(int('I'));  // Reset all settings
//
//  _serial.putc(D_DC1);     // Automatic CR mode, Cursor to 0 when reaching right end (Default)
//  _serial.putc(D_DC2);     // Overputc mode, Overputc when reaching right end, Cursor does not move
//  _serial.putc(D_DC3);     // Scroll mode, Text shifts left when reaching right end, Cursor does not move
//
//  _serial.putc(D_CT0);     // International Character Font (Default)
//  _serial.putc(D_CT1);     // Katakana Character Font
//
//  _serial.putc(D_ESC);
//  _serial.putc(int('T'));  
//  _serial.putc(20);        // Cursor Blinkspeed = 20 * 14.5ms (Default)
  _serial.putc(80);        // Cursor Blinkspeed = 80 * 14.5ms

  setBrightness(CU209SCPB_BRT_DEF); // Default Brightness

  // Clear the DCRAM (undefined at Reset)
  cls(); 

  // Clear the UDC RAM (undefined at Reset)
  const char udc_none[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00}; 
  for (int idx=0; idx < CU209SCPB_NR_UDC; idx++) {
    setUDC(idx, (char *)udc_none);
  }     
}   


/** Clear the screen and locate to 0
  *
  * @param none
  * @return none  
  */ 
void CU209SCPB::cls() {  
  _serial.putc(D_CLR); //Clear
  _serial.putc(D_CR);  //Cursor left
  
  _column = 0;   
}    


/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  * @return none  
  */
void CU209SCPB::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  

  _serial.putc(D_ESC);
  _serial.putc(int('H'));  
  _serial.putc(column);    
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int CU209SCPB::columns() {
    return _columns;
}


/** Set Brightness
  *
  * @param  char brightness (valid range 0..FF)  
  * @return none
  */
void CU209SCPB::setBrightness(char brightness){
  
//Sanity check
  brightness = brightness & CU209SCPB_BRT_MSK; // mask invalid bits
 
  _serial.putc(D_ESC);
  _serial.putc(int('L'));  
  _serial.putc(brightness);   
}

/** Set the Cursor mode On/off
  *
  * @param bool cursor mode
  * @return none  
  */
void CU209SCPB::setCursor(bool on) {
     
  if (on) {
    _serial.putc(D_DC5); // Blinking Cursor
  }
  else {
    _serial.putc(D_DC4); // Invisible
  }
}

/** Set Font
  *
  * @param  int font (valid range: 0 == International, other == Katakana)
  * @return none
  */
void CU209SCPB::setFont(int font){
  
  if (font == 0) {
    _serial.putc(D_CT0); // International (Default)
  }
  else {
    _serial.putc(D_CT1); // Katakana
  }
}

/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx   The Index of the UDC (0..7)
  * @param UDCData_t udc_data      The bitpattern for the UDC (7 bytes)
  * @return none  
  */
void CU209SCPB::setUDC(unsigned char udc_idx, UDCData_t udc_data) {

//Sanity check
  udc_idx = udc_idx & CU209SCPB_UADR_MSK; // mask invalid bits

  _serial.putc(D_ESC);
  _serial.putc(int('C'));  // UDC command
  _serial.putc(udc_idx);   

  _serial.putc( ((udc_data[1] & 0x07) << 5) | ((udc_data[0] & 0x1F) << 0) );                               // P8  ......  P1
  _serial.putc( ((udc_data[3] & 0x01) << 7) | ((udc_data[2] & 0x1F) << 2) | ((udc_data[1] & 0x18) >> 3) ); // P16 ......  P9
  _serial.putc( ((udc_data[4] & 0x0F) << 4) | ((udc_data[3] & 0x1E) >> 1) );                               // P24 ......  P17  
  _serial.putc( ((udc_data[6] & 0x03) << 6) | ((udc_data[5] & 0x1F) << 1) | ((udc_data[4] & 0x10) >> 4) ); // P32 ......  P25
  _serial.putc( ((udc_data[6] & 0x1C) >> 2) );                                                             // ... P35 P34 P33       
}

/** putc a single character (Stream implementation)
  * 
  * @param value char to print
  * @return value;
  */
int CU209SCPB::_putc(int value) {
    
    if ((value == '\n') || (value == '\r')) {
      //No character to display
      _serial.putc(D_CR); // Cursor to 0, Text remains as is
      
      //Update Cursor
      _column = 0;
    }
    else if ( ((value >= 0) && (value < CU209SCPB_NR_UDC)) ||
              ((value >= CU209SCPB_CHR_STRT) && (value < CU209SCPB_CHR_END)) ){

      //UDCs or Characters to display
      _serial.putc(value);
                              
      //Update Cursor
      _column++;
      if (_column > (CU209SCPB_NR_COLS - 1)) {        
        _column = 0;
      }
    }

    return value;
}

/** Get a single character (Stream implementation)
  *
  * @param none  
  * @return -1
  */
int CU209SCPB::_getc() {
    return -1;
}
