/* mbed CU209SCPB Library, for Noritake Itron CU209SCPB VFD module
 *
 * Copyright (c) 2017, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef CU209SCPB_H
#define CU209SCPB_H

// Select Noritake Itron CU209SCPB VFD module settings
#include "CU209SCPB_Config.h"

#include "CU209SCPB_UDC.h"

/** An interface for driving Noritake Itron CU209SCPB VFD module
 *
 * @code
 *
 * #include "mbed.h"
 * #include "CU209SCPB.h" 
 *
 * DigitalOut myled(LED1);
 * Serial pc(USBTX, USBRX);
 * 
 * // CU209SCPB declaration, Default setting
 * CU209SCPB CU209SCPB(p9); // TXD
 *
 * int main() {
 *   pc.printf("Hello World: CU209SCPB test\n\r");
 *
 *   CU209SCPB.cls(); 
 *
 *   CU209SCPB.putc('H');
 *   CU209SCPB.putc('e');
 *   CU209SCPB.putc('l');
 *   CU209SCPB.putc('l');
 *   CU209SCPB.putc('0');
 *   CU209SCPB.putc(' ');  
 *   CU209SCPB.printf("World);
 *   wait(2);
 *   CU209SCPB.setBrightness(CU209SCPB_BRT0);
 *   wait(2);
 *   CU209SCPB.setBrightness(CU209SCPB_BRT3);
 *           
 *   while(1) {
 *     myled = !myled;
 *     wait(1);
 *   }
 * }
 * #endif
 *
 * @endcode
 */


//CU209SCPB Display data
#define CU209SCPB_MAX_NR_GRIDS   20
#define CU209SCPB_BYTES_PER_GRID  1

//Memory size in bytes for Display
#define CU209SCPB_DSP_MEM    (CU209SCPB_MAX_NR_GRIDS * CU209SCPB_BYTES_PER_GRID)

//CU209SCPB Characters per line
#define CU209SCPB_NR_COLS     CU209SCPB_MAX_NR_GRIDS

//CU209SCPB User Defined Characters
#define CU209SCPB_NR_UDC          8
//#define CU209SCPB_UDC_MEM         8


//Serial control data consists of an 8-bit command and one or more data bytes.
//Command and data are sent LSB first. Data address is auto incremented.

//Command delay
#define CU209SCPB_CMD_DLY         8

//Commands Characters
#define D_BS                   0x08
#define D_HT                   0x09
#define D_LF                   0x0A
#define D_FF                   0x0C
#define D_CR                   0x0D
#define D_CLR                  0x0E

#define D_DC1                  0x11
#define D_DC2                  0x12
#define D_DC3                  0x13
#define D_DC4                  0x14
#define D_DC5                  0x15
#define D_DC6                  0x16
#define D_DC7                  0x17
#define D_CT0                  0x18
#define D_CT1                  0x19
#define D_ESC                  0x1B

//Display Characters
#define CU209SCPB_CHR_STRT     0x20
#define CU209SCPB_CHR_END      0xFF


//User Defined Characters (UDCs) are a 5x7 Matrix pattern that will show on the VFD as
// 0   P1  P2  P3  P4  P5
// 1   P6  P7 .....    P10
// .    .............
// .    .............
// .    .............
// 6   P31 P32 ...     P35
//

//UDCs are defined by sending a 5 byte bitpattern to the display.
//UDC Command: ESC, 'C', UDC idx
//Followed by:
//UDC Data (4th byte .. 8th byte)
//    D7  D6   D5   D4  D3  D2  D1  D0 
// 0  P8  P7   P6   ......  P3  P2  P1
// 1  P16 P15  P14  ........    P10 P9
// 2  P24 P23  P22  ..........      P17
// 3  P32 P31  P30  ............    P25
// 4  *   *    *    .... *  P35 P34 P33
//
//UDCs are defined by a 5x7 matrix and stored for convenience as 7 bytes with 5 significant bits
typedef char UDCData_t[7];

//Any character in the CU209SCPB ROM can be redefined as UDC.
//However, the lib restricts UDCs to the first 8 characters (which are not in use anyhow).
#define CU209SCPB_UADR_MSK     0x07


//Brightness Level (0..3)
#define CU209SCPB_BRT_0        0x00   //Duty  25%
#define CU209SCPB_BRT_1        0x40   //Duty  50%
#define CU209SCPB_BRT_2        0x80   //Duty  75%
#define CU209SCPB_BRT_3        0xC0   //Duty 100% (Default)

#define CU209SCPB_BRT_DEF      (CU209SCPB_BRT_1)

#define CU209SCPB_BRT_MSK      0xFF



/** A class for driving Noritake Itron CU209SCPB VFD module
 *
 *  @brief Supports 1 line of 20 chars (5x7 matrix segments).
 *         Serial bus interface device. 
 *  @param  PinName TX Serial bus pin
 *  @param  Baud baud selects baudrate (default 19200)
 *  @param  Int Parity selects paritybits (default Even)  
 */
class CU209SCPB : public Stream {

 public:

/** Enums
  */
  enum Baud {
    B_300   = 300,
    B_600   = 600,
    B_1200  = 1200,
    B_2400  = 2400,
    B_9600  = 9600,
    B_19200 = 19200
  };

/** Constructor for class for driving Noritake Itron CU209SCPB VFD module
  *
  *  @brief Supports 1 line of 20 chars (5x7 matrix segments).
  *         Serial bus interface device. 
  *  @param  PinName TX Serial bus pin
  *  @param  Baud baud selects baudrate (default 19200)  
  *  @param  Parity parity selects paritybits (default Even)  
  */
  CU209SCPB(PinName TXD, Baud baud = B_19200, SerialBase::Parity parity = SerialBase::Even);
      
  /** Clear the screen and locate to 0
    *
    * @param none
    * @return none  
    */
  void cls();

  /** Locate cursor to a screen column
    *
    * @param column  The horizontal position from the left, indexed from 0
    * @return none     
    */
  void locate(int column);
    
  /** Number of screen columns
    *
    * @param none
    * @return columns
    */
  int columns();  

#if DOXYGEN_ONLY
  /** Write a character to the Display
    *
    * @param c The character to write to the display
    * @return char written
    */
  int putc(int c);

  /** Write a formatted string to the Display
    *
    * @param format A printf-style format string, followed by the
    *               variables to use in formatting the string.
    */
  int printf(const char* format, ...);   
#endif

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (dutycycle linked to number of grids)  
    * @return none
    */    
  void setBrightness(char brightness = CU209SCPB_BRT_DEF);
  
  /** Set the Cursor mode On/off
    *
    * @param bool cursor mode
    * @return none    
    */
  void setCursor(bool on);

  /** Set Font
    *
    * @param  int font (valid range: 0 == International, other == Katakana)  
    * @return none
    */
  void setFont(int font = 0);

  /** Set User Defined Characters (UDC)
    *
    * @param unsigned char udc_idx   The Index of the UDC (0..7)
    * @param UDCData_t udc_data      The bitpattern for the UDC (7 bytes)
    * @return none    
    */
  void setUDC(unsigned char udc_idx, UDCData_t udc_data);

 protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();
 
 
  /** Init the Serial interface and the module
    *
    * @param  none
    * @return none
    */ 
  void _init();

 private:  
   Serial _serial;
   Baud _baud;
   SerialBase::Parity _parity;
  
   int _column;                     // Current cursor location
   int _columns;                    // Max number of columns
};
#endif

