/* mbed Microcontroller Library
 * Copyright (c) 2014, STMicroelectronics
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of STMicroelectronics nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include "mbed_assert.h"
#include "analoginDMA_api.h"

#if DEVICE_ANALOGIN

#include "wait_api.h"
#include "cmsis.h"
#include "pinmap.h"
#include "PeripheralPins.h"

ADC_HandleTypeDef AdcDMAHandle;
DMA_InitTypeDef DMA_InitStructure; //Variable used to setup the DMA
DMA_HandleTypeDef DMAHandle;

int adcDMA_inited = 0;

void analoginDMA_init(analogin_t *obj, PinName pin)
{
    // Get the peripheral name from the pin and assign it to the object
    obj->adc = (ADCName)pinmap_peripheral(pin, PinMap_ADC);
    MBED_ASSERT(obj->adc != (ADCName)NC);

    // Configure GPIO
    pinmap_pinout(pin, PinMap_ADC);

    // Save pin number for the read function
    obj->pin = pin;

    // The ADC initialization is done once
    if (adcDMA_inited == 0) {
        adcDMA_inited = 1;

        // Enable ADC clock
        __ADC1_CLK_ENABLE();

        // Configure ADC
        AdcDMAHandle.Instance = (ADC_TypeDef *)(obj->adc);
        AdcDMAHandle.Init.ClockPrescaler        = ADC_CLOCKPRESCALER_PCLK_DIV2;
        AdcDMAHandle.Init.Resolution            = ADC_RESOLUTION12b;
        AdcDMAHandle.Init.ScanConvMode          = DISABLE;
        AdcDMAHandle.Init.ContinuousConvMode    = ENABLE;
        AdcDMAHandle.Init.DiscontinuousConvMode = DISABLE;
        AdcDMAHandle.Init.NbrOfDiscConversion   = 0;
        AdcDMAHandle.Init.ExternalTrigConvEdge  = ADC_EXTERNALTRIGCONVEDGE_NONE;
        AdcDMAHandle.Init.ExternalTrigConv      = ADC_EXTERNALTRIGCONV_T1_CC1;
        AdcDMAHandle.Init.DataAlign             = ADC_DATAALIGN_RIGHT;
        AdcDMAHandle.Init.NbrOfConversion       = 1;
        AdcDMAHandle.Init.DMAContinuousRequests = ENABLE;
        AdcDMAHandle.Init.EOCSelection          = ENABLE;
        HAL_ADC_Init(&AdcDMAHandle);
 
    }
}

static inline HAL_StatusTypeDef adc_read(analogin_t *obj,uint16_t* pData, uint32_t Length)
{
    ADC_ChannelConfTypeDef sConfig;
    AdcDMAHandle.Instance = (ADC_TypeDef *)(obj->adc);

    // Configure ADC channel
    sConfig.Rank         = 1;
    sConfig.SamplingTime = ADC_SAMPLETIME_3CYCLES;
    sConfig.Offset       = 0;

    switch (obj->pin) {
        case PA_0:
            sConfig.Channel = ADC_CHANNEL_0;
            break;
        case PA_1:
            sConfig.Channel = ADC_CHANNEL_1;
            break;
        case PA_2:
            sConfig.Channel = ADC_CHANNEL_2;
            break;
        case PA_3:
            sConfig.Channel = ADC_CHANNEL_3;
            break;
        case PA_4:
            sConfig.Channel = ADC_CHANNEL_4;
            break;
        case PA_5:
            sConfig.Channel = ADC_CHANNEL_5;
            break;
        case PA_6:
            sConfig.Channel = ADC_CHANNEL_6;
            break;
        case PA_7:
            sConfig.Channel = ADC_CHANNEL_7;
            break;
        case PB_0:
            sConfig.Channel = ADC_CHANNEL_8;
            break;
        case PB_1:
            sConfig.Channel = ADC_CHANNEL_9;
            break;
        case PC_0:
            sConfig.Channel = ADC_CHANNEL_10;
            break;
        case PC_1:
            sConfig.Channel = ADC_CHANNEL_11;
            break;
        case PC_2:
            sConfig.Channel = ADC_CHANNEL_12;
            break;
        case PC_3:
            sConfig.Channel = ADC_CHANNEL_13;
            break;
        case PC_4:
            sConfig.Channel = ADC_CHANNEL_14;
            break;
        case PC_5:
            sConfig.Channel = ADC_CHANNEL_15;
            break;
        default:
            return HAL_ERROR;
        }

    HAL_ADC_ConfigChannel(&AdcDMAHandle, &sConfig);

    HAL_ADC_Start_DMA(&AdcDMAHandle, (uint32_t*) pData, Length);

    // Wait end of conversion and get value
    return (HAL_ADC_PollForConversion(&AdcDMAHandle, 1)); 
}

HAL_StatusTypeDef analogin_readdMA_u16(analogin_t *obj,uint16_t* pData, uint32_t Length)
{
    return adc_read(obj,pData, Length);
 
}

HAL_StatusTypeDef analoginDMA_read(analogin_t *obj,uint16_t* pData, uint32_t Length)
{
    return adc_read(obj, pData, Length);
}

#endif
