/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef MBED_ANALOGINDMA_H
#define MBED_ANALOGINDMA_H

#include "platform.h"

#if DEVICE_ANALOGIN

#include "analoginDMA_api.h"


namespace mbed {

/** An analog input, used for reading the voltage on a pin
 *
 * Example:
 * @code
 * // Print messages when the AnalogIn is greater than 50%
 *
 * #include "mbed.h"
 *
 * AnalogIn temperature(p20);
 *
 * int main() {
 *     while(1) {
 *         if(temperature > 0.5) {
 *             printf("Too hot! (%f)", temperature.read());
 *         }
 *     }
 * }
 * @endcode
 */
class AnalogInDMA {

public:

    /** Create an AnalogIn, connected to the specified pin
     *
     * @param pin AnalogIn pin to connect to
     * @param name (optional) A string to identify the object
     */
    AnalogInDMA(PinName pin) {
        analoginDMA_init(&_adc, pin);
    }

 
    /** Read the 'Length' input voltage(s), represented as an array of  unsigned short pData
     *
     * @returns status: 0 = OK, 1 = ERROR, 2=BUSY, 3 = TIMEOUT
     */
    HAL_StatusTypeDef read(uint16_t* pData, uint32_t Length) {
         return analoginDMA_read_u16(&_adc, pData, Length);
    }


protected:
    analogin_t _adc;
};

} // namespace mbed

#endif

#endif
