 
#include "GraphicLCD.h"
#include "mbed.h"
#include "stdio.h"

#define   MaxColumn    240
#define   MaxRow       64
#define   graphic_home_address  ((MaxColumn /8) *(MaxRow/8))
#define   LOW_BYTE  0
#define   HI_BYTE   30

const uint8_t Font_HelvR14[]  = {  // {Width, Height, Down, Start, Data30} https://github.com/olikraus/u8glib/blob/master/tools/font/bdf/courR14.bdf

    4,0,0,0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // Space
    8,4,0,0,0x03,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // ! Tilde
    6,5,0,0,0xCC,0x00,0xCC,0x00,0xCC,0x00,0x44,0x00,0x44,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // "
    8,15,1,0,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x12,0x00,0x7F,0x00,0x24,0x00,0x24,0x00,0x24,0x00,0xFE,0x00,0x24,0x00,0x24,0x00,0x24,0x00,0x24,0x00,0x24,0x00,    // #
    6,15,1,0,0x10,0x00,0x10,0x00,0x74,0x00,0x8C,0x00,0x84,0x00,0x80,0x00,0x60,0x00,0x18,0x00,0x04,0x00,0x84,0x00,0xC4,0x00,0xB8,0x00,0x10,0x00,0x10,0x00,0x10,0x00,    // $
    9,12,0,1,0x38,0x00,0x44,0x00,0x44,0x00,0x44,0x00,0x39,0x80,0x0E,0x00,0x30,0x00,0xCE,0x00,0x11,0x00,0x11,0x00,0x11,0x00,0x0E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // %
    7,10,0,4,0x38,0x00,0x40,0x00,0x40,0x00,0x40,0x00,0x20,0x00,0x74,0x00,0x88,0x00,0x88,0x00,0x98,0x00,0x66,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // &
    2,5,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x40,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // '
    3,14,0,0,0x20,0x00,0x40,0x00,0x40,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x40,0x00,0x40,0x00,0x20,0x00,0x00,0x00,    // (
    8,8,0,4,0x10,0x00,0x10,0x00,0x10,0x00,0xFE,0x00,0x10,0x00,0x28,0x00,0x44,0x00,0x44,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // )
    8,14,0,0,0x14,0x00,0x14,0x00,0x3E,0x00,0x7F,0x00,0x77,0x00,0xD4,0x00,0xD4,0x00,0xD4,0x00,0xD4,0x00,0x77,0x00,0x7F,0x00,0x3E,0x00,0x14,0x00,0x14,0x00,0x00,0x00,    // *
    9,9,0,3,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0xFF,0x80,0x08,0x00,0x08,0x00,0x08,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // +
    5,4,0,9,0x30,0x00,0x30,0x00,0x60,0x00,0x40,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // ,
    8,2,0,6,0xFF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // -
    2,2,0,12,0xC0,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // .
    8,16,0,0,0x01,0x00,0x01,0x00,0x02,0x00,0x02,0x00,0x04,0x00,0x04,0x00,0x08,0x00,0x08,0x00,0x10,0x00,0x10,0x00,0x20,0x00,0x20,0x00,0x40,0x00,0x40,0x00,0x80,0x00,    // /
    8,13,0,1,0x3C,0x00,0x7E,0x00,0x66,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x66,0x00,0x7E,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,    // 0
    5,13,0,1,0x18,0x00,0xF8,0x00,0xF8,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x00,0x00,    // 1
    8,13,0,1,0x3C,0x00,0xFE,0x00,0xC3,0x00,0x03,0x00,0x07,0x00,0x0E,0x00,0x1C,0x00,0x38,0x00,0x70,0x00,0xE0,0x00,0xC0,0x00,0xFF,0x00,0xFF,0x00,0x00,0x00,0x00,0x00,    // 2
    8,13,0,1,0x3E,0x00,0x7F,0x00,0xC3,0x00,0xC3,0x00,0x06,0x00,0x1C,0x00,0x1E,0x00,0x07,0x00,0x03,0x00,0xC3,0x00,0xC7,0x00,0x7E,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,    // 3
    9,13,0,1,0x03,0x00,0x07,0x00,0x0F,0x00,0x1B,0x00,0x33,0x00,0x33,0x00,0x63,0x00,0xC3,0x00,0xFF,0x80,0xFF,0x80,0x03,0x00,0x03,0x00,0x03,0x00,0x00,0x00,0x00,0x00,    // 4
    8,13,0,1,0xFE,0x00,0xFE,0x00,0xC0,0x00,0xC0,0x00,0xFC,0x00,0xFE,0x00,0xC7,0x00,0x03,0x00,0x03,0x00,0xC3,0x00,0xC7,0x00,0xFE,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,    // 5
    8,13,0,1,0x3C,0x00,0x7F,0x00,0x63,0x00,0xC0,0x00,0xC0,0x00,0xDC,0x00,0xFE,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xE3,0x00,0x7E,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,    // 6
    8,13,0,1,0xFF,0x00,0xFF,0x00,0x03,0x00,0x06,0x00,0x0C,0x00,0x0C,0x00,0x18,0x00,0x18,0x00,0x30,0x00,0x30,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x00,0x00,0x00,0x00,    // 7
    8,13,0,1,0x3C,0x00,0x7E,0x00,0xE7,0x00,0xC3,0x00,0xC3,0x00,0x66,0x00,0x7E,0x00,0xE7,0x00,0xC3,0x00,0xC3,0x00,0xE7,0x00,0x7E,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,    // 8
    8,13,0,1,0x3C,0x00,0x7E,0x00,0xC7,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x7F,0x00,0x3B,0x00,0x03,0x00,0x03,0x00,0xC6,0x00,0xFE,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,    // 9
    2,8,0,5,0xC0,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // :
    4,11,2,5,0x30,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x60,0x00,0x40,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // ;
    10,9,0,4,0x00,0xC0,0x03,0x00,0x0C,0x00,0x30,0x00,0xC0,0x00,0x30,0x00,0x0C,0x00,0x03,0x00,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // <
    9,4,0,5,0xFF,0x80,0x00,0x00,0x00,0x00,0xFF,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // =
    10,9,0,4,0xC0,0x00,0x30,0x00,0x0C,0x00,0x03,0x00,0x00,0xC0,0x03,0x00,0x0C,0x00,0x30,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // >
    7,11,0,2,0x7C,0x00,0x82,0x00,0x82,0x00,0x02,0x00,0x02,0x00,0x1C,0x00,0x10,0x00,0x10,0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // ?
    8,13,0,1,0x38,0x00,0x44,0x00,0x82,0x00,0x82,0x00,0x8E,0x00,0x92,0x00,0x92,0x00,0x92,0x00,0x8F,0x00,0x80,0x00,0x80,0x00,0x43,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,    // @
    12,14,0,0,0x06,0x00,0x06,0x00,0x0F,0x00,0x0F,0x00,0x19,0x80,0x19,0x80,0x30,0xC0,0x30,0xC0,0x3F,0xC0,0x7F,0xE0,0x60,0x60,0x60,0x60,0xC0,0x30,0xC0,0x30,0x00,0x00,   // A
    11,14,0,0,0xFF,0x00,0xFF,0x80,0xC1,0xC0,0xC0,0xC0,0xC0,0xC0,0xC1,0x80,0xFF,0x80,0xFF,0xC0,0xC0,0xE0,0xC0,0x60,0xC0,0x60,0xC0,0xE0,0xFF,0xC0,0xFF,0x80,0x00,0x00,   // B
    12,14,0,0,0x0F,0x80,0x3F,0xE0,0x70,0x70,0x60,0x30,0xE0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xE0,0x00,0x60,0x30,0x70,0x70,0x3F,0xE0,0x0F,0x80,0x00,0x00,   // C
    12,14,0,0,0xFF,0x80,0xFF,0xC0,0xC0,0xE0,0xC0,0x60,0xC0,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x60,0xC0,0xE0,0xFF,0xC0,0xFF,0x80,0x00,0x00,   // D
    10,14,0,0,0xFF,0xC0,0xFF,0xC0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xFF,0x80,0xFF,0x80,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xFF,0xC0,0xFF,0xC0,0x00,0x00,   // E
    9,14,0,0,0xFF,0x80,0xFF,0x80,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xFF,0x00,0xFF,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,    // F
    13,14,0,0,0x0F,0xC0,0x3F,0xF0,0x70,0x38,0x60,0x18,0xE0,0x18,0xC0,0x00,0xC0,0x00,0xC0,0xF8,0xC0,0xF8,0xE0,0x18,0x60,0x18,0x70,0x38,0x3F,0xF8,0x0F,0xD8,0x00,0x00,   // G
    11,14,0,0,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xFF,0xE0,0xFF,0xE0,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0x00,0x00,   // H
    2,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,    // I
    8,14,0,0,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0x03,0x00,0xC3,0x00,0xC3,0x00,0xE7,0x00,0x7E,0x00,0x3C,0x00,0x00,0x00,    // J
    12,14,0,0,0xC0,0xE0,0xC1,0xC0,0xC3,0x80,0xC7,0x00,0xCE,0x00,0xDC,0x00,0xF8,0x00,0xFC,0x00,0xCE,0x00,0xC7,0x00,0xC3,0x80,0xC1,0xC0,0xC0,0xE0,0xC0,0x70,0x00,0x00,   // K
    9,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xFF,0x80,0xFF,0x80,0x00,0x00,    // L
    14,14,0,0,0xC0,0x0C,0xC0,0x0C,0xE0,0x1C,0xE0,0x1C,0xF0,0x3C,0xF0,0x3C,0xD8,0x6C,0xD8,0x6C,0xCC,0xCC,0xCC,0xCC,0xC4,0x8C,0xC7,0x8C,0xC3,0x0C,0xC3,0x0C,0x00,0x00,   // M
    11,14,0,0,0xC0,0x60,0xE0,0x60,0xF0,0x60,0xF0,0x60,0xD8,0x60,0xCC,0x60,0xCC,0x60,0xC6,0x60,0xC6,0x60,0xC3,0x60,0xC1,0xE0,0xC1,0xE0,0xC0,0xE0,0xC0,0x60,0x00,0x00,   // N
    13,14,0,0,0x0F,0x80,0x3F,0xE0,0x70,0x70,0x60,0x30,0xE0,0x38,0xC0,0x18,0xC0,0x18,0xC0,0x18,0xC0,0x18,0xE0,0x38,0x60,0x30,0x70,0x70,0x3F,0xE0,0x0F,0x80,0x00,0x00,   // O
    10,14,0,0,0xFF,0x00,0xFF,0x80,0xC1,0xC0,0xC0,0xC0,0xC0,0xC0,0xC1,0xC0,0xFF,0x80,0xFF,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,   // P
    13,15,1,0,0x0F,0x80,0x3F,0xE0,0x70,0x70,0x60,0x30,0xE0,0x38,0xC0,0x18,0xC0,0x18,0xC0,0x18,0xC0,0x18,0xE1,0xB8,0x61,0xB0,0x70,0xF0,0x3F,0xE0,0x0F,0xB0,0x00,0x30,   // Q
    11,14,0,0,0xFF,0x80,0xFF,0xC0,0xC0,0xE0,0xC0,0x60,0xC0,0x60,0xC0,0xE0,0xFF,0xC0,0xFF,0x80,0xC0,0xC0,0xC0,0xC0,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0x00,0x00,   // R
    10,14,0,0,0x1E,0x00,0x7F,0x80,0xE1,0xC0,0xC0,0xC0,0xE0,0x00,0x7C,0x00,0x1F,0x00,0x03,0x80,0x01,0xC0,0x00,0xC0,0xC0,0xC0,0xE1,0xC0,0x7F,0x80,0x3F,0x00,0x00,0x00,   // S
    10,14,0,0,0xFF,0xC0,0xFF,0xC0,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x0C,0x00,0x00,0x00,   // T
    11,14,0,0,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0xC0,0x60,0x60,0xC0,0x7F,0xC0,0x1F,0x00,0x00,0x00,   // U
    12,14,0,0,0xC0,0x30,0xC0,0x30,0x60,0x60,0x60,0x60,0x60,0x60,0x30,0xC0,0x30,0xC0,0x30,0xC0,0x19,0x80,0x19,0x80,0x19,0x80,0x0F,0x00,0x0F,0x00,0x06,0x00,0x00,0x00,   // V
    16,14,0,0,0xC1,0x83,0xC1,0x83,0xC1,0x83,0xC3,0xC3,0x63,0xC6,0x62,0x46,0x66,0x66,0x66,0x66,0x36,0x6C,0x36,0x6C,0x34,0x2C,0x1C,0x38,0x18,0x18,0x18,0x18,0x00,0x00,   // W
    11,14,0,0,0xC0,0x60,0xC0,0x60,0x60,0xC0,0x71,0xC0,0x31,0x80,0x1B,0x00,0x0E,0x00,0x0E,0x00,0x1B,0x00,0x31,0x80,0x71,0xC0,0x60,0xC0,0xC0,0x60,0xC0,0x60,0x00,0x00,   // X
    12,14,0,0,0xC0,0x30,0xC0,0x30,0x60,0x60,0x60,0x60,0x30,0xC0,0x39,0xC0,0x19,0x80,0x0F,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x06,0x00,0x00,0x00,   // Y
    10,14,0,0,0xFF,0xC0,0xFF,0xC0,0x00,0xC0,0x01,0x80,0x03,0x00,0x06,0x00,0x0C,0x00,0x1C,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0xC0,0x00,0xFF,0xC0,0xFF,0xC0,0x00,0x00,   // Z
    4,15,1,0,0xF0,0x00,0xF0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xF0,0x00,0xF0,0x00,    // [
    5,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x60,0x00,0x60,0x00,0x60,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x00,0x00,    // /
    4,15,1,0,0xF0,0x00,0xF0,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0xF0,0x00,0xF0,0x00,    // ]
    7,6,0,0,0x10,0x00,0x38,0x00,0x6C,0x00,0x6C,0x00,0xC6,0x00,0xC6,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // ^
    11,2,0,12,0xFF,0xE0,0xFF,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // _
    4,3,0,0,0xC0,0x00,0x60,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,     // "
    9,10,0,4,0x7E,0x00,0xE7,0x00,0xC3,0x00,0x07,0x00,0x7F,0x00,0xE3,0x00,0xC3,0x00,0xC3,0x00,0xE7,0x80,0x79,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // a
    9,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xDE,0x00,0xFF,0x00,0xE3,0x00,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xE3,0x00,0xFF,0x00,0xDE,0x00,0x00,0x00,    // b
    8,14,0,4,0x3E,0x00,0x7F,0x00,0x63,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x63,0x00,0x7F,0x00,0x3E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // c
    9,14,0,0,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x3D,0x80,0x7F,0x80,0x63,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0x63,0x80,0x7F,0x80,0x3D,0x80,0x00,0x00,    // d
    8,10,0,4,0x3C,0x00,0x7E,0x00,0xC3,0x00,0xC3,0x00,0xFF,0x00,0xC0,0x00,0xC0,0x00,0xE3,0x00,0x7F,0x00,0x3C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // e
    6,14,0,0,0x1C,0x00,0x3C,0x00,0x30,0x00,0x30,0x00,0xFC,0x00,0xFC,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x00,0x00,    // f
    9,14,4,4,0x3D,0x80,0x7F,0x80,0x61,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0x63,0x80,0x7F,0x80,0x3D,0x80,0x01,0x80,0x63,0x00,0x7F,0x00,0x1C,0x00,0x00,0x00,    // g
    8,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xDE,0x00,0xFF,0x00,0xE3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x00,0x00,    // h
    2,12,0,2,0xC0,0x00,0xC0,0x00,0x00,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,    // i
    4,15,4,3,0x30,0x00,0x30,0x00,0x00,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x30,0x30,0xF0,0xE0,0x00,    // j
    8,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC6,0x00,0xCC,0x00,0xD8,0x00,0xF0,0x00,0xF8,0x00,0xD8,0x00,0xCC,0x00,0xCE,0x00,0xC6,0x00,0xC7,0x00,0x00,0x00,    // k
    2,14,0,0,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,    // l
    14,10,0,4,0xDE,0x78,0xFF,0xFC,0xE3,0x8C,0xC3,0x0C,0xC3,0x0C,0xC3,0x0C,0xC3,0x0C,0xC3,0x0C,0xC3,0x0C,0xC3,0x0C,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,   // m
    8,10,0,4,0xDE,0x00,0xFF,0x00,0xE3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // n
    9,10,0,4,0x3E,0x00,0x7F,0x00,0x63,0x00,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0x63,0x00,0x7F,0x00,0x3E,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // o
    9,14,4,4,0xDE,0x00,0xFF,0x00,0xE3,0x00,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xE3,0x00,0xFF,0x00,0xDE,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,    // p
    9,14,4,4,0x3D,0x80,0x7F,0x80,0x63,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0xC1,0x80,0x63,0x80,0x7F,0x80,0x3D,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x00,0x00,    // q
    5,10,0,4,0xD8,0x00,0xD8,0x00,0xE0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // r
    7,10,0,4,0x3C,0x00,0x7E,0x00,0xC6,0x00,0xC0,0x00,0xFC,0x00,0x3E,0x00,0x06,0x00,0xC6,0x00,0xFC,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // s
    6,13,0,1,0x30,0x00,0x30,0x00,0x30,0x00,0xFC,0x00,0xFC,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x30,0x00,0x3C,0x00,0x1C,0x00,0x00,0x00,0x00,0x00,    // t
    8,10,0,4,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC3,0x00,0xC7,0x00,0xFF,0x00,0x7B,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // u
    8,10,0,4,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x24,0x00,0x3C,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // v
    12,10,0,4,0xC6,0x30,0xC6,0x30,0xC6,0x30,0x66,0x60,0x66,0x60,0x69,0x60,0x29,0x40,0x39,0xC0,0x19,0x80,0x19,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,   // w
    8,10,0,4,0xC3,0x00,0xE7,0x00,0x66,0x00,0x3C,0x00,0x18,0x00,0x18,0x00,0x3C,0x00,0x66,0x00,0xE7,0x00,0xC3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // x
    8,14,4,4,0xC3,0x00,0xC3,0x00,0xC3,0x00,0x66,0x00,0x66,0x00,0x66,0x00,0x24,0x00,0x3C,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x70,0x00,0x70,0x00,0x00,0x00,    // y
    7,10,0,4,0xFE,0x00,0xFE,0x00,0x06,0x00,0x0C,0x00,0x18,0x00,0x30,0x00,0x60,0x00,0xC0,0x00,0xFE,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,    // z
};

uint8_t GLCD_Picture[30][64];

GLCD::GLCD(PinName D1, PinName D2,PinName D3, PinName D4,PinName D5, PinName D6,PinName D7, PinName D8,PinName CD,PinName RESET,PinName CE,PinName WR,PinName RD,PinName Power, PinName Light) : _D1(D1),_D2(D2),_D3(D3),_D4(D4),_D5(D5),_D6(D6),_D7(D7),_D8(D8),_CD(CD),_RESET(RESET),_CE(CE),_WR(WR),_RD(RD), _Power(Power), _LIGHT(Light){
    _D1 = 0;
    _D2 = 0;
    _D3 = 0;
    _D4 = 0;
    _D5 = 0;
    _D6 = 0;
    _D7 = 0;
    _D8 = 0;
    _CD = 0;
    _CE = 0;
    _WR = 0;
    _RD = 0;
    _RESET = 0;
    _Power = 1;
    _LIGHT = 0;
}
void GLCD::WriteData(uint8_t Data)
{
    _D1=((Data >> 0) & 1);
    _D2=((Data >> 1) & 1);
    _D3=((Data >> 2) & 1);
    _D4=((Data >> 3) & 1);
    _D5=((Data >> 4) & 1);
    _D6=((Data >> 5) & 1);
    _D7=((Data >> 6) & 1);
    _D8=((Data >> 7) & 1);

    wait_us(10);
    _CD=0;
    wait_us(10);
    _CE=0;
    wait_us(10);
    _WR=0;
    wait_us(15);
    _CE=1;
    wait_us(10);
    _WR=1;
    wait_us(10);
}

void GLCD::WriteCommand(uint8_t Command)
{
    _D1=((Command >> 0) & 1);
    _D2=((Command >> 1) & 1);
    _D3=((Command >> 2) & 1);
    _D4=((Command >> 3) & 1);
    _D5=((Command >> 4) & 1);
    _D6=((Command >> 5) & 1);
    _D7=((Command >> 6) & 1);
    _D8=((Command >> 7) & 1);

    wait_us(10);
    _CD=1;
    wait_us(10);
    _CE=0;
    wait_us(10);
    _WR=0;
    wait_us(15);
    _CE=1;
    wait_us(10);
    _WR=1;
    wait_us(10);
} 

void GLCD::Limits()
{
    uint8_t x,y;
    GLCD_Picture[0][0]=0x1F;
    GLCD_Picture[0][63]=0x1F;
    GLCD_Picture[29][0]=0xF8;
    GLCD_Picture[29][63]=0xF8;

    GLCD_Picture[0][1]=0x20;
    GLCD_Picture[29][1]=0x04;
    GLCD_Picture[0][62]=0x20;
    GLCD_Picture[29][62]=0x04;

    GLCD_Picture[0][2]=0x40;
    GLCD_Picture[29][2]=0x02;
    GLCD_Picture[0][61]=0x40;
    GLCD_Picture[29][61]=0x02;

    for (x=1; x<29; x++) {
        GLCD_Picture[x][0]=0xFF;
        GLCD_Picture[x][63]=0xFF;
    }
    for (y=3; y<61; y++) {
        GLCD_Picture[0][y]=0x80;
        GLCD_Picture[29][y]=0x01;
    }
}
 
void GLCD::Fill()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0xFF);
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Clean()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x00);
            GLCD_Picture[x][y]=0x00;
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::On(uint32_t Hora)
{
    _Power = 0;
    wait_us(50);
    if(Hora>990 || Hora<360){
        _LIGHT=1;
    } else {
        _LIGHT=0;
    }
}

void GLCD::Off()
{
    _D1=0;
    _D2=0;
    _D3=0;
    _D4=0;
    _D5=0;
    _D6=0;
    _D7=0;
    _D8=0;
    _CD=0;
    _CE=0;
    _WR=0;
    _RD = 0;
    _RESET = 0;
    _Power = 1;
    _LIGHT = 0;
}


void GLCD::Horizontal()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    for (y=0; y<MaxRow/2; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0xFF);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x00);
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Cross()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow/8; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x01);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x03);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x07);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x0F);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x1F);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x3F);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x7F);
        }
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0xFF);
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Vertical()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(0x0F);
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Show()
{
    uint8_t x,y;
    WriteCommand(0x98);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(GLCD_Picture[x][y]);
            //print_bits(GLCD_Picture[x][y]);
            GLCD_Picture[x][y]=0x00;
        }
        //Computer.printf("\n");
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Characters()
{
    uint8_t x,y,k=0;
    WriteCommand(0x94);        //Text display only
    WriteData(0x00);
    WriteData(0x00);
    WriteCommand(0x24);
    WriteCommand(0xB0);
    
    for (y=0; y<MaxRow/8; y++) {
        for (x=0; x<MaxColumn/8; x++) {
            WriteData(k+6);
            k++;
            if(k==0x80) {
                k=0x00;
            }
        }
    }
    WriteCommand(0xB2); //set auto write reset
}

void GLCD::Init()
{
    uint8_t    LOW_BYTE1;

    _RESET=0;
    wait_us(120);
    _RESET=1;
    wait_us(120);
    WriteData(0x00);
    WriteData(0x00);
    WriteCommand(0x40);
    WriteData(LOW_BYTE);
    WriteData(HI_BYTE);
    WriteCommand(0x42);
    LOW_BYTE1 = MaxColumn /8;
    WriteData(LOW_BYTE1);
    WriteData(0x00);
    WriteCommand(0x41);
    LOW_BYTE1 = MaxColumn /8;
    WriteData(LOW_BYTE1);
    WriteData(0x00);
    WriteCommand(0x43);
    WriteCommand(0x80);
    WriteCommand(0x98);
}

void GLCD::Pixel(int x, int y)
{
    if(x > MaxColumn || y > MaxRow || x < 0 || y < 0) {  // first check parameter
        return;
    } else {                                             // XOR mode
        GLCD_Picture[(x/8)][y] ^= (1 << (7-(x%8)));
    }
}

void GLCD::Rectangle(int X0, int Y0, int X1, int Y1)
{
    int LongR, LongC, Var;  
    Y1=Y1+Y0;                    // Rotate variables
    X1=X1+X0;  
    if(X0 > X1) {
        Var = X0;
        X0 = X1;
        X1 = Var;
    }
    if(Y0 > Y1) {
        Var = Y0;
        Y0 = Y1;
        Y1 = Var;
    }
    for(LongR=X0; LongR<=X1; LongR++) {        // Limits and export by pixel funcion
        for(LongC=Y0; LongC<=Y1; LongC++) {
            Pixel(LongR,LongC);
        }
    }
}
int  GLCD::PutChar(int X,int Y,int Letter)          // recibe position XY and position of the letter in font array
{
    Letter=(Letter-32)*34;
    int Char_Width=Font_HelvR14[Letter];// {Width, Height, Down, Start, Data30}
    int Char_Start=Y+Font_HelvR14[Letter+3];
    int Char_Finish=Y+Font_HelvR14[Letter+1]+Font_HelvR14[Letter+3];
    int Byte1=X/8;                         // 3 pixel bytes per char, start in X
    int Byte2=Byte1+1;
    int Byte3=Byte1+2;
    int ByteLine=Letter+4;
    int ByteShiftRight=X%8;
    int ByteShiftLeft=8-(X%8);
    if ((Char_Finish>MaxRow)||((Char_Width+X)>MaxColumn)) {
        return 0;
    }
    if (Char_Width>8) {
        for(int Line=Char_Start; Line<Char_Finish; Line++) {
            GLCD_Picture[Byte1][Line] |= Font_HelvR14[ByteLine]>>ByteShiftRight;
            GLCD_Picture[Byte2][Line] |= Font_HelvR14[ByteLine]<<ByteShiftLeft;
            GLCD_Picture[Byte2][Line] |= Font_HelvR14[ByteLine+1]>>ByteShiftRight;
            GLCD_Picture[Byte3][Line] |= Font_HelvR14[ByteLine+1]<<ByteShiftLeft;
            ByteLine+=2;
        }
        return Char_Width;
    }
    if (Char_Width<=8) {
        for(int Line=Char_Start; Line<Char_Finish; Line++) {
            GLCD_Picture[Byte1][Line] |= Font_HelvR14[ByteLine]>>ByteShiftRight;
            GLCD_Picture[Byte2][Line] |= Font_HelvR14[ByteLine]<<ByteShiftLeft;
            ByteLine+=2;
        }
        return Char_Width;
    }
    return 0;
}

void GLCD::PutString(int X, int Y, char String[])
{
    int Char;
    int Position=X;

    for(int StringLine=0; StringLine<strlen(String); StringLine++) {
        Char=String[StringLine];
        Position+=GLCD::PutChar(Position,Y,Char)+2;
    }
} 
