#include "MotorControl.h"
#include "mbed.h"

// constructor 
MotorControl::MotorControl(PinName forward, PinName backward, int period, int safetyPeriod) :  
        _forward(forward),
        _backward(backward) {
    
    _forward = 0; // turn motor off
    _backward = 0; // turn motor off
    
    _period = period;
    _safetyPeriod = safetyPeriod;
}

// forward at full speed
void MotorControl::forward(int duration) {
    if(duration < _safetyPeriod) return;
    if(duration < CHECK_TIME) return;
    
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    
    int total_cycles = duration / CHECK_TIME;
    
    // turn on forward direction
    _forward = 1;
    for(int i = 0; i < total_cycles; i++)
    {
        if(_backward.read()) break;  // periodically check that _backward has not turned on
        wait_us(CHECK_TIME);
        }
        
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    }
    
// backward at full speed
void MotorControl::backward(int duration) {
    if(duration < _safetyPeriod) return;
    if(duration < CHECK_TIME) return;
    
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    
    int total_cycles = duration / CHECK_TIME;
    
    // turn on forward direction
    _backward = 1;
    for(int i = 0; i < total_cycles; i++)
    {
        if(_forward.read()) break;  // periodically check that _forward has not turned on
        wait_us(CHECK_TIME);
        }
        
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    }

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 * 
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */ 
void MotorControl::forward(float duty_cycle, uint32_t duration) {
    if(duty_cycle > 1) return;
    if(duration < _period) return;
    
    // turn off motors and wait sufficient time to guarantee motors are off
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    
    __disable_irq();    // Disable Interrupts
    
    // perform some calculations
    uint32_t total_cycles = duration / _period;
    uint32_t on_cycles = (uint32_t) _period * duty_cycle;
    uint32_t off_cycles = _period - on_cycles;
    uint32_t on_cycles_sub = on_cycles / CHECK_TIME;
    
    
    for(int i = 0; i < total_cycles; i++)
    {
        if(_backward.read()) return;
        _forward = 1;
        
        for(int j = 0; j < on_cycles_sub; j++)
        {
            if(_backward.read()) break;  // periodically check that _backward is off
            wait_us(CHECK_TIME);
            }
        
        _forward = 0;
        _backward = 0;
        wait_us(off_cycles);
        }
    
    __enable_irq();     // Enable Interrupts
    wait_us(_safetyPeriod);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void MotorControl::backward(float duty_cycle, uint32_t duration) {
    if(duty_cycle > 1) return;
    if(duration < _period) return;
    
    // turn off motors and wait sufficient time to guarantee motors are off
    _forward = 0;
    _backward = 0;
    wait_us(_safetyPeriod);
    
    __disable_irq();    // Disable Interrupts
    
    // perform some calculations
    uint32_t total_cycles = duration / _period;
    uint32_t on_cycles = (uint32_t)_period * duty_cycle;
    uint32_t off_cycles = _period - on_cycles;
    uint32_t on_cycles_sub = on_cycles / CHECK_TIME;
    
    
    for(int i = 0; i < total_cycles; i++)
    {
        if(_forward.read()) return;
        _backward = 1;
        
        for(int j = 0; j < on_cycles_sub; j++)
        {
            if(_forward.read()) break;  // periodically check that _forward is off
            wait_us(CHECK_TIME);
            }
        
        _forward = 0;
        _backward = 0;
        wait_us(off_cycles);
        }
    
    __enable_irq();     // Enable Interrupts
    wait_us(_safetyPeriod);
}