// /*******************************************************************************
// * Copyright (C) 2019 Maxim Integrated Products, Inc., All Rights Reserved.
// *
// * Permission is hereby granted, free of charge, to any person obtaining a
// * copy of this software and associated documentation files (the "Software"),
// * to deal in the Software without restriction, including without limitation
// * the rights to use, copy, modify, merge, publish, distribute, sublicense,
// * and/or sell copies of the Software, and to permit persons to whom the
// * Software is furnished to do so, subject to the following conditions:
// *
// * The above copyright notice and this permission notice shall be included
// * in all copies or substantial portions of the Software.
// *
// * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
// * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// * OTHER DEALINGS IN THE SOFTWARE.
// *
// * Except as contained in this notice, the name of Maxim Integrated
// * Products, Inc. shall not be used except as stated in the Maxim Integrated
// * Products, Inc. Branding Policy.
// *
// * The mere transfer of this software does not imply any licenses
// * of trade secrets, proprietary technology, copyrights, patents,
// * trademarks, maskwork rights, or any other form of intellectual
// * property whatsoever. Maxim Integrated Products, Inc. retains all
// * ownership rights.
// *******************************************************************************
// */
// *********************************************************************
// @file MaximTinyTester.h
// *********************************************************************

// Prevent multiple declaration
#ifndef __MaximTinyTester_H__
#define __MaximTinyTester_H__

#include "mbed.h"
#include "CmdLine.h"  // https://whismanoid@os.mbed.com/users/whismanoid/code/CmdLine/

/**
 * @brief MaximTinyTester class supports software and hardware testing
 *
 */
class MaximTinyTester
{
protected:
    CmdLine& associatedCmdLine;

public:
    /** Constructor for MaximTinyTester class.
    *
    * @param[in] AssociatedCmdLine = reference to serial CmdLine
    * @param[in] analogInPin0 = port for analog input measurement
    * @param[in] analogInPin1 = port for analog input measurement
    * @param[in] analogInPin2 = port for analog input measurement
    * @param[in] analogInPin3 = port for analog input measurement
    * @param[in] analogInPin4 = port for analog input measurement
    * @param[in] analogInPin5 = port for analog input measurement
    * @param[in] m_RFailLED = port for red LED; use NC if not connected
    * @param[in] m_GPassLED = port for green LED; use NC if not connected
    * @param[in] m_BBusyLED = port for blue LED; use NC if not connected
    *
    */
    MaximTinyTester(CmdLine& AssociatedCmdLine,
        AnalogIn& analogInPin0,
        AnalogIn& analogInPin1,
        AnalogIn& analogInPin2,
        AnalogIn& analogInPin3,
        AnalogIn& analogInPin4,
        AnalogIn& analogInPin5,
        DigitalOut& m_RFailLED,
        DigitalOut& m_GPassLED,
        DigitalOut& m_BBusyLED);

    AnalogIn& analogInPin0;
    AnalogIn& analogInPin1;
    AnalogIn& analogInPin2;
    AnalogIn& analogInPin3;
    AnalogIn& analogInPin4;
    AnalogIn& analogInPin5;
    float analogInPin_fullScaleVoltage[6];

    // MaximTinyTester add LED indicators m_RFailLED, m_GPassLED, m_BBusyLED
    DigitalOut& m_RFailLED;
    DigitalOut& m_GPassLED;
    DigitalOut& m_BBusyLED;
    int blink_time_msec;

    /** err_threshold determines how closely a float or double result must match the expected value
    */
    double err_threshold;

    /** mask determines which bits must match the expected value
    */
    int mask;

    void clear();

    /** serial returns reference to the associated serial port */
    CmdLine& cmdLine(void) const {
        return associatedCmdLine;
    };

    int nPass;
    int nFail;

    void PASS();

    void FAIL();

    void print(const char* stringLiteralMessage);

    void Report_Summary(void);


    bool Expect(const char *nameOfTest, int actual_result, int expect_result);
    bool Expect(const char *nameOfTest, double actual_result, double expect_result);

    bool FunctionCall_u_f_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint16_t(double)> functionUnderTest,
        double arg_1_voltageV,
        uint16_t expect_result);

    // MaximTinyTester FunctionCall_lu_f_Expect support MAX5719
    bool FunctionCall_lu_f_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint32_t(double)> functionUnderTest,
        double arg_1_voltageV,
        uint32_t expect_result);

    bool FunctionCall_f_d_Expect(const char *nameOfFunctionUnderTest, 
        Callback<double(int)> functionUnderTest,
        int arg_1_u16,
        double expect_result);

    bool FunctionCall_f_d_d_Expect(const char *nameOfFunctionUnderTest, 
        Callback<double(int, int)> functionUnderTest,
        int arg_1_int,
        int arg_2_int,
        double expect_result);

    bool FunctionCall_f_u_Expect(const char *nameOfFunctionUnderTest, 
        Callback<double(uint16_t)> functionUnderTest,
        uint16_t arg_1_u16,
        double expect_result);

    bool FunctionCall_f_lu_Expect(const char *nameOfFunctionUnderTest, 
        Callback<double(uint32_t)> functionUnderTest,
        uint32_t arg_1_u32,
        double expect_result);

    bool FunctionCall_f_f_Expect(const char *nameOfFunctionUnderTest, 
        Callback<double(double)> functionUnderTest,
        double arg_1_d,
        double expect_result);

    bool FunctionCall_d_Expect(const char *nameOfFunctionUnderTest, 
        Callback<int()> functionUnderTest,
        int expect_result);

    bool FunctionCall_md_Expect(const char *nameOfFunctionUnderTest, 
        Callback<int16_t()> functionUnderTest,
        int16_t expect_result);

    bool FunctionCall_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t()> functionUnderTest,
        uint8_t expect_result);

    bool FunctionCall_su_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(uint8_t)> functionUnderTest,
        uint8_t arg_1,
        uint8_t expect_result);

    bool FunctionCall_su_d_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(int)> functionUnderTest,
        int arg_1,
        uint8_t expect_result);

    bool FunctionCall_su_su_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(uint8_t, uint8_t)> functionUnderTest,
        uint8_t arg_1,
        uint8_t arg_2,
        uint8_t expect_result);

    bool FunctionCall_su_d_d_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(int, int)> functionUnderTest,
        int arg_1,
        int arg_2,
        uint8_t expect_result);

    bool FunctionCall_su_d_lu_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(int, uint32_t)> functionUnderTest,
        int arg_1,
        uint32_t arg_2,
        uint8_t expect_result);


    bool FunctionCall_su_d_plu_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(int, uint32_t*)> functionUnderTest,
        int arg_1,
        uint32_t* arg_2,
        uint8_t expect_result,
        uint32_t expect_buffer);

    bool FunctionCall_su_su_su_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(uint8_t, uint8_t, uint8_t)> functionUnderTest,
        uint8_t arg_1,
        uint8_t arg_2,
        uint8_t arg_3,
        uint8_t expect_result);

    bool FunctionCall_su_su_su_su_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(uint8_t, uint8_t, uint8_t, uint8_t)> functionUnderTest,
        uint8_t arg_1,
        uint8_t arg_2,
        uint8_t arg_3,
        uint8_t arg_4,
        uint8_t expect_result);

    bool FunctionCall_su_su_su_su_su_su_Expect(const char *nameOfFunctionUnderTest, 
        Callback<uint8_t(uint8_t, uint8_t, uint8_t, uint8_t, uint8_t)> functionUnderTest,
        uint8_t arg_1,
        uint8_t arg_2,
        uint8_t arg_3,
        uint8_t arg_4,
        uint8_t arg_5,
        uint8_t expect_result);

// TODO1 #167 support callback<uint8_t(uint8_t, uint8_t, uint8_t, uint8_t, uint8_t, uint8_t)>
//    bool FunctionCall_su_su_su_su_su_su_su_Expect(const char *nameOfFunctionUnderTest, 
//        Callback<uint8_t(uint8_t, uint8_t, uint8_t, uint8_t, uint8_t, uint8_t)> functionUnderTest,
//        uint8_t arg_1,
//        uint8_t arg_2,
//        uint8_t arg_3,
//        uint8_t arg_4,
//        uint8_t arg_5,
//        uint8_t arg_6,
//        uint8_t expect_result);

    int input_timeout_time_msec;

    bool AnalogIn0_Read_Expect_voltageV(double expect_result);
    bool AnalogIn1_Read_Expect_voltageV(double expect_result);
    bool AnalogIn2_Read_Expect_voltageV(double expect_result);
    bool AnalogIn3_Read_Expect_voltageV(double expect_result);
    bool AnalogIn4_Read_Expect_voltageV(double expect_result);
    bool AnalogIn5_Read_Expect_voltageV(double expect_result);

    bool AnalogIn_Read_Expect_voltageV(AnalogIn& analogInPin, double expect_result);

    double AnalogIn0_Read_Report_voltageV();
    double AnalogIn1_Read_Report_voltageV();
    double AnalogIn2_Read_Report_voltageV();
    double AnalogIn3_Read_Report_voltageV();
    double AnalogIn4_Read_Report_voltageV();
    double AnalogIn5_Read_Report_voltageV();

    bool DigitalIn_Read_Expect_WarnOnly(DigitalIn& digitalInPin, const char* pinName, int expect_result, const char *expect_description);

    int settle_time_msec;

    void Wait_Output_Settling();

};

#endif // __MaximTinyTester_H__

// End of file
