// /*******************************************************************************
// * Copyright (C) 2019 Maxim Integrated Products, Inc., All Rights Reserved.
// *
// * Permission is hereby granted, free of charge, to any person obtaining a
// * copy of this software and associated documentation files (the "Software"),
// * to deal in the Software without restriction, including without limitation
// * the rights to use, copy, modify, merge, publish, distribute, sublicense,
// * and/or sell copies of the Software, and to permit persons to whom the
// * Software is furnished to do so, subject to the following conditions:
// *
// * The above copyright notice and this permission notice shall be included
// * in all copies or substantial portions of the Software.
// *
// * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
// * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// * OTHER DEALINGS IN THE SOFTWARE.
// *
// * Except as contained in this notice, the name of Maxim Integrated
// * Products, Inc. shall not be used except as stated in the Maxim Integrated
// * Products, Inc. Branding Policy.
// *
// * The mere transfer of this software does not imply any licenses
// * of trade secrets, proprietary technology, copyrights, patents,
// * trademarks, maskwork rights, or any other form of intellectual
// * property whatsoever. Maxim Integrated Products, Inc. retains all
// * ownership rights.
// *******************************************************************************
// */
// *********************************************************************
// @file CmdLine.h
// *********************************************************************

// Prevent multiple declaration
#ifndef __CmdLine_H__
#define __CmdLine_H__

// standard include for target platform
#include "mbed.h"

/**
 @brief CmdLine class manages an editable serial input line buffer
 */
class CmdLine
{
/// @page Immediate Immediate command character
/// Class CmdLine supports immeidate command character handler functions.
/// Immediate command characters get handled immediately,
/// without waiting for End Of Line CR or LF,
/// and without appearing in the command buffer.
/// Avoid using characters that may appear in other commands,
/// such as 0-9 A-Z a-z and some punctuation %*+-./=
/// so these 25 characters are available: !"#$&'(),:;<>?@[\]^_`{|}~
/// @see on_immediate_0x21 handler for Unicode (U+0021) ! EXCLAMATION MARK
/// @see on_immediate_0x22 handler for Unicode (U+0022) " QUOTATION MARK
/// @see on_immediate_0x23 handler for Unicode (U+0023) # NUMBER SIGN = pound sign, hash, crosshatch
/// @see on_immediate_0x24 handler for Unicode (U+0024) $ DOLLAR SIGN
/// @see on_immediate_0x26 handler for Unicode (U+0026) & AMPERSAND
/// @see on_immediate_0x27 handler for Unicode (U+0027) ' APOSTROPHE
/// @see on_immediate_0x28 handler for Unicode (U+0028) ( LEFT PARENTHESIS
/// @see on_immediate_0x29 handler for Unicode (U+0029) ) RIGHT PARENTHESIS
/// @see on_immediate_0x2c handler for Unicode (U+002C) , COMMA
/// @see on_immediate_0x3a handler for Unicode (U+003A) : COLON
/// @see on_immediate_0x3b handler for Unicode (U+003B) ; SEMICOLON
/// @see on_immediate_0x3c handler for Unicode (U+003C) < LESS-THAN SIGN
/// @see on_immediate_0x3e handler for Unicode (U+003E) > GREATER-THAN SIGN
/// @see on_immediate_0x3f handler for Unicode (U+003F) ? QUESTION MARK
/// @see on_immediate_0x40 handler for Unicode (U+0040) @ COMMERCIAL AT = at sign
/// @see on_immediate_0x5b handler for Unicode (U+005B) [ LEFT SQUARE BRACKET
/// @see on_immediate_0x5c handler for Unicode (U+005C) \ REVERSE SOLIDUS
/// @see on_immediate_0x5d handler for Unicode (U+005D) ] RIGHT SQUARE BRACKET
/// @see on_immediate_0x5e handler for Unicode (U+005E) ^ CIRCUMFLEX ACCENT
/// @see on_immediate_0x5f handler for Unicode (U+005F) _ LOW LINE =SPACING UNDERSCORE
/// @see on_immediate_0x60 handler for Unicode (U+0060) ` GRAVE ACCENT (also called backtick)
/// @see on_immediate_0x7b handler for Unicode (U+007B) { LEFT CURLY BRACKET
/// @see on_immediate_0x7c handler for Unicode (U+007C) | VERTICAL LINE
/// @see on_immediate_0x7d handler for Unicode (U+007D) } RIGHT CURLY BRACKET
/// @see on_immediate_0x7e handler for Unicode (U+007E) ~ TILDE

protected:
    static const unsigned int COMMAND_BUFFER_LENGTH = 96; // buffer includes null termination
    unsigned int indexOfNextEmptyCell;
    char buf[COMMAND_BUFFER_LENGTH];
    Stream& associatedSerialPort;
    const char *name;

public:
    CmdLine(Stream& AssociatedSerialPort, const char *Name);

    void clear();

    //void idleAppendIfReadable(); // append ch to buf, unless BS or EOL or other editing character

    bool quiet; // suppress append(ch) from writing to associatedSerialPort
    void append(char ch); // append ch to buf, unless BS or EOL or other editing character

    // onEOLcommandParser
    Callback<void(CmdLine&)> onEOLcommandParser; //!< optional immediate handler for End Of Line CR or LF, parse the command buffer

    // diagnostic_led_EOF
    Callback<void(void)> diagnostic_led_EOF; //!< optional @ref diagnostic_led_EOF

    // Immediate handlers on_immediate_0x21 .. on_immediate_0x7e
    Callback<void(void)> on_immediate_0x21; //!< optional @ref Immediate handler for Unicode (U+0021) ! EXCLAMATION MARK
    Callback<void(void)> on_immediate_0x22; //!< optional @ref Immediate handler for Unicode (U+0022) " QUOTATION MARK
    Callback<void(void)> on_immediate_0x23; //!< optional @ref Immediate handler for Unicode (U+0023) # NUMBER SIGN = pound sign, hash, crosshatch
    Callback<void(void)> on_immediate_0x24; //!< optional @ref Immediate handler for Unicode (U+0024) $ DOLLAR SIGN
    Callback<void(void)> on_immediate_0x26; //!< optional @ref Immediate handler for Unicode (U+0026) & AMPERSAND
    Callback<void(void)> on_immediate_0x27; //!< optional @ref Immediate handler for Unicode (U+0027) ' APOSTROPHE
    Callback<void(void)> on_immediate_0x28; //!< optional @ref Immediate handler for Unicode (U+0028) ( LEFT PARENTHESIS
    Callback<void(void)> on_immediate_0x29; //!< optional @ref Immediate handler for Unicode (U+0029) ) RIGHT PARENTHESIS
    Callback<void(void)> on_immediate_0x2c; //!< optional @ref Immediate handler for Unicode (U+002C) , COMMA
    Callback<void(void)> on_immediate_0x3a; //!< optional @ref Immediate handler for Unicode (U+003A) : COLON
    Callback<void(void)> on_immediate_0x3b; //!< optional @ref Immediate handler for Unicode (U+003B) ; SEMICOLON
    Callback<void(void)> on_immediate_0x3c; //!< optional @ref Immediate handler for Unicode (U+003C) < LESS-THAN SIGN
    Callback<void(void)> on_immediate_0x3e; //!< optional @ref Immediate handler for Unicode (U+003E) > GREATER-THAN SIGN
    Callback<void(void)> on_immediate_0x3f; //!< optional @ref Immediate handler for Unicode (U+003F) ? QUESTION MARK
    Callback<void(void)> on_immediate_0x40; //!< optional @ref Immediate handler for Unicode (U+0040) @ COMMERCIAL AT = at sign
    Callback<void(void)> on_immediate_0x5b; //!< optional @ref Immediate handler for Unicode (U+005B) [ LEFT SQUARE BRACKET
    Callback<void(void)> on_immediate_0x5c; //!< optional @ref Immediate handler for Unicode (U+005C) \ REVERSE SOLIDUS
    Callback<void(void)> on_immediate_0x5d; //!< optional @ref Immediate handler for Unicode (U+005D) ] RIGHT SQUARE BRACKET
    Callback<void(void)> on_immediate_0x5e; //!< optional @ref Immediate handler for Unicode (U+005E) ^ CIRCUMFLEX ACCENT
    Callback<void(void)> on_immediate_0x5f; //!< optional @ref Immediate handler for Unicode (U+005F) _ LOW LINE =SPACING UNDERSCORE
    Callback<void(void)> on_immediate_0x60; //!< optional @ref Immediate handler for Unicode (U+0060) ` GRAVE ACCENT (also called backtick)
    Callback<void(void)> on_immediate_0x7b; //!< optional @ref Immediate handler for Unicode (U+007B) { LEFT CURLY BRACKET
    Callback<void(void)> on_immediate_0x7c; //!< optional @ref Immediate handler for Unicode (U+007C) | VERTICAL LINE
    Callback<void(void)> on_immediate_0x7d; //!< optional @ref Immediate handler for Unicode (U+007D) } RIGHT CURLY BRACKET
    Callback<void(void)> on_immediate_0x7e; //!< optional @ref Immediate handler for Unicode (U+007E) ~ TILDE

    /** serial returns reference to the associated serial port */
    Stream& serial(void) const {
        return associatedSerialPort;
    };

    /** CmdLine[index] returns the character at specified index; [0] is the first character */
    char operator[] (unsigned int index) const {
        if (index >= indexOfNextEmptyCell) return '\0';
        return buf[index];
    };

    /** str returns the c-string buffer */
    const char *str() const {
        return buf;
    };

    /** nameStr returns the names of the associated serial port */
    const char *nameStr() const {
        return name;
    };

    // bool parse_and_remove_key(const char *key, int& matched_index);
    bool parse_and_remove_key(const char *key, char *valueBuf, size_t valueBufLen);
    // WIP #347 - parse_and_remove_key_and_arrayIndex(const char *key, size_t *pindex, char *valueBuf, int valueBufLen) based on parse_and_remove_key
    bool parse_and_remove_key_and_arrayIndex(const char *key, size_t *pindex, char *valueBuf, size_t valueBufLen);
    char chSeparator;
    bool match_is_case_sensitive;

    bool parse_frequency_Hz(const char *key, uint32_t& frequency_Hz);
    bool parse_interval_usec(const char *key, us_timestamp_t& interval_usec);
    bool parse_flag(const char *key, uint8_t& nFlagVar, const uint8_t nFlagBitMask);
    bool parse_byte_hex(const char *key, uint8_t& nByteVar);
    bool parse_byte_dec(const char *key, uint8_t& nByteVar);
    bool parse_uint8_dec(const char *key, uint8_t& nByteVar) { return parse_byte_dec(key, nByteVar); };
    // WIP #347 - parse_uint8_dec_arrayUpdate(const char *key, /*array=*/&AINcode[0], /*arrayIndexLimit=*/10) based on parse_uint8_dec(key, pValue)
    bool parse_uint8_dec_arrayUpdate(const char *key, uint8_t* nBytearray, size_t arrayIndexLimit);
    bool parse_uint16_hex(const char *key, uint16_t& uint16Var);
    bool parse_uint16_dec(const char *key, uint16_t& uint16Var);
    // WIP #347 - parse_uint16_dec_arrayUpdate(/*name=*/"AINcode", /*array=*/&AINcode[0], /*arrayIndexLimit=*/10) based on parse_uint16_dec(key, pValue)
    bool parse_uint16_dec_arrayUpdate(const char *key, uint16_t* uint16Var, size_t arrayIndexLimit);
    bool parse_int_dec(const char *key, int& intVar);
    bool parse_int16_hex(const char *key, int16_t& int16Var);
    bool parse_int16_dec(const char *key, int16_t& int16Var);
    bool parse_uint32_hex(const char *key, uint32_t& uint32Var);
    bool parse_uint32_dec(const char *key, uint32_t& uint32Var);
    bool parse_int32_hex(const char *key, int32_t& int32Var);
    bool parse_int32_dec(const char *key, int32_t& int32Var);
    uint8_t parse_double_or_int16(const char *key, double& doubleVar, int16_t& int16Var);
    uint8_t parse_double_or_uint16(const char *key, double& doubleVar, uint16_t& uint16Var);
    uint8_t parse_double_or_int32(const char *key, double& doubleVar, int32_t& int32Var);
    uint8_t parse_double_or_uint32(const char *key, double& doubleVar, uint32_t& uint32Var);
    bool parse_double(const char *key, double& doubleVar);
    bool parse_float(const char *key, float& floatVar);
    bool parse_byteCount_byteList_hex(size_t& byteCount, char *mosiDataBuf, size_t mosiDataBufSize);
    bool parse_byteCount_byteList_dec(size_t& byteCount, char *mosiDataBuf, size_t mosiDataBufSize);
};

#endif // __CmdLine_H__

// End of file
