/*****************************************************************************
//   $Workfile: $
//    Function: graphics drive - EZLCD04 implementation
//      Author: Bill Basser
//   $JustDate: $
//   $Revision: $
//
//    This document contains proprietary data and information of
//  Cyber Integration LLC.  It is the exclusive property of
//  Cyber Integration, LLC and will not be disclosed in any form to
//  any party without prior written permission of Cyber Integration, LLC.
//  This document may not be reproduced or further used without the
//  prior written permission of Cyber Integration, LLC.
//
//  Copyright (C) 2011 Cyber Integration, LLC. All Rights Reserved
//
//  $History: $
 *
 *****************************************************************************/

// local includes
#include "gdiezl4.h"

// define the MSB/LSB extraction utilities
#define    MSB(w)    (( w >> 8 ) & 0xFF )
#define    LSB(w)    ( w & 0xFF )

/******************************************************************************
//    Function Name: GdiEzL4( tTx, tRx )
//      Description: construction
//            Entry: tTx = transmit pin
//                   tRx = receive pin    
//             Exit: none
// Globals modified: none
//  Locals modified: none
//  Locals modified: none
******************************************************************************/
GdiEzL4::GdiEzL4( PinName pinTx, PinName pinRx ) : m_serDisp( pinTx, pinRx )
{
    // create the interface
    m_serDisp.baud( 115200 );
    m_serDisp.attach( this, &GdiEzL4::LocalCallback );
    
    // initialize
    m_serDisp.putc( EZ_BTP );
    m_serDisp.putc( ON );

    // turn the backlight on/clear screen to white
    BacklightCtl( ON );
    ClearScreen( RGB( 255, 255, 255 ));
}

/******************************************************************************
//    Function Name: GdiEzL4( )
//      Description: destruction
//            Entry: none
//             Exit: none
// Globals modified: none
//  Locals modified: none
//  Locals modified: none
******************************************************************************/
GdiEzL4::~GdiEzL4( void )
{
}

/******************************************************************************
//    Function Name: GdiBacklightCtl( bOffOn )
//      Description: turn backlight off/on
//            Entry: bOffOn
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::BacklightCtl( BOOL bOffOn )
{
    // set the correct state of the backlight
    m_serDisp.putc(( bOffOn ) ? EZ_LON : EZ_LOF );
}

/******************************************************************************
//    Function Name: GdiClearScreen( wColor )
//      Description: clear the screen to the color
//            Entry: wColor = background color
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::ClearScreen( U16 wColor )
{
    // clear the screen
    m_serDisp.putc( EZ_FGC );
    m_serDisp.putc( LSB( wColor ));
    m_serDisp.putc( MSB( wColor ));
    m_serDisp.putc( EZ_CLS );
}

/******************************************************************************
//    Function Name: GdiDrawRect( U16 wColor, U16 wSx, U8 nSy, U16 wWidth, U8 nHeight, BOOL bFill )
//      Description: draw a rectangle
//            Entry: wColor = rectangle color
//                   wSx = starting X
//                   nSy = starting Y
//                   wWidth = width of box
//                   nHeight = height of box
//                   bFill = fill color
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::DrawRect( U16 wColor, U16 wSx, U8 nSy, U16 wWidth, U8 nHeight, BOOL bFill )
{
    U16 wEx = wSx + wWidth - 1;
    U8 nEy = nSy + nHeight - 1;

    // draw a rectangle
    m_serDisp.putc( EZ_FGC );
    m_serDisp.putc( LSB( wColor ));
    m_serDisp.putc( MSB( wColor ));
    m_serDisp.putc( EZ_SXY );
    m_serDisp.putc( MSB( wSx ));
    m_serDisp.putc( LSB( wSx ));
    m_serDisp.putc( nSy );
    m_serDisp.putc( ( bFill ) ? EZ_BXF : EZ_BOX );
    m_serDisp.putc( MSB( wEx ));
    m_serDisp.putc( LSB( wEx ));
    m_serDisp.putc( nEy );
}

/******************************************************************************
//    Function Name: GdiDrawLine( wColor, wSx, nSy, wEx, nEy )
//      Description: draw line
//            Entry: wColor = color
//                   wSx = starting X
//                   nSy = starting Y
//                   wEx = ending X
//                   nEy = ending Y
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::DrawLine( U16 wColor, U16 wSx, U8 nSy, U16 wEx, U8 nEy )
{
    // set the color/set starting x/y
    m_serDisp.putc( EZ_FGC );
    m_serDisp.putc( LSB( wColor ));
    m_serDisp.putc( MSB( wColor ));
    m_serDisp.putc( EZ_SXY );
    m_serDisp.putc( MSB( wSx ));
    m_serDisp.putc( LSB( wSx ));
    m_serDisp.putc( nSy );

    // determine type of line
    if ( wSx == wEx )
    {
        // vertical line
        m_serDisp.putc( EZ_VLN );
        m_serDisp.putc( nEy );
    }
    else if ( nSy == nEy )
    {
        // horizontal line
        m_serDisp.putc( EZ_HLN );
        m_serDisp.putc( MSB( wEx ));
        m_serDisp.putc( LSB( wEx ));
    }
    else
    {
        // diagonal line
        m_serDisp.putc( EZ_LIN );
        m_serDisp.putc( MSB( wEx ));
        m_serDisp.putc( LSB( wEx ));
        m_serDisp.putc( nEy );
    }
}


/******************************************************************************
//    Function Name: GdiDrawIcon( wSx, nSy, nIcon )
//      Description: draw an icon
//            Entry: wSx = starting X
//                   nSy = starting Y
//                   nIcon = icon number
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::DrawIcon( U16 wSx, U8 nSy, U8 nIcon )
{
    // draw a icon
    m_serDisp.putc( EZ_SXY );
    m_serDisp.putc( MSB( wSx ));
    m_serDisp.putc( LSB( wSx ));
    m_serDisp.putc( nSy );
    m_serDisp.putc( EZ_ICF );
    m_serDisp.putc( nIcon );
}

/******************************************************************************
//    Function Name: GdiDrawChar( wColor, nFont, wSx, nSy, bBackground, nDir, cChar )
//      Description: draw a character with the designated font, color
//            Entry: wColor = color
//                   nFont = font number
//                   wSx = starting X
//                   nSy = starting Y
//                   bBackground = true if drawn on background
//                   nDir = text direction
//                   cChar = color
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::DrawChar( U16 wColor, U8 nFont, U16 wSx, U8 nSy, BOOL bBackground, GDITXTDIR eDir, C8 cChar )
{
    // set the color/set xy/set font/draw character
    m_serDisp.putc( EZ_FGC );
    m_serDisp.putc( LSB( wColor ));
    m_serDisp.putc( MSB( wColor ));
    m_serDisp.putc( EZ_SXY );
    m_serDisp.putc( MSB( wSx ));
    m_serDisp.putc( LSB( wSx ));
    m_serDisp.putc( nSy );
    m_serDisp.putc( EZ_FNT );
    m_serDisp.putc( nFont );
    m_serDisp.putc( EZ_TXN + ( U8 )eDir );
    m_serDisp.putc(( bBackground ) ? EZ_CHB : EZ_CHR );
    m_serDisp.putc( cChar );
}

/******************************************************************************
//    Function Name: GdiDrawString( wFgcClr, wBgcClr, nFont, wSx, nSy, bBackground, eDir, szMsg )
//      Description: draw a string with the desinated font, color
//            Entry: wFgClr = foreground color
//                   wBgClr = background color
//                   nFont = font number
//                   wSx = starting X
//                   nSy = starting Y
//                   bBackground = true if drawn on background
//                   eDir = text direction
//                   szMsg -> pointer to zero delimited string
//             Exit: none
// Globals modified: none
//  Locals modified: anBuffer, nBufLen
******************************************************************************/
void GdiEzL4::DrawString( U16 wFgClr, U16 wBgClr, U8 nFont, U16 wSx, U8 nSy, BOOL bBackground, GDITXTDIR eDir, PC8 pszMsg )
{
    // if backgound
    if ( bBackground )
    {
        // set the background color
        m_serDisp.putc( EZ_BGC );
        m_serDisp.putc( LSB( wBgClr ));
        m_serDisp.putc( MSB( wBgClr ));
    }

    // set color/set sx/set font/send command
    m_serDisp.putc( EZ_FGC );
    m_serDisp.putc( LSB( wFgClr ));
    m_serDisp.putc( MSB( wFgClr ));
    m_serDisp.putc( EZ_SXY );
    m_serDisp.putc( MSB( wSx ));
    m_serDisp.putc( LSB( wSx ));
    m_serDisp.putc( nSy );
    m_serDisp.putc( EZ_FNT );
    m_serDisp.putc( nFont );
    m_serDisp.putc( EZ_TXN + ( U8 )eDir );
    m_serDisp.putc(( bBackground ) ? EZ_STB : EZ_STR );

    // copy the string till buffer size
    U8    nChar = 0;
    while (( nChar = *( pszMsg++ )) != '\0' )
        m_serDisp.putc( nChar );

    // stuff the delimiter
    m_serDisp.putc( nChar );
}

/*****************************************************************************
//    Function Name: GdiDrawButton( nIndex, ptBtnDef )
//      Description: draws a button
//            Entry: nIndex = button index
//                   ptBtnDef -> button definition structure
//             Exit: none
// Globals modified: none
//  Locals modified: none
******************************************************************************/
void GdiEzL4::DrawButton( U8 nBtnIdx, GDIBTNDEF* ptBtnDef )
{
    // draw a button
    m_serDisp.putc( EZ_BTD );
    m_serDisp.putc( nBtnIdx );
    m_serDisp.putc( ptBtnDef->eInitalState );
    m_serDisp.putc( ptBtnDef->nUpIcon );
    m_serDisp.putc( ptBtnDef->nDnIcon );
    m_serDisp.putc( ptBtnDef->nDsIcon );
    m_serDisp.putc( MSB( ptBtnDef->wUpLfX ));
    m_serDisp.putc( LSB( ptBtnDef->wUpLfX ));
    m_serDisp.putc( ptBtnDef->nUpLfY );
    m_serDisp.putc( ptBtnDef->nTouchWidth );
    m_serDisp.putc( ptBtnDef->nTouchHeight );
}

/*****************************************************************************
//    Function Name: GdiRemoveAllButton( )
//      Description: remove all button
//            Entry: none
//             Exit: none
// Globals modified: none
//  Locals modified: none
******************************************************************************/
void GdiEzL4::RemoveAllButtons( void )
{
    // send the remove all buttons command
    m_serDisp.putc( EZ_BDL );
}

void GdiEzL4::PingDisplay( void )
{
    // send a ping
    m_serDisp.putc( EZ_PNG );
}

void GdiEzL4::LocalCallback( void )
{
    U8    nChar;
    
    // call the callback
    nChar = m_serDisp.getc( );
    m_pvFuncCallBack.call( nChar );
}
