/* mbed R/C Servo Library
 * Copyright (c) 2007-2010 sford, cstyles
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
*/

// ensure only one copy
#ifndef _GDIEZL4_H
#define _GDIEZL4_H

// include files
#include "mbed.h"
#include "types.h"
#include "MODSERIAL.h"
#include "FPointer.h"

// define the RGB macro
#define    RGB( R, G, B )    (((( R >> 3 ) & 0x1F ) << 11 ) | ((( G >> 2 ) & 0x3F ) << 5 ) | (( B >> 3 ) & 0x1F ))

// define the max X/Y in landscaple mode
#define    MAX_X_SIZE    319
#define    MAX_Y_SIZE    233

// define some standard colcors
#define    GDI_CLR_BLK        RGB(   0,   0,   0 )
#define    GDI_CLR_WHT        RGB( 255, 255, 255 )
#define    GDI_CLR_BLU        RGB(   0,   0, 255 )
#define    GDI_CLR_RED        RGB( 255,   0,   0 )
#define    GDI_CLR_GRN        RGB(   0, 255,   0 )

// define the no button icon valu
#define GDI_BTN_NOICON  255

// define the button up/down masks
#define    GDI_BTN_DNMSK    0x40
#define    GDI_BTN_UPMSK    0x80
#define    GDI_BTN_VLMSK    0x3F

// define the pong value
#define    GDI_PNG_VALUE    0x38

// class definition
class GdiEzL4 
{
public:
    // construction/destruction
    /** Graphics Drawing Interface (EZLCD4 ) construction
     *
     * @param    pinTx = transmit pin
     * @param    pinRx = receive pin
     */
    GdiEzL4( PinName pinTx, PinName pinRx );
    ~GdiEzL4(  );

// attributes
public:
    // enumerate the text direction-
    enum     GDITXTDIR {
        GDI_TXT_NORTH = 0,            // text direction north
        GDI_TXT_EAST,                // text direction east
        GDI_TXT_SOUTH,                // text direction south
        GDI_TXT_WEST                // text direction west
    };

    // enumerate the button state
    enum    GDIBTNSTATE {
        GDI_BTN_UP = 1,                // button up
        GDI_BTN_DN,                    // button down
        GDI_BTN_DSB,                // button disabled
        GDI_BTN_NVS                // button non-visible
    };

    // define the button parameters
    struct    GDIBTNDEF {
        GDIBTNSTATE    eInitalState;    // initial state
        U8            nUpIcon;        // up icon index
        U8            nDnIcon;        // down icon index
        U8            nDsIcon;        // disabled icon index
        U16            wUpLfX;            // upper left X
        U8            nUpLfY;            // upper left Y
        U8            nTouchWidth;    // touch width
        U8            nTouchHeight;    // touch height
    };

private:
    MODSERIAL    m_serDisp;
    FPointer    m_pvFuncCallBack;    
    
// local functions    
    void        LocalCallback( void );

    // enumerate the LCD commands
    enum COMMANDS {
        EZ_CLS = 0x21,        // 0x21 - clear screen
        EZ_LON,                // 0x22 - backlight on
        EZ_LOF,                // 0x23 - backlight off
        EZ_PLT = 0x26,        // 0x26 - plot a pixel
        EZ_PIC = 0x2A,        // 0x2A - draw a picture
        EZ_FNT,                // 0x2B - select the font
        EZ_CHR,                // 0x2C - draw a character
        EZ_STR,                // 0x2D - draw a string
        EZ_SPS = 0x35,        // 0x35 - store position
        EZ_RPS,                // 0x36 - restore position
        EZ_CHB = 0x3C,        // 0x3C - draw character on the background
        EZ_STB,                // 0x3D - draw a string on the background
        EZ_VLN = 0x41,        // 0x41 - draw a vertical line
        EZ_ICF = 0x58,        // 0x58 - draw an icon from serial flash
        EZ_STY = 0x5F,        // 0x5F - set y position
        EZ_TXN = 0x60,        // 0x60 - set text direction to north
        EZ_TXE,                // 0x61 - set text direction to east
        EZ_TXS,                // 0x62 - set text direction to south
        EZ_TXW,                // 0x63 - set text direction to west
        EZ_STX = 0x6E,        // 0x6E - set X position
        EZ_BKL = 0x80,        // 0x80 - brightness
        EZ_PNG = 0x83,        // 0x83 - ping the unit
        EZ_FGC,                // 0x84 - set foreground color
        EZ_SXY,                // 0x85 - set X/y position
        EZ_PXY = 0x87,        // 0x87 - plot a pixel at x,y
        EZ_LIN,                // 0x88 - draw a line
        EZ_CIR,                // 0x89 - draw a circle
        EZ_ARC = 0x8F,        // 0x8F - draw an arc
        EZ_PIE,                // 0x90 - draw a pie
        EZ_BGC = 0x94,        // 0x94 - set background color
        EZ_CRF = 0x99,        // 0x99    - draw a filled circle
        EZ_BMP = 0x9E,        // 0x9E - draw a bitmap
        EZ_HLN = 0xA0,        // 0xA0 - draw a horizontal line
        EZ_BOX = 0xA2,        // 0xA2 - draw a box
        EZ_BXF,                // 0xA3 - draw a filled box
        EZ_BTD = 0xB0,        // 0xB0 - define a touch button
        EZ_BTS,                // 0xB1 - changes a button state
        EZ_BTP,                // 0xB2 - sets the button protocol
        EZ_BUP,                // 0xB3 - all buttons up
        EZ_BDL,                // 0xB4 - delete all buttons
        EZ_BOF = 0xD0,        // 0xD0 - buzzer off
        EZ_BON,                // 0xD1 - buzzer on
        EZ_BEE                 // 0xD2 - beep the buzzer for some time
    };

public:
    // implementation
    /** backlight ontrol
     *
     * @param    bOffOn = state of the backlight
     */
    void    BacklightCtl( BOOL bOffOn );

    /** clear the screen
     *
     * @param    wColor = background color
     */
    void    ClearScreen( U16 wColor );

    /** Draw a rectangle
     *
     * @param    wColor = color of the rectangle
     * @param    wSx    = starting X location ( top left )
     * @param    nSy = starting Y location ( top left )
     * @param    wWidth = width of the rectangle
     * @param    nHeight = height of the rectangle
      * @param    bFill = TRUE if rectangle is filled
     */
    void    DrawRect( U16 wColor, U16 wSx, U8 nSy, U16 wWidth, U8 nHeight, BOOL fFill );

    /** Draw a line
     *
     * @param    wColor = color of the line
     * @param    wSx    = starting X location ( top left )
     * @param    nSy = starting Y location ( top left )
     * @param    wEx = ending X location ( bottom right )
     * @param    nEy = ending Y location ( bottom right )
     */
    void    DrawLine( U16 wColor, U16 wSx, U8 nSy, U16 wEx, U8 nEy );

    /** Draw an Icon 
     *
     * @param    wSx    = starting X location ( top left )
     * @param    nSy = starting Y location ( top left )
     * @param    nIcon = icon index
     */
    void    DrawIcon( U16 wSx, U8 nSy, U8 nIcon );

    /** Draw a character
     *
     * @param    wColor = color of the character
     * @param    nFont = font index
     * @param    wSx    = starting X location ( top left )
     * @param    nSy = starting Y location ( top left )
     * @param    bBackground = display as inverted background
     * @param    eDir = text direction
      * @param    cChar = character to draw
     */
    void    DrawChar( U16 wColor, U8 nFont, U16 wSx, U8 nSy, BOOL bBackground, GDITXTDIR eDir, C8 cChar );

    /** draw a string
     *
     * @param    wColor = color of the character
     * @param    nFont = font index
     * @param    wSx    = starting X location ( top left )
     * @param    nSy = starting Y location ( top left )
     * @param    bBackground = display as inverted background
     * @param    eDir = text direction
      * @param    pszMsg = pointer to the message
     */
    void    DrawString( U16 wFgClr, U16 wBgClr, U8 nFont, U16 wSx, U8 nSy, BOOL bBackground, GDITXTDIR eDir, PC8 pszMsg );

    /** backlight ontrol
     *
     * @param    nBtnIdx = button index
     * @param    ptBugDef -> pointer to the button definition structure
     */
    void    DrawButton( U8 nBtnIdx, GDIBTNDEF* ptButDef );

    /** backlight ontrol
     *
     * @param    bOffOn = state of the backlight
     */
    void    RemoveAllButtons( void );

    /** backlight ontrol
     *
     * @param    bOffOn = state of the backlight
     */
    void    PingDisplay( void );

    /** backlight ontrol
     *
     * @param    bOffOn = state of the backlight
     */
    void     attach( U32 ( *function )( U32 ) = 0) { m_pvFuncCallBack.attach( function ); }

    /** backlight ontrol
     *
     * @param    bOffOn = state of the backlight
     */
    template<class T> 
    void     attach( T* item, U32 ( T::*method )( U32 )) { m_pvFuncCallBack.attach( item, method ); }    
};


#endif

