#include "mbed.h"
#include "PinMap.h"
#include "main.h"
#include "sx1276-mbed-hal.h"


#ifdef FEATURE_LORA

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* Set this flag to '1' to use the LoRa modulation or to '0' to use FSK modulation */
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA
#define RF_FREQUENCY            RF_FREQUENCY_868_1  // Hz
#define TX_OUTPUT_POWER         14                  // 14 dBm

#if USE_MODEM_LORA == 1

#define LORA_BANDWIDTH          125000  // LoRa default, details in SX1276::BandwidthMap
#define LORA_SPREADING_FACTOR   LORA_SF7
#define LORA_CODINGRATE         LORA_ERROR_CODING_RATE_4_5

#define LORA_PREAMBLE_LENGTH    8       // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT     5       // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON  false
#define LORA_FHSS_ENABLED       false
#define LORA_NB_SYMB_HOP        4
#define LORA_IQ_INVERSION_ON    false
#define LORA_CRC_ENABLED        true

#elif USE_MODEM_FSK == 1

#define FSK_FDEV                25000     // Hz
#define FSK_DATARATE            19200     // bps
#define FSK_BANDWIDTH           50000     // Hz
#define FSK_AFC_BANDWIDTH       83333     // Hz
#define FSK_PREAMBLE_LENGTH     5         // Same for Tx and Rx
#define FSK_FIX_LENGTH_PAYLOAD_ON   false
#define FSK_CRC_ENABLED         true

#else
#error "Please define a modem in the compiler options."
#endif


#define BUFFER_SIZE         2048

/*
 *  Global variables declarations
 */
typedef enum {
    LOWPOWER = 0,
    IDLE,

    RX,
    RX_TIMEOUT,
    RX_ERROR,

    TX,
    TX_TIMEOUT,

    CAD,
    CAD_DONE
} AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1276Generic *Radio;

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

DigitalOut *led3;

BufferedSerial *serial;
int Transmitter()
{
#if( defined ( TARGET_KL25Z ) || defined ( TARGET_LPC11U6X ) )
    DigitalOut *led = new DigitalOut(LED2);
#elif defined(TARGET_NUCLEO_L073RZ) || defined(TARGET_DISCO_L072CZ_LRWAN1)
    DigitalOut *led = new DigitalOut(LED4);   // RX red
    led3 = new DigitalOut(LED3);  // TX blue
#else
    DigitalOut *led = new DigitalOut(LED1);
    led3 = led;
#endif

////////////////////////////////////////////////////////////////
    *led3 = 1;
    
    serial = new BufferedSerial(USBTX,USBRX);
    serial->baud(115200*2);
    serial->format(8);

////////////////////////////////////////////////////////////////
    Radio = new SX1276Generic(NULL, MURATA_SX1276,
                              LORA_SPI_MOSI, LORA_SPI_MISO, LORA_SPI_SCLK, LORA_CS, LORA_RESET,
                              LORA_DIO0, LORA_DIO1, LORA_DIO2, LORA_DIO3, LORA_DIO4, LORA_DIO5,
                              LORA_ANT_RX, LORA_ANT_TX, LORA_ANT_BOOST, LORA_TCXO);

    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.TxTimeout = OnTxTimeout;
    RadioEvents.RxTimeout = OnRxTimeout;
    if (Radio->Init( &RadioEvents ) == false) {
        while(Radio->Init( &RadioEvents ) == false) {
            serial->printf("Radio could not be detected!");
            wait( 1 );
        }
    }

 
 
    
    Radio->SetChannel(RF_FREQUENCY );


    Radio->SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                        LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                        LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                        LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                        LORA_IQ_INVERSION_ON, 2000 );

     while( 1 ) {
#ifdef TARGET_STM32L4
        WatchDogUpdate();
#endif
		
        switch( State ) {
            case TX:
                State = LOWPOWER;
                break;
            case RX_TIMEOUT:
                State = LOWPOWER;
                break;
            case RX_ERROR:
                State = LOWPOWER;
                break;
            case TX_TIMEOUT:
                State = LOWPOWER;
                break;
            case LOWPOWER:
                // going to send" );
                int sendSize = 0;
                serial->printf("reached lowpower\n");
                int i = 0;
                while(i < BufferSize) {
                	char temp;
                	while(!serial->readable()) ;
                	temp = serial->getc();
                	if(temp == '\n') break;
                	Buffer[i++] = temp;
                	sendSize++;
                }

                Buffer[i++] = '\n';
                Buffer[i++] = 0;
                
                sendSize++;
                sendSize++;
                
                serial->printf("Sending: %s", Buffer);
              	Radio->Send( Buffer, sendSize);
              	wait(0.5);
              	*led3 = !*led3 ;
                break;
            default:
                State = LOWPOWER;
                break;
        }
    }
}

void OnTxDone(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    State = TX;
    if (DEBUG_MESSAGE)
        serial->printf("> OnTxDone [7]");
}

void OnRxDone(void *radio, void *userThisPtr, void *userData, uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio->Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    State = LOWPOWER;
    //if (DEBUG_MESSAGE)
//        dprintf("> OnRxDone: RssiValue=%d dBm, SnrValue=%d", rssi, snr);
    //dump("Data:", payload, size);
}

void OnTxTimeout(void *radio, void *userThisPtr, void *userData)
{
    *led3 = 0;
    Radio->Sleep( );
    State = LOWPOWER;
    //if(DEBUG_MESSAGE)
//        serial->printf("> OnTxTimeout  [8]  ");
}

void OnRxTimeout(void *radio, void *userThisPtr, void *userData)
{
    *led3 = 0;
    Radio->Sleep( );
    State = LOWPOWER;
    if (DEBUG_MESSAGE)
        dprintf("> OnRxTimeout  [9]  ");
}

void OnRxError(void *radio, void *userThisPtr, void *userData)
{
    Radio->Sleep( );
    State = LOWPOWER;
    if (DEBUG_MESSAGE)
        dprintf("> OnRxError   [100]     ");
}

#endif
