#ifndef M3PI_H
#define M3PI_H

#include "mbed.h"

/** m3pi Class
@brief Library to control m3pi robot from Polulu running the serial slave code
@brief Revision 1.0
@author Craig A. Evans <C.A.Evans@leeds.ac.uk>
@date   April 2017

*/
class m3pi
{

public:

    /** m3pi constructor
    */
    m3pi();
    /** m3pi destructor
    */
    ~m3pi();


    /** Initialisation function
    * @details Should be called at the start of the program to interrupt demo
    * @details code and start 3pi in serial slave mode
    */
    void init();
    
    /** Causes the robot to read the IR sensors
    * @details is an alterative to get_*_values methods if you don't need the values themselves
    */
    void scan();

    /** Get signature of slave firmware
    * @param signature - array of size 7 to store signature
    */
    void get_signature(char *signature);

    /** Read raw sensor values from IR sensors (0 - 2000)
    * @param values - array of size 5 to store values
    */
    void get_raw_values(unsigned int *values);

    /** Read calibrated sensor values from IR sensors (0 - 1000)
    * @param values - array of size 5 to store values
    */
    void get_calibrated_values(unsigned int *values);

    /** Read user potentiometer values (0.0 - 1.0)
    * @returns float in range  0.0 to 1.0
    */
    float get_trimpot_value();

    /** Read battery voltage
    * @returns battery voltage in volts
    */
    float get_battery_voltage();

    /** Play music
    * @param notes - const array containing Pololu notes
    * @param length - number of notes in array
    */
    void play_music(const char notes[],int length);

    /** Calibrate
    * @details manual calibration of sensors. Should be called repeatedly.
    */
    void calibrate();

    /** Reset previous calibration values
    */
    void reset_calibration();

    /** Returns estimate of normalised line position
    * @returns float in the range -1.0 to 1.0
    * @details -1.0 for line under sensor 0, -0.5 for line under sensor 1
    * @details 0.0 for line under sensor 2, 0.5 for line under sensor 3 etc.
    */
    float get_line_position();
    
    /** Returns an estimate of the normalised line position
    * @returns float in the range -1.0 to 1.0
    * @details uses the values of sensor stored in the class - must call scan() first!
    * @details -1.0 for line under sensor 0, -0.5 for line under sensor 1
    * @details 0.0 for line under sensor 2, 0.5 for line under sensor 3 etc.
    */
    float read_line(); 

    /** Clear LCD
    */
    void lcd_clear();

    /** Print text on LCD
    * @param text[] - string of max size 8
    * @param - length of string
    */
    void lcd_print(char text[],int length);

    /** Move cursor on LCD
    * @param x - x position
    * @param y - line number
    */
    void lcd_goto_xy(int x, int y);

    /** Call automatic calibration function
    */
    void auto_calibrate();

    /** Move left motor
    * @param speed - value in range -1.0 (full-speed backward) to 1.0 (full-speed forward)
    */
    void left_motor(float speed);

    /** Move right motor
    * @param speed - value in range -1.0 (full-speed backward) to 1.0 (full-speed forward)
    */
    void right_motor(float speed);

    /** Move both motors
    * @param left speed - value in range -1.0 (full-speed backward) to 1.0 (full-speed forward)
    * @param right speed - value in range -1.0 (full-speed backward) to 1.0 (full-speed forward)
    */
    void motors(float left_speed,float right_speed);

    /** Stop both motors
    */
    void stop();

    /** Move buggy forward
    * @param speed - value in range 0.0 to 1.0 (full-speed forward)
    */
    void forward(float speed);

    /** Reverse buggy
    * @param speed - value in range 0.0 to 1.0 (full-speed forward)
    */
    void reverse(float speed);

    /** Spin right
    * @param speed - value in range 0.0 to 1.0
    */
    void spin_right(float speed);

    /** Spin left
    * @param speed - value in range 0.0 to 1.0
    */
    void spin_left(float speed);

    /** Display battery voltage on LCD
    * @param x - x position on LCD to display
    * @param y - line on LCD to display
    */
    void display_battery_voltage(int x,int y);

    /** Display slave firmware signature on LCD
    * @param x - x position on LCD to display
    * @param y - line on LCD to display
    */
    void display_signature(int x,int y);
    
    /** Display line position and bar graph of sensor readings on LCD
    * @details must call scan() and read_line() first
    */
    void display_data();
    
    /** Display sensor values on LCD
    * @param values - array of calibrated sensor values
    * @param y - line on LCD to display
    */
    void display_sensor_values(unsigned int values[],int y);
    
    /** Gets value of sensor array expressed as decimal number
    * @details Each sensor is a bit (PC4 is bit 0, PC0 is bit 5)
    * @returns value - 5-bit value of sensor reading
    */
    unsigned int get_sensor_array_value(unsigned int values[]);
    
    /** Calculates an estimate of normalised line position from the sensor readings
    * @returns float in the range -1.0 to 1.0
    * @details -1.0 for line under sensor 0, -0.5 for line under sensor 1
    * @details 0.0 for line under sensor 2, 0.5 for line under sensor 3 etc.
    */
    float calc_line_position(unsigned int values[]);


private:
    BufferedSerial* _serial;
    DigitalOut* _reset;
    
    float _last_line_position;
    
    char _bar_graph[7];
    unsigned int _values[5];

    void reset();

};



#endif