// 9430 Quadrature Decoder using mbed edge-triggered interrupts
// 
// Read quadrature encoder and drive LEDs
// Version 0.1
// Created by Wayne Chin
// October 28, 2010

#include "mbed.h"

#define ON 0 // LED drives are inverted
#define OFF 1 // LED drives are inverted
#define MAXAMPS200 200
#define MINAMPSET 25
#define DEBOUNCETIME 6000 // Debounce delay in us

// mbed LEDs
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

// Quadrature encoder inputs
InterruptIn quada(p21);
InterruptIn quadb(p22);
InterruptIn quadsw(p30);

// mbed API timers used for edge sensing and debouncing
Timer debouncetimerrisea;
Timer debouncetimerfalla;
Timer debouncetimerriseb;
Timer debouncetimerfallb;
Timeout debouncerisea;
Timeout debouncefalla;
Timeout debounceriseb;
Timeout debouncefallb;

Ticker updateserial;
Serial pc(USBTX, USBRX); // tx, rx

int counter; // Global counter variable

void serialout()
{
    pc.printf("%d\n\r", counter);
}

void display()
{
    if (counter > MAXAMPS200)
        counter = MAXAMPS200; // Clip counter at high end
    else if (counter < MINAMPSET)
        counter = MINAMPSET; // Clip at low end
    switch (counter % 4) // Get 2 LSBs and display sequence on mbed LEDs
    {
        case 0:
            led1 = 1; led2 = 0; led3 = 0; led4 = 0;
            break;
        case 1:
            led1 = 0; led2 = 1; led3 = 0; led4 = 0;
            break;
        case 2:
            led1 = 0; led2 = 0; led3 = 1; led4 = 0;
            break;
        default:
            led1 = 0; led2 = 0; led3 = 0; led4 = 1;
            break;
    } // switch
}

// Read quadrature encoder, update counter variable, and update LEDs
// Quadrature clockwise states: 00 -> 10 -> 11 -> 01
// Quadrature counter-clockwise states: 00 -> 01 -> 11 -> 10
void checkrisea()
{
    debouncetimerrisea.stop();
    debouncetimerrisea.reset();
    debouncerisea.detach();
    if (quada == 1 && quadb == 0)
        counter++;
    else if (quada == 1 && quadb == 1)
        counter--;
    display();
}

void checkfalla()
{
    debouncetimerfalla.stop();
    debouncetimerfalla.reset();
    debouncefalla.detach();
    if (quada == 0 && quadb == 1)
        counter++;
    else if (quada == 0 && quadb == 0)
        counter--;
    display();
}

void checkriseb()
{
    debouncetimerriseb.stop();
    debouncetimerriseb.reset();
    if (quadb == 1 && quada == 1)
        counter++;
    else if (quadb == 1 && quada == 0)
        counter--;
    display();
}

void checkfallb()
{
    debouncetimerfallb.stop();
    debouncetimerfallb.reset();
    if (quadb == 0 && quada == 0)
        counter++;
    else if (quadb == 0 && quada == 1)
        counter--;
    display();
}

// Start of edge-triggered interrupts
void risea()
{
    if (debouncetimerrisea.read_ms() == 0)
    {
        debouncetimerrisea.start();
        debouncerisea.attach_us(&checkrisea, DEBOUNCETIME); // Check on switch later
    }
} // risea()

void falla()
{
    if (debouncetimerfalla.read_ms() == 0)
    {
        debouncetimerfalla.start();
        debouncefalla.attach_us(&checkfalla, DEBOUNCETIME); // Check on switch later
    }
} // falla()

void riseb()
{
    if (debouncetimerriseb.read_ms() == 0)
    {
        debouncetimerriseb.start();
        debounceriseb.attach_us(&checkriseb, DEBOUNCETIME); // Check on switch later
    }
} // riseb()

void fallb()
{
    if (debouncetimerfallb.read_ms() == 0)
    {
        debouncetimerfallb.start();
        debouncefallb.attach_us(&checkfallb, DEBOUNCETIME); // Check on switch later
    }
} // fallb()

int main() {
    quada.mode(PullUp); // Enable pullup
    quadb.mode(PullUp); // Enable pullup
    quadsw.mode(PullUp); // Enable pullup
    quada.rise(&risea); // Call function risea() on rising edge
    quada.fall(&falla); // Call function falla() on falling edge
    quadb.rise(&riseb); // Call function riseb() on rising edge
    quadb.fall(&fallb); // Call function fallb() on falling edge
    counter = 0; // Reset counter

    pc.baud(19200);
    pc.printf("\n\rConnected to mBed...\r\n");
    
    debouncetimerrisea.stop();
    debouncetimerrisea.reset();
    debouncetimerfalla.stop();
    debouncetimerfalla.reset();
    debouncetimerriseb.stop();
    debouncetimerriseb.reset();
    debouncetimerfallb.stop();
    debouncetimerfallb.reset();

    // Set up interrupt call for serial port update
    //updateserial.attach_us(&serialout, 500000); // setup updateserial to call serialout every 500 ms

    while (1)
    {
    }// while (1)
} // main()
