/*
 * Value.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file Value.h
 * Represents a generic data value.
 */

#ifndef VALUE_H
#define VALUE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>
#include <stdint.h>
#include "AbstractDataSink.h"

/** Return value indicating a null value. */
#define VALUE_NULL 0
/** Return value indicating an integer value. */
#define VALUE_INTEGER 1
/** Return value indicating a floating-point number. */
#define VALUE_FLOAT 2
/** Return value indicating a character string. */
#define VALUE_CHARACTER 3

/**
 * A generic value offering means to obtain the value type and contents
 * as well as writing the value to a sink.
 */
class Value
{
public:
    virtual ~Value() { };

    /**
     * Returns the value type which can be VALUE_NULL, VALUE_INTEGER,
     * VALUE_FLOAT or VALUE_CHARACTER.
     * @return the value type
     */
    virtual uint8_t valueType() const = 0;

    /**
     * Returns the integer value if an integer type, otherwise zero.
     * @return integer value
     */
    virtual long integerValue() const = 0;

    /**
     * Returns the float value if a float type, otherwise zero.
     * @return float value
     */
    virtual double floatValue() const = 0;

    /**
     * Returns the character value if a character type, otherwise zero.
     * @return character value
     */
    virtual const char * characterValue() const = 0;

    /**
     * Writes the value to a sink.
     * If necessary, quotes and escape characters may be added.
     * @param sink the sink to write the value to
     * @return the number of bytes written
     */
    virtual size_t write(AbstractDataSink&) const = 0;

    /**
     * Returns the number of bytes required to write this value to a sink.
     * @return number of bytes required to write this value
     */
    virtual size_t length() const = 0;

    /**
     * Copies this value to the heap. Referenced contents must also be
     * copied.
     * @return a new value instance allocated on the heap
     */
    virtual Value* copy() const = 0;
};

#endif
