/*
 * ParsedValue.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ParsedValue.h"
#include "NullValue.h"
#include "IntegerValue.h"
#include "FloatValue.h"
#include "CharValue.h"
#include <stdio.h>
#include <string.h>
#include <ctype.h>

ParsedValue::ParsedValue(const char *str, bool copy) : _value(str, copy)
{
    if (_value.valueType() != VALUE_NULL) {
        _type = VALUE_CHARACTER;
        extractValue();
    } else {
        _type = VALUE_NULL;
    }
}

uint8_t ParsedValue::valueType() const
{
    return _type;
}

long ParsedValue::integerValue() const
{
    if (_type != VALUE_INTEGER)
        return 0;
    return _integer;
}

double ParsedValue::floatValue() const
{
    if (_type != VALUE_FLOAT)
        return 0.0;
    return _float;
}

const char * ParsedValue::characterValue() const
{
    if (_type != VALUE_CHARACTER)
        return NULL;
    return _value.characterValue();
}

size_t ParsedValue::write(AbstractDataSink& sink) const
{
    return _value.write(sink);
}

size_t ParsedValue::length() const
{
    return _value.length();
}

Value* ParsedValue::copy() const
{
    if (_type == VALUE_NULL)
        return new NullValue();
    if (_type == VALUE_INTEGER)
        return new IntegerValue(_integer);
    if (_type == VALUE_FLOAT)
        return new FloatValue(_float, _digits, _zflag);
    return new CharValue(_value.characterValue(), true);
}

void ParsedValue::extractValue()
{
    int len;
    const char *str = _value.characterValue();

    sscanf(str, "%li%n", &_integer, &len);
    if ((size_t)len == strlen(str)) {
        _type = VALUE_INTEGER;
        return;
    }

    sscanf(str, "%lf%n", &_float, &len);
    if ((size_t)len == strlen(str)) {
        bool floating = false;
        _digits = 0;
        _zflag = false;
        for (char c, *ptr = (char*)str; (c = *ptr) > 0; ptr++) {
            if (floating) {
                if (c == '.')
                    return;
                _digits++;
            } else {
                if (c == '.')
                    floating = true;
                else if (isdigit(c))
                    _zflag = true;
            }
        }
        if (!((floating) && (!_digits))) {
            _type = VALUE_FLOAT;
            return;
        }
    }
}
