/*
 * Aggregator.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef AGGREGATOR_H
#define AGGREGATOR_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>
#include "DataGenerator.h"

#ifndef AGGREGATOR_INITIAL_CAPACITY
#define AGGREGATOR_INITIAL_CAPACITY 50
#endif
#ifndef AGGREGATOR_MEMORY_INCREMENT
#define AGGREGATOR_MEMORY_INCREMENT 25
#endif
//#define AGGREGATOR_FIXED_SIZE 100

/**
 * An aggregator of data generators. This class can aggregate instances of
 * itself.
 * If the aggregator is set to copying all added data generators using
 * the copy() method, all objects are being properly deallocated on
 * destruction.
 */
class Aggregator : public DataGenerator
{
public:
    #ifndef AGGREGATOR_FIXED_SIZE
    /**
     * Creates a new Aggregator instance.
     * @param capacity the initial capacity of the instance
     * @param growing specifies the capability of this instance to grow
     * @param copy specifies whether all added data generators shall be
     *             copied using the copy() method
     */
    Aggregator(size_t=AGGREGATOR_INITIAL_CAPACITY, bool=true, bool=false);
    #else
    /**
     * Creates a new Aggregator instance.
     * @param copy specifies whether all added data generators shall be
     *             copied using the copy() method
     */
    Aggregator(bool=false);
    #endif
    ~Aggregator();

    /**
     * Adds a data generator to the aggregator.
     * @param generator the data generator to add
     * @return true if added, false otherwise.
     */
    bool add(const DataGenerator&);

    /**
     * Clears the aggregator. The capacity will shrink to it's initial
     * size.
     */
    void clear();

    /**
     * Returns the number of data generators aggregated by this instance.
     * @return the number of data generators aggregated
     */
    size_t length();

    /**
     * Returns whether the aggregator is full. If growing, this will
     * always return false.
     * @return whether the aggregator is full
     */
    bool full();

    /**
     * Returns the capacity of the aggregator. This will always return zero
     * if the aggregator is growing.
     */
    size_t capacity();

    size_t writeTo(AbstractDataSink&) const;
    size_t writtenLength() const;
    DataGenerator* copy() const;

private:
    #ifdef AGGREGATOR_FIXED_SIZE
    const DataGenerator *_list[AGGREGATOR_FIXED_SIZE];
    #else
    const DataGenerator **_list;
    #endif
    size_t _length;
    bool _alloc;
    #ifndef AGGREGATOR_FIXED_SIZE
    size_t _capacity, _initial;
    bool _growing;
    #endif
};

#endif
