/*
 * AbstractDataSource.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file AbstractDataSource.h
 * An abstraction layer which provides means to read data from a
 * connection.
 * 
 * Example:
 * @code
 * AbstractDataSource source;
 * char c;
 * while (((c = source.read()) > 0) || (source.status() == DS_STATUS_OK)) {
 *     // read action
 * }
 * switch (source.status()) {
 * case DS_STATUS_OK:
 *     // ok
 * case DS_STATUS_CLOSED:
 *     // connection closed
 * case DS_STATUS_TIMEOUT:
 *     // timeout
 * default:
 *     // other error
 * }
 * @encode
 * 
 */

#ifndef ABSTRACTDATASOURCE_H
#define ABSTRACTDATASOURCE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdint.h>

/** Return value indicating that no error has occurred. */
#define DS_STATUS_OK 0
/** Return value indicating that the connection has been closed by this
 * or the foreign host. */
#define DS_STATUS_CLOSED 1
/** Return value indicating a transmission timeout. */
#define DS_STATUS_TIMEOUT 2
/** Return value indicating a state error. */
#define DS_STATUS_ERROR 3

/*
 * The AbstractDataSource class provides a way to read data from a
 * connection.
 */
class AbstractDataSource
{
public:
    virtual ~AbstractDataSource() { };

    /**
     * Reads one character. When successful, the returned value
     * represents the character. Otherwise, status() will return a
     * non-zero value. This function is blocking.
     * @return the caracter read or zero. If the return value is
     * non-zero, status() is zero. Otherwise, check status() for a
     * non-zero value.
     */
    virtual char read() = 0;

    /**
     * Returns the current read state. If a non-zero value is returned,
     * there is an error. Otherwise, everything is ok.
     * DS_STATUS_TIMEOUT means that there has been a read timeout.
     * DS_STATUS_CLOSED means that the connection has been closed
     * by either this or the foreign host.
     * @return the status value
     */
    virtual uint8_t status() = 0;
};

#endif
