/*
 * SmartRest.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file SmartRest.h
 * The main SmartRest facade class. This class is abstract, because the
 * actual implementation has to supply a client.
 */

#ifndef SMARTREST_H
#define SMARTREST_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>
#include <stdint.h>
#include "AbstractClient.h"
#include "ParsedRecord.h"
#include "Parser.h"

#ifndef SMARTREST_MOGID_BUFFER_SIZE
#define SMARTREST_MOGID_BUFFER_SIZE 11
#endif

/** Return value indicating that no error occurred. */
#define SMARTREST_SUCCESS 0
/** Return value indicating that the connection has been closed during
 * data transmission. */
#define SMARTREST_CONNECTION_FAILED 1
/** Return value indicating an internal state error. */
#define SMARTREST_INTERNAL_ERROR 2
/** Return value indicating a transmission timeout. */
#define SMARTREST_TIMEOUT_ERROR 3
/** Return value indicating an end of response indicated by the
 * Content-Length header. */
#define SMARTREST_END_OF_RESPONSE 4
/** Return value indicating that the connection has been closed. */
#define SMARTREST_CONNECTION_CLOSED 5

/**
 * SmartRest Arduino client implementation.
 * This class provides methods to send a request and receive a response
 * from the server.
 */
class SmartRest
{
protected:
    /**
     * Creates a new GenericSmartRest object.
     * @param client the abstract client to use
     * @param identifier the device identifier
     */
    SmartRest(AbstractClient&, const char*);

public:
    virtual ~SmartRest() { };

    /**
     * Sends a smart request.
     * @param generator the generator which will generate the data to be
     *                  sent.
     * @return a non-zero value if and only if an error occured
     */
    int8_t send(DataGenerator&);

    /**
     * Tries to receive a parsed response row.
     * When the function succeeds, but the row pointer is NULL, there are
     * no more rows to be read.
     * @param record an instance to where the parsed row is written
     * @return a non-zero value if and only if an error occured
     */
    int8_t receive(ParsedRecord&);

    /*
     * Initiates the SmartRest bootstrap process.
     * When successful, the template identifier will be replaced by the
     * global managed object ID in future requests.
     * @param generator the generator which will generate the data to be
     *                  sent as a template.
     * @return a non-zero value if and only if an error occured
     */
    int8_t bootstrap(DataGenerator&);

    /*
     * Closes the connection.
     */
    void stop();

private:
    uint8_t beginRequest();
    uint8_t awaitResponse();
    bool setMoGid(Record&);

private:
    AbstractClient& _client;
    AbstractDataSource *_source;
    Parser _parser;
    const char *_identifier;
    char _mogid[SMARTREST_MOGID_BUFFER_SIZE];
};

#endif
