/*
 * Aggregator.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "Aggregator.h"
#include <stdlib.h>

#ifndef AGGREGATOR_FIXED_SIZE
Aggregator::Aggregator(size_t capacity, bool growing, bool copy)
{
    _alloc = copy;
    _growing = growing;
    _initial = _capacity = capacity;
    _list = (const DataGenerator**)malloc(_capacity*sizeof(DataGenerator*));
    _length = 0;
}
#else
Aggregator::Aggregator(bool copy)
{
    _alloc = copy;
    _length = 0;
}
#endif

Aggregator::~Aggregator()
{
    if (_alloc) {
        for (size_t n = 0; n < _length; n++)
            delete _list[n];
    }
    #ifndef AGGREGATOR_FIXED_SIZE
    free(_list);
    #endif
}

bool Aggregator::add(const DataGenerator& generator)
{
    if (this == &generator)
        return false;
    #ifndef AGGREGATOR_FIXED_SIZE
    if (_length == _capacity) {
        if (!_growing)
            return false;

        size_t capacity = _capacity + AGGREGATOR_MEMORY_INCREMENT;
        const DataGenerator **list = (const DataGenerator**)realloc(_list,
            capacity*sizeof(DataGenerator*));
        if (list == NULL)
            return false;
        _list = list;
        _capacity = capacity;
    }
    #else
    if (_length == AGGREGATOR_FIXED_SIZE)
        return false;
    #endif

    if (_alloc) {
        DataGenerator *copy = generator.copy();
        if (copy == NULL)
            return false;
        _list[_length++] = copy;
    } else {
        _list[_length++] = &generator;
    }
    return true;
}

void Aggregator::clear()
{
    if (_alloc) {
        for (size_t n = 0; n < _length; n++)
            delete _list[n];
    }
    _length = 0;

    #ifndef AGGREGATOR_FIXED_SIZE
    if (_growing) {
        const DataGenerator **list = (const DataGenerator**)realloc(_list,
            _initial*sizeof(DataGenerator*));
        if (list == NULL)
            return;
        _list = list;
        _capacity = _initial;
    }
    #endif
}

size_t Aggregator::length()
{
    return _length;
}

bool Aggregator::full()
{
    #ifndef AGGREGATOR_FIXED_SIZE
    return (_growing) ? false : (_length == _capacity);
    #else
    return (_length == AGGREGATOR_FIXED_SIZE);
    #endif
}

size_t Aggregator::capacity()
{
    #ifndef AGGREGATOR_FIXED_SIZE
    return (_growing) ? 0 : _capacity;
    #else
    return AGGREGATOR_FIXED_SIZE;
    #endif
}

size_t Aggregator::writeTo(AbstractDataSink& sink) const
{
    size_t len = 0;
    for (size_t n = 0; n < _length; n++)
        len += _list[n]->writeTo(sink);
    return len;
}

size_t Aggregator::writtenLength() const
{
    size_t len = 0;
    for (size_t n = 0; n < _length; n++)
        len += _list[n]->writtenLength();
    return len;
}

DataGenerator* Aggregator::copy() const
{
    #ifndef AGGREGATOR_FIXED_SIZE
    Aggregator *copy = new Aggregator(_length, _growing, true);
    #else
    Aggregator *copy = new Aggregator(true);
    #endif
    for (size_t n = 0; n < _length; n++)
        copy->add(*_list[n]);
    return copy;
}

