/*
 * ComposedRecord.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "ComposedRecord.h"
#include "NullValue.h"
#include <stdlib.h>
#include <string.h>

ComposedRecord::ComposedRecord(bool copy)
{
    _alloc = copy;
    #ifndef COMPOSED_FIXED_SIZE
    _capacity = COMPOSED_INITIAL_CAPACITY;
    _values = (const Value**)malloc(_capacity*sizeof(Value*));
    #endif
    _count = 0;
}

ComposedRecord::~ComposedRecord()
{
    if (_alloc) {
        for (size_t n = 0; n < _count; n++)
            delete _values[n];
    }
    #ifndef COMPOSED_FIXED_SIZE
    free(_values);
    #endif
}

ComposedRecord& ComposedRecord::add(const Value& value)
{
    #ifndef COMPOSED_FIXED_SIZE
    if (_capacity == _count) {
        size_t capacity = _capacity + COMPOSED_MEMORY_INCREMENT;
        const Value **values = (const Value**)realloc(_values,
            capacity*sizeof(Value*));
        if (values == NULL)
            return *this;
        _values = values;
        _capacity = capacity;
    }
    #else
    if (_count == COMPOSED_FIXED_SIZE)
        return *this;
    #endif

    if (_alloc) {
        Value *copy = value.copy();
        if (copy == NULL)
            return *this;
        _values[_count++] = copy;
    } else {
        _values[_count++] = &value;
    }
    return *this;
}

void ComposedRecord::clear()
{
    _count = 0;
    #ifndef COMPOSED_FIXED_SIZE
    if (_capacity > COMPOSED_INITIAL_CAPACITY) {
        Value** values = (Value**)realloc(_values,
            COMPOSED_INITIAL_CAPACITY*sizeof(Value*));
        if (values == NULL)
            return;
        _capacity = COMPOSED_INITIAL_CAPACITY;
    }
    #endif
}

size_t ComposedRecord::values() const
{
    return _count;
}

const Value& ComposedRecord::value(size_t index) const
{
    if (index >= _count)
        return aNullValue;
    return *_values[index];
}

DataGenerator* ComposedRecord::copy() const
{
    ComposedRecord *copy = new ComposedRecord(true);
    for (size_t n = 0; n < _count; n++)
        copy->add(*_values[n]);
    return copy;
}

