/*
 * CharValue.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "CharValue.h"
#include "NullValue.h"
#include <stdlib.h>
#include <string.h>

inline int8_t __charvalue_need_escape(const char *str);

CharValue::CharValue(const char *str, bool copy)
{
    if ((str == NULL) || (strlen(str) == 0)) {
        _alloc = false;
        _str = NULL;
    } else {
        _alloc = copy;
        if (copy) {
            _str = (const char*)malloc(strlen(str)*sizeof(char));
            strcpy((char*)_str, str);
        } else {
            _str = str;
        }
    }
}

CharValue::~CharValue()
{
    if (_alloc)
        free((void*)_str);
}

uint8_t CharValue::valueType() const
{
    if (_str == NULL)
        return VALUE_NULL;
    return VALUE_CHARACTER;
}

long CharValue::integerValue() const
{
    return 0;
}

double CharValue::floatValue() const
{
    return 0.0;
}

const char * CharValue::characterValue() const
{
    return _str;
}

size_t CharValue::write(AbstractDataSink& sink) const
{
    if (_str == NULL)
        return 0;
    size_t n = 0;
    int8_t esc = __charvalue_need_escape(_str) ? 1 : 0;
    if (esc) n += sink.write('"');
    for (char *q = (char*)_str, c; (c = *q) > 0; q++) {
        if ((esc) && (c == '"'))
            n += sink.write('"');
        n += sink.write(c);
    }
    if (esc) n += sink.write('"');
    return n;
}

size_t CharValue::length() const
{
    if (_str == NULL)
        return 0;
    size_t n = 0;
    int8_t esc = __charvalue_need_escape(_str) ? 1 : 0;
    if (esc)
        n += 2;
    for (char *q = (char*)_str, c; (c = *q) > 0; q++) {
        if ((esc) && (c == '"'))
            n++;
        n++;
    }
    return n;
}

int8_t __charvalue_need_escape(const char *str)
{
    int8_t w = 0;
    for (char c; (c = *str) > 0; str++) {
        if ((c == ' ') || (c == '\t')) {
            if (w == 0)
                return 1;
            w = 2;
        } else {
            w = 1;
            if ((c == '"') || (c == ',') || (c == '\r') || (c == '\n'))
                return 1;
        }
    }
    if (w == 2)
        return 1;
    return 0;
}

Value* CharValue::copy() const
{
    if (_str == NULL)
        return new NullValue();
    return new CharValue(_str, true);
}

