/*
 * AbstractDataSink.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file AbstractDataSink.h
 * An abstraction for a data sink which can be used to send data
 * over a connection.
 * 
 * Following write actions are supported:
 * - write a character
 * - write a buffer of certain length
 * - write a character string
 * - write an unsigned number
 */

#ifndef ABSTRACTDATASINK_H
#define ABSTRACTDATASINK_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stddef.h>

/*
 * The AbstractDataSink class provides a way to write data to a
 * connection.
 */
class AbstractDataSink
{
public:
    virtual ~AbstractDataSink() { };

    /**
     * Writes a single character.
     * @param c the character to write
     * @return the number of characters written, in this case 1 or 0
     */
    virtual size_t write(char c) = 0;

    /**
     * Writes a buffer.
     * @param buf the buffer to write
     * @param length the length of the buffer in bytes
     * @return the number of characters written
     */
    virtual size_t write(void *buf, size_t length) = 0;

    /**
     * Writes a character string.
     * @param str the string to write
     * @return the number of characters written
     */
    virtual size_t write(const char *str) = 0;

    /**
     * Writes a number with base 10.
     * @param number the number to write
     * @return the number of digits written
     */
    virtual size_t write(unsigned long number) = 0;
};

#endif
