//  --------------------------------------------------------------------------------------------------------
//
//  September 2018
//  Author: Valeria Toffoli, Rohan Gurav 
//
// --------------------------------------------------------------------------------------------------------
//
//  ADXL100x.h
//
// --------------------------------------------------------------------------------------------------------
// 
//  This library provides all the functions necessary to interface the ADXL355 with EV-COG-AD3029 or  
//  EV-COG-AD4050 Board. Functions for reads and writes,and scaling are included. 
//  This library may be used for the entire ADXL100x family of devices 
//  with some modification.
//
//  Permission is hereby granted, free of charge, to any person obtaining
//  a copy of this software and associated documentation files (the
//  "Software"), to deal in the Software without restriction, including
//  without limitation the rights to use, copy, modify, merge, publish,
//  distribute, sublicense, and/or sell copies of the Software, and to
//  permit persons to whom the Software is furnished to do so, subject to
//  the following conditions:
//
//  The above copyright notice and this permission notice shall be
//  included in all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// --------------------------------------------------------------------------------------------------------

#include <stdint.h>
#include <math.h>
#include "mbed.h"
#include "ADXL100x.h"



/* Constructor with configurable ADXL1005 under study */
ADXL100x::ADXL100x(PinName ST_pin , PinName STB_pin, PinName OR_pin , PinName Vout_pin, int device, double power):  
    st(ST_pin),  stb(STB_pin), over(OR_pin), vout(Vout_pin) { // pins declarations
        st = 0; stb = 0; 
        offset = power/2;  
        fact = 13e-3;    // V/g 
        sens = 1/fact;   // g/V  
        dyn = 200;       // g
    }
/* Low power mode - standby */
void ADXL100x::standby(){
    stb = 1;
    wait(0.001);
    }
/* Wake up rutine: turn On Time <500us */
void ADXL100x::wakeup(){
    stb = 0;
    wait(0.001);  
    }
float ADXL100x::standard_dev(int bit){
    float sum, sd, mean; 
    float data[128];
    for (int j = 0; j < 128; j++){
        data[j] = accelScale(scanx(), bit);
        sum += data[j];
        }
    mean = sum / 128;
    for (int j = 0; j < 128; j++){
        sd += pow(data[j] - mean, 2);;
        }
    return sqrt(sd/128);
    }
/* Perform calibration over 1g and -1g*/
void ADXL100x::calibrate1g(float value_p1g, float value_n1g){
    float m, b;
    m = 2/(value_p1g-value_n1g);
    b = -(value_p1g+value_n1g)/(value_p1g-value_n1g);
    fact /= m; 
    sens = 1/fact;
    offset -= b;
    }    
/* Perform the self test rutine: output takes approximately 300 μs to assenst*/
bool ADXL100x::selftest(int bit, double power){
    double data0, data1, range;
    st = 0; wait(0.3);
    data0 = accelScale(scanx(), bit);
    st = 1; wait(0.3);
    data1 = accelScale(scanx(), bit);
    st = 0; wait(0.3);
    range = 0.1e-3+0.4e-3*(power-3)/2;
    if ( abs(data1-data0-range) < 0.05e-3 ){return true;}
    else {return false;}
    }
/* Read data as 16bit from the ADUCM converter */
uint16_t ADXL100x::scanx(){
    return vout.read_u16();
    }
 /* Data conversion from 16bit to g. Need to specify the converter bit num */   
float ADXL100x::accelScale(uint16_t data, int bit){
    float result;
    result = float(scanx())*(dyn*fact)/pow(2.0,double(bit));
    return (result -offset)*sens;
    }
    
