#ifndef _LDC1000_H_
#define _LDC1000_H_

#include "FastPWM.h"
/**
* @file LDC1000.h
* @brief this header file will contain all required
* definitions for the functions to interface with Texas
* Instruments' LDC1000.
*
* @author Victor Sluiter
*
* @date 2015-04-01
*/

#include "mbed.h"

#ifndef PI
#define PI 3.14
#endif

typedef enum {  LDC_RESPONSE_192=2,\
                LDC_RESPONSE_384= 3,  \
                LDC_RESPONSE_768,  \
                LDC_RESPONSE_1536, \
                LDC_RESPONSE_3072, \
                LDC_RESPONSE_6144} LDC_RESPONSE;

typedef enum {  LDC_AMPLITUDE_1V=0,\
                LDC_AMPLITUDE_2V,  \
                LDC_AMPLITUDE_4V} LDC_AMPLITUDE;

typedef enum { LDC_MODE_STANDBY = 0, LDC_MODE_ACTIVE = 1} LDC_MODE;

/**
* Class for the LDC1000.
* @author Victor Sluiter
* @date 2015-04-01
*/
class LDC1000
{
    public:
    /**
    * @brief Create a new Class to interface to an LDC1000
    **/
    LDC1000(PinName mosi, PinName miso, PinName sck, PinName cs, float capacitor, float f_external, PinName clock_out=NC);
    /**
    * @brief Set power mode.
    * The constructor sets the LDC1000 in Active mode.
    * @param mode choose from LDC_MODE_ACTIVE or LDC_MODE STANDBY
    **/
    void mode(LDC_MODE mode){writeSPI((uint8_t *)(&mode), 0x0B);};
    /**
    * @brief get the calculated inductance value
    **/
    float getInductance(void);


    // EXTRA test get variables values to verify (to calculate the induction)
    float get_raw_l(void);
    float get_fsensor(void);
    float get_frequency(void);
    float get_responsetime(void);
    float get_cap(void);
    

    /**
    * @brief Set the value of the external capacitor
    * This is needed for the calculation of the inductance.
    **/
    void  setCapacitor(float c){cap = c;};
    /**
    * @brief set the value of the external clock
    * If PWMout is used to generate a clock signal, this will update the output frequency.s
    **/
    void  setFrequency(float frequency);
    /**
    * @brief Read the raw 24-bit inductance value.
    * This is needed for the calculation of the inductance.
    **/
    uint32_t readRawL(void){_raw_l = readRawCounts(); return _raw_l;};


    /**
    * @brief Set the Response Time parameters. 
    * @param responsetime 
    * Larger value increases accuracy, but slows down the output data rate. Choose one of these values:
    * - LDC_RESPONSE_192
    * - LDC_RESPONSE_384
    * - LDC_RESPONSE_768
    * - LDC_RESPONSE_1536
    * - LDC_RESPONSE_3072
    * - LDC_RESPONSE_6144
    **/
    void setResponseTime(LDC_RESPONSE responsetime);
    /**
    * @brief Set the oscilation amplitude.
    * Use one of these values:
    * - LDC_AMPLITUDE_1V
    * - LDC_AMPLITUDE_2V
    * - LDC_AMPLITUDE_4V
    **/
    void setOutputPower(LDC_AMPLITUDE amplitude);
    
    /** set Watchdog timer **/
    void setWatchdog(float frequency);
    
    private:
    void readSPI(uint8_t *data, uint8_t address, uint8_t num_bytes = 1);
    void writeSPI(uint8_t *data, uint8_t address, uint8_t num_bytes = 1);
    void writeSPIregister(uint8_t reg, uint8_t value){writeSPI(&value,reg);};
    uint32_t readRawCounts(void);
    uint32_t readINTB(void); // EXTRA UNTB Read register
    LDC_RESPONSE _responsetime;
    LDC_AMPLITUDE _amplitude;
    float _fsensor;
    float _inductance;
    float _frequency; //frequency of external clock
    float cap;    
    uint32_t _raw_l;
    uint32_t INTB; // extra: read register INTB
    SPI _spiport;
    DigitalOut _cs_pin;
    FastPWM _clock;
};

#endif