/**
* @file LDC1000.h
* @brief this C++ file wcontains all required
* functions to interface with Texas
* Instruments' LDC1000.
*
* @author Victor Sluiter
*
* @date 2015-04-01
*/

#include "LDC1000.h"

LDC1000::LDC1000(PinName mosi, PinName miso, PinName sck, PinName cs, float capacitor, float f_external, PinName clock_out) : _spiport(mosi,miso,sck,NC), _cs_pin(cs), _clock(clock_out,1)
{
    cap = capacitor;
    _spiport.format(8,3);
    _spiport.frequency(1E6);
    _cs_pin.write(1);
    wait_us(100);
    mode(LDC_MODE_STANDBY);
    setFrequency(f_external);
    wait(0.1);
    wait_us(10);

    setWatchdog(5000);
    setResponseTime(LDC_RESPONSE_6144);
    setOutputPower(LDC_AMPLITUDE_4V);

    writeSPIregister(0x05,0x00);   // clock config >> we get 0x00 if this line is disabled and the cable is reconnected 
    writeSPIregister(0x0C,0x01);   // Register 0x0C enables a function that can improve L measurements while disabling RP measurements 

    mode(LDC_MODE_ACTIVE);
}

void LDC1000::setOutputPower(LDC_AMPLITUDE amplitude)
{
    uint8_t buffer;
    _amplitude  = amplitude;
    readSPI(&buffer, 0x04);
    buffer &= 0xE7; //clear amplitude bits
    buffer |= (amplitude<<3) & 0x18;
    writeSPI(&buffer,0x04);
}

void LDC1000::setWatchdog(float frequency)
{
    uint8_t buffer;
    buffer = 68.94*log(frequency/2500);
    writeSPI(&buffer,0x03);
}

void LDC1000::setResponseTime(LDC_RESPONSE responsetime)
{
    uint8_t buffer;
    _responsetime = responsetime;
    readSPI(&buffer, 0x04);
    buffer &= 0xF8; //clear responsetime bits
    buffer |= responsetime & 0x07;
    //writeSPIregister(0x04,buffer);
    writeSPI(&buffer,0x04);
}

void LDC1000::setFrequency(float frequency)
{
    _frequency = frequency;
    _clock.period(1.0/frequency);
    _clock.pulsewidth(0.5/frequency);
}

float LDC1000::getInductance()
{
    uint16_t resp[] = {0,0,192, 384, 768, 1536, 3072, 6144};
    _raw_l = readRawCounts();
    _fsensor = (_frequency/(_raw_l*3.0))*resp[(uint8_t)(_responsetime)];
    return 1./(cap*pow(2*PI*_fsensor,2));
};


uint32_t LDC1000::readRawCounts(void)
{
    uint8_t val[5];
    readSPI(val,0x21,5);
    uint32_t combinedbytes = (val[4]<<16)| (val[3]<<8) | val[2];  // combine the content of the 3 bytes from registers 23, 24 and 25 
    return combinedbytes;
}

void LDC1000::readSPI(uint8_t *data, uint8_t address, uint8_t num_bytes)
{
    _cs_pin.write(0);
    _spiport.write(address | 0x80); //read flag 
    for(int i=0; i < num_bytes ; i++)
    {
        data[i] = _spiport.write(0xFF);
    }
    _cs_pin.write(1);
}

void LDC1000::writeSPI(uint8_t *data, uint8_t address, uint8_t num_bytes)
{
    _cs_pin.write(0);
    _spiport.write(address); 
    for(int i=0; i < num_bytes ; i++)
    {
        _spiport.write(data[i]);
    }
    _cs_pin.write(1);
}


// EXTRA test: Get&print values of all variables to verify (to calculate the induction)
// The data will be printed on the screen using RealTerm: baud 9600.
// Begin ***********************************************************
    float LDC1000::get_raw_l()          {_raw_l = readRawCounts(); 
                                        return _raw_l;};        
    float LDC1000::get_fsensor()        {
    uint16_t resp[] = {0, 0, 192, 384, 768, 1536, 3072, 6144};
    _raw_l = readRawCounts();
    _fsensor = (_frequency/(_raw_l*3.0))*resp[(uint8_t)(_responsetime)];                
        return _fsensor;};        
    
    float LDC1000::get_frequency()      {return _frequency;};    
    float LDC1000::get_responsetime()   {return _responsetime;};    
    float LDC1000::get_cap()            {return cap;};
// END ***********************************************************