/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "LEDService.h"
#include "ButtonService.h"
#include "hcsr04.h"
#define LED_ON          0
#define LED_OFF         1

DigitalOut alivenessLED(LED1, LED_OFF); // green
DigitalOut actuatedLED(LED2, LED_OFF);  // red
InterruptIn button(PC_13);
HCSR04 usensor(D14,D15);
const static char     DEVICE_NAME[] = "vs_device";
static const uint16_t uuid16_list[] = { 
    LEDService::LED_SERVICE_UUID, 
    ButtonService::BUTTON_SERVICE_UUID
};

LEDService *ledServicePtr;
ButtonService *buttonServicePtr;

Ticker ticker;

void periodicCallback(void)
{
    alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
    
    ticker.attach(periodicCallback, 1);
}

void connectionCallback(const Gap::ConnectionCallbackParams_t * params)
{
    // stop blinking when we connect
    ticker.detach();
    // also put the led off
    alivenessLED = LED_OFF;
}

/**
 * This callback allows the LEDService to receive updates to the ledState Characteristic.
 *
 * @param[in] params
 *     Information about the characterisitc being updated.
 */
void onDataWrittenCallback(const GattWriteCallbackParams *params) {
    // handle corresponds to the characteristic being written
    // then we can read data to get a buffer of the actual data
    if ((params->handle == ledServicePtr->getValueHandle()) && (params->len == 1)) {
        // When writing 1 -> turn LED on, 0 -> turn LED off
        char val = params->data[0];
        actuatedLED = val == 1 ? LED_ON : LED_OFF;
    }
}

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    // blink fast when we encountered an error
    ticker.attach(periodicCallback, 0.2);
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
 
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(connectionCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);

    // Begin - If you add a new service, add it here!
    ledServicePtr = new LEDService(ble, false /* inital value */);
    buttonServicePtr = new ButtonService(ble, false /* initial value */);
    // End - If you add a new service, add it here!

    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(50); /* 50ms. Hack for EvoThings. */
    ble.gap().startAdvertising();
}

void button_down() {
    if (!buttonServicePtr) return;
    buttonServicePtr->updateButtonState(usensor.get_dist_cm());
}
void button_up() {
    if (!buttonServicePtr) return;
    
}

int main(void)
{
    // Blink the green LED!
    ticker.attach(periodicCallback, 1);
    usensor.start();
    button.fall(&button_down);
    button.rise(&button_up);
    
    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }

    while (true) {
        ble.waitForEvent();
    }
}
