#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "SPI_TFT_ILI9341.h"
#include "SDFileSystem.h"
#include "Shell.h"
#include "HTU21D.h"
#include "oauth.h"

Serial pc(p28, p27); // (USBTX, USBRX);
DigitalOut myled(LED1);
EthernetInterface eth;
HTU21D htu21d(p9,p10);

SDFileSystem sd(p5,p6,p7,p8,"sd"); // mosi, miso, sck, cs 
SPI_TFT_ILI9341 TFT(p11,p12,p13,p15, p16, p17 ); // mosi, miso, sck, cs, eset, dc 
DigitalOut lcdOn(p14);

#define SHELL_STACK_SIZ 1024
// Pre-allocate the shell's stack (on global mem)
unsigned char shellStack[SHELL_STACK_SIZ];
Shell shell(&pc);

// IMPORTANT: please change the following keys for your application.

static char const consumer_key[] = "AAAAAAAAAAAAAAAAAAAAAA";
static char const consumer_secret[] = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";

static char const token_key[] = "00000000-AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";
static char const token_secret[] = "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";

//
// Sends tweets to Twitter
//
void tweet(char const *message)
{

    std::string uri = "http://api.twitter.com/1/statuses/update.xml";
    uri += "?status=";
    uri += oauth_url_escape(message);

    std::string req_url;
    std::string postarg;

    req_url = oauth_sign_url2(uri.c_str(), &postarg, OA_HMAC, 0, consumer_key, consumer_secret, token_key, token_secret);
    oauth_http_post(req_url.c_str(), postarg.c_str());
}

static uint32_t get_mem()
{
   // In order to get free mem within RTOS
   // we need to get the main thread's stack pointer
   // and subtract it with the top of the heap
   // ------+-------------------+   Last Address of RAM (INITIAL_SP)
   //       | Scheduler Stack   |
   //       +-------------------+
   //       | Main Thread Stack |
   //       |         |         |
   //       |         v         |
   //       +-------------------+ <- bottom_of_stack/__get_MSP()
   // RAM   |                   | 
   //       |  Available RAM    |  
   //       |                   |  
   //       +-------------------+ <- top_of_heap
   //       |         ^         |
   //       |         |         |
   //       |       Heap        |
   //       +-------------------+ <- __end__ / HEAP_START (linker defined var)
   //       | ZI                |
   //       +-------------------+
   //       | ZI: Shell Stack   |
   //       +-------------------+
   //       | ZI: Idle Stack    |
   //       +-------------------+
   //       | ZI: Timer Stack   |
   //       +-------------------+
   //       | RW                |  
   // ------+===================+  First Address of RAM
   //       |                   |
   // Flash |                   |
   //

   uint32_t bottom_of_stack = __get_MSP();
   char     * top_of_heap =  (char *) malloc(sizeof(char));
   uint32_t diff = bottom_of_stack - (uint32_t) top_of_heap;

   free((void *) top_of_heap);
   
   return diff;    
}

// Local Commands
/**
 *  \brief Gets the amount of free memory
 *  \param none
 *  \return none
 **/
static void cmd_mem(Stream * chp, int argc, char * argv[])
{
   chp->printf("Available Memory : %d bytes\r\n",
        get_mem());
}

/**
 *  \brief List Directories and files 
 *  \param none
 *  \return int
 **/
static void cmd_ls(Stream * chp, int argc, char * argv[])
{
   DIR * dp;
   struct dirent * dirp;
   char dirroot[256];
   
   if (argc >= 1)
       sprintf(dirroot, "/sd/%s", argv[0]);
   else
       sprintf(dirroot, "/sd");
   
   chp->printf("Listing directory [%s]\r\n", dirroot);
   
   dp = opendir(dirroot);           
   while((dirp = readdir(dp)) != NULL)
   {
       chp->printf("\t%s\r\n", dirp->d_name);
   }
   closedir(dp);
}

static void cmd_load(Stream * chp, int argc, char * argv[])
{
   char filename[256];
   
   if (argc != 1)
   {
       chp->printf("load <bitmapfile>\r\n");
       return;
   }
   
   sprintf(filename, "/sd/%s", argv[0]);
       // Load a bitmap startup file
   int err = TFT.BMP_16(0,0, filename);
   if (err != 1) TFT.printf(" - Err: %d", err); 
}

/**
 *  \brief Gets sensor data on HTU21D
 *  \param none
 *  \return int
 **/
static void cmd_sensor(Stream * chp, int argc, char * argv[])
{
    chp->printf("Temperature : %d °C\r\n", htu21d.sample_ctemp());
    chp->printf("Humitdity : %d%%\r\n", htu21d.sample_humid());
}

/**
 * \brief Initialize LCD
 * \param none
 * \return void
 **/
void init_LCD()
{
    pc.printf("Initializing LCD Screen ...\r\n");

    // Turn on the LCD
    lcdOn = 1;

    TFT.claim(stdout);  
    TFT.set_orientation(2);
    TFT.background(Black);    // set background to black
    TFT.foreground(White);    // set chars to white
    TFT.cls();                // clear the screen


    TFT.set_font((unsigned char*) Arial12x12);
    TFT.locate(0,0);

    printf("Hello World of EMBED!\n");

}

void led1_thread(void const *args) {
    while (true) {
        myled = !myled;
        Thread::wait(1000);
    }
}

int main() {
    Thread * thread;
    Thread * thread2;
    pc.baud(115200);

    pc.printf("\r\nStarting Mbed ...\r\n");
    //Initialize the LCD
    pc.printf("Initializing LCD ...\r\n");
    init_LCD();
    
    printf("Inititalizing ethernet ....\r\n");
    eth.init(); // Use DHCP
    eth.connect();
    printf("IP Address is %s\n", eth.getIPAddress());
    
    // After initializing the ethernet interface
    // run it in its own thread
    thread = new Thread(led1_thread);

    // Start the shell
    printf("Starting debug shell ...\r\n");
    shell.addCommand("ls", cmd_ls);
    shell.addCommand("load", cmd_load);
    shell.addCommand("mem", cmd_mem);
    shell.addCommand("sensor", cmd_sensor);
    shell.start(osPriorityNormal, SHELL_STACK_SIZ, shellStack);
    printf("Shell now running!\r\n");
    printf("Available Memory : %d\r\n", get_mem());
        
    // Do something logical here
    // other than looping
    while(1) {
        printf("Temperature : %d °C\r\n", htu21d.sample_ctemp());
        printf("Humitdity : %d%%\r\n", htu21d.sample_humid());        
        wait(10);
    }
    
    thread->terminate();
    delete thread;
}
