#include "mbed.h"
#include "EthernetInterface.h"
#include "SDFileSystem.h"
#include "Shell.h"
 
#define MBED_DEV_IP       "192.168.0.52"
#define MBED_DEV_MASK     "255.255.255.0"
#define MBED_DEV_GW       "0.0.0.0"
#define ECHO_SERVER_PORT   5000

#define SD_MOSI PTE3
#define SD_MISO PTE1
#define SD_SCLK PTE2
#define SD_CS   PTE4
#define SD_DETECT PTE6

Serial pc(PTC17,PTC16);
SDFileSystem sd(SD_MOSI, SD_MISO, SD_SCLK, SD_CS, "sd");

#define SHELL_STACK_SIZ 1024
// Pre-allocate the shell's stack (on global mem)
unsigned char shellStack[SHELL_STACK_SIZ];
Shell shell(&pc);

/**
 *  \brief Gets the amount of free memory
 *  \param none
 *  \return none
 **/
static void cmd_mem(Stream * chp, int argc, char * argv[])
{
   // In order to get free mem within RTOS
   // we need to get the main thread's stack pointer
   // and subtract it with the top of the heap
   // ------+-------------------+   Last Address of RAM (INITIAL_SP)
   //       | Scheduler Stack   |
   //       +-------------------+
   //       | Main Thread Stack |
   //       |         |         |
   //       |         v         |
   //       +-------------------+ <- bottom_of_stack/__get_MSP()
   // RAM   |                   | 
   //       |  Available RAM    |  
   //       |                   |  
   //       +-------------------+ <- top_of_heap
   //       |         ^         |
   //       |         |         |
   //       |       Heap        |
   //       +-------------------+ <- __end__ / HEAP_START (linker defined var)
   //       | ZI                |
   //       +-------------------+
   //       | ZI: Shell Stack   |
   //       +-------------------+
   //       | ZI: Idle Stack    |
   //       +-------------------+
   //       | ZI: Timer Stack   |
   //       +-------------------+
   //       | RW                |  
   // ------+===================+  First Address of RAM
   //       |                   |
   // Flash |                   |
   //

   uint32_t bottom_of_stack = __get_MSP();
   char     * top_of_heap =  (char *) malloc(sizeof(char));
   uint32_t diff = bottom_of_stack - (uint32_t) top_of_heap;

   free((void *) top_of_heap);

   chp->printf("Available Memory : %d bytes\r\n",
        diff);
}

/**
 *  \brief List Directories and files 
 *  \param none
 *  \return int
 **/
static void cmd_ls(Stream * chp, int argc, char * argv[])
{
   DIR * dp;
   struct dirent * dirp;
   FILINFO fileInfo;
   char dirroot[256];
   
   if (argc >= 1)
       sprintf(dirroot, "/sd/%s", argv[0]);
   else
       sprintf(dirroot, "/sd");
   
   chp->printf("Listing directory [%s]\r\n", dirroot);
   
   dp = opendir(dirroot);           
   while((dirp = readdir(dp)) != NULL)
   {
       if (sd.stat(dirp->d_name, &fileInfo) == 0)
       {
           if (fileInfo.fattrib & AM_DIR )
                   chp->printf("<DIR>\t\t");
           else
                   chp->printf("%ld\t\t", fileInfo.fsize);
       }
       chp->printf("%s\r\n", dirp->d_name);
   }
   closedir(dp);
}

 
int main (void) {
    EthernetInterface eth;
    eth.init(MBED_DEV_IP, MBED_DEV_MASK, MBED_DEV_GW); //Assign a device ip, mask and gateway
    eth.connect();
    printf("IP Address is %s\n", eth.getIPAddress());
    
    TCPSocketServer server;
    server.bind(ECHO_SERVER_PORT);
    server.listen();

    // Start the shell
    pc.printf("Starting debug shell ...\r\n");
    shell.addCommand("ls", cmd_ls);
    shell.addCommand("mem", cmd_mem);
    // Start the thread statically (separate stack)
    shell.start(osPriorityNormal, SHELL_STACK_SIZ, shellStack);
    
        
    while (true) {
        printf("\nWait for new connection...\n");
        TCPSocketConnection client;
        server.accept(client);
        client.set_blocking(false, 1500); // Timeout after (1.5)s
        
        printf("Connection from: %s\n", client.get_address());
        char buffer[256];
        while (true) {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) break;
            
            client.send_all(buffer, n);
            if (n <= 0) break;
        }
        
        client.close();
    }
}