#include "LoRa.h"
#include "Logging.h"

#define MAX_SEND_LIST_SIZE 10
#define KEEPALIVE_MESSAGE "0K\n"

RtosTimer* LoRaReadTimer;

LoRa::LoRa()
    : m_nextSend(0),
    m_nextKeepalive(0)
{
    m_dot = mDot::getInstance();
}

LoRa::~LoRa()
{
}

void LoRa::PeriodicTimer()
{
    // On commence par gérer les cas ou on a un vrai message à envoyer
    if (m_toSend.size() > 0) {
        if (_send(m_toSend.front()) == LoRa::e_srWaitMore) {
            return; // <==
        }
        m_toSend.pop_front();
    }
    else if (m_currentMessage.length() > 0) {
        if (_send(m_currentMessage) == LoRa::e_srWaitMore) {
            return; // <==
        }
        m_currentMessage.clear();
    }
    else if (m_nextKeepalive > time(NULL)) {
        if (_send(KEEPALIVE_MESSAGE) == LoRa::e_srWaitMore) {
            return; // <==
        }
    }
    else {
        return; // <==
    }
    
    std::string data = _receive();
    if (data.size() > 0) {
        m_readed.push_back(data);
    }
}

bool LoRa::init(LoRaConfig_t p_loraConfig)
{
    m_delayForKeepAlive = (uint32_t)p_loraConfig.loraDelayForKeepAlive;
    m_minDelayBetweenMessage = (uint32_t)p_loraConfig.loraMinDelayBetweenMessage;
    
    m_dot->resetConfig();
    m_dot->resetNetworkSession();
       
    m_errorCode = mDot::MDOT_OK;
    std::vector<uint8_t> temp;
    
    LOG_DEBUG("Setting frequency sub band to '%d'...", p_loraConfig.subBand);
    if ((m_errorCode = m_dot->setFrequencySubBand(p_loraConfig.subBand)) != mDot::MDOT_OK) {
        LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
        return false;
    }
    
    LOG_DEBUG("Setting network name to '%s'", p_loraConfig.networkName.c_str());
    if ((m_errorCode = m_dot->setNetworkName(p_loraConfig.networkName)) != mDot::MDOT_OK) {
        LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
        return false;
    }
    
    LOG_DEBUG("Setting network passphrase to '%s'", p_loraConfig.networkPassphrase.c_str());
    if ((m_errorCode = m_dot->setNetworkPassphrase(p_loraConfig.networkPassphrase)) != mDot::MDOT_OK) {
        LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
        return false;
    }
    
    LOG_DEBUG("Setting TX spreading factor");
    if ((m_errorCode = m_dot->setTxDataRate(p_loraConfig.spreadingFactor)) != mDot::MDOT_OK) {
         LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
         return false;
    }
    
    // request receive confirmation of packets from the gateway
    LOG_DEBUG("Disabling ACKs");
    if ((m_errorCode = m_dot->setAck(0)) != mDot::MDOT_OK) {
         LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
         return false;
    }
    
    LOG_DEBUG("Saving config");
    if (!m_dot->saveConfig()) {
        LOG_ERROR("Failed to save configuration");
    }

    LOG_DEBUG("LoRa initialized");
    return true;
}

bool LoRa::connect()
{
    m_dot->resetNetworkSession();
    LOG_DEBUG("Joining network");
    while ((m_errorCode = m_dot->joinNetwork()) != mDot::MDOT_OK) {
        m_dot->resetNetworkSession();
        LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
        osDelay(m_dot->getNextTxMs());
    }
    LOG_DEBUG("Network Joined!\r\n");
    return true;
}

std::string LoRa::getVersion()
{
    return m_dot->getId();
}

LoRa::SendResult LoRa::_send(const std::string text)
{
    time_t now = time(NULL);
    
    if (m_nextSend > now) {
        return e_srWaitMore; // <==
    }
    
    uint32_t timeTillSend = m_dot->getNextTxMs();
    if (timeTillSend != 0) {
        LOG_DEBUG("waiting %lu ms to send\r\n", timeTillSend);
        m_nextSend = time(NULL) + (uint16_t)(timeTillSend / 1000);
        return e_srWaitMore;
    }

    m_errorCode = mDot::MDOT_OK;
     
    std::string data_str = text;
    LOG_DEBUG("Sending data...  %s", data_str.c_str());
    std::vector<uint8_t> data;
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);

    if ((m_errorCode = m_dot->send(data)) != mDot::MDOT_OK)
    {
        LOG_ERROR("(%d): %s", m_errorCode, getError().c_str());
        return e_srError;
    }
    LOG_DEBUG("Sended data...");
    
    uint16_t secTillSend =  (uint16_t)(m_dot->getNextTxMs() / 1000);
    m_nextSend = time(NULL) + std::max(m_minDelayBetweenMessage, secTillSend);
    m_nextKeepalive = time(NULL) + std::max(m_delayForKeepAlive, secTillSend);
    
    return e_srOK;
}

void LoRa::send(const std::string text)
{
    if (text.length() > LORA_PACKET_MAX_SIZE) {
        return; // Message trop gros on ne fait rien
    }
    
    if ((text.length() + m_currentMessage.length()) <= LORA_PACKET_MAX_SIZE) {
        m_currentMessage = m_currentMessage + text;
        return;
    }
    
    if (m_toSend.size() >= MAX_SEND_LIST_SIZE) {
        return;
    }
    m_toSend.push_back(m_currentMessage);
    m_currentMessage = text;
}

std::string LoRa::_receive()
{
    LOG_DEBUG("Receiving packet...  ");
    std::vector<uint8_t> data;
 
    if (m_dot->recv(data) == mDot::MDOT_OK) {
        if (data.size() > 0) {
            std::string res = std::string((const char*) &data[0], data.size());
            LOG_DEBUG("Received data...  %s", res.c_str());
            return res;
        }
    }
    return "";
}
 
std::string LoRa::receive()
{
    if (m_readed.size() > 0) {
        std::string res = m_readed.front();
        m_readed.pop_front();
        return res;
    }
    return "";
}
 
std::string LoRa::getError()
{   
    return mDot::getReturnCodeString(m_errorCode) + " - " + m_dot->getLastError();
}
