#define REVISION "1.4"
 
#include <string>
#include <vector>
#include <MTSText.h>
 
#include <mbed.h>
 
#include "DHT.h"
#include "LoRa.h"
#include "Sensors.h"
 
#include "Config.h"
 
#include "Logging.h"

LoRa* loRa;
Sensors* sensors;
int forceUpdate;
 
struct SensorsVal_t {
    float temp;
    float humidity; 
    bool motion;
    float light;
    bool switch1;
    bool switch2;
} sensorsVal;

void SendLoraError(char p_idx) {
    std::string res;
    res += p_idx;
    res += "0\n";
    loRa->send(res);
}

void CmdGet(std::string p_line)
{
    if (p_line.size() < 3) {
        SendLoraError(p_line[0]);
        return; // <==
    }

    char val[255];

    if (p_line[2] == 'L') {
        sprintf(val, "%4.2f", sensors->getLight());
    } else if (p_line[2] == 'T') {
        sprintf(val, "%4.2f", sensors->getTemp());
    } else if (p_line[2] == 'H') {
        sprintf(val, "%4.2f", sensors->getHumidity());
    } else if (p_line[2] == 'M') {
        sprintf(val, "%d", sensors->getMotion() ? 1 : 0);
    } else if (p_line[2] == '1') {
        sprintf(val, "%d", (sensors->getSwitch1() ? 1 : 0));
    } else if (p_line[2] == '2') {
        sprintf(val, "%d", (sensors->getSwitch2() ? 1 : 0));
    } else if (p_line[2] == 'N') {
        mDot::link_check lc = mDot::getInstance()->networkLinkCheck();
        sprintf(val, "%d", lc.dBm);
    } else  {
        SendLoraError(p_line[0]);
        return; // <==
    }

    string res;
    res = p_line[0];
    res = "A";
    res += p_line[2];
    res += val;
    res += "\n";
    loRa->send(res);
}

void CmdSet(std::string p_line)
{
    if (p_line.size() < 4) {
        SendLoraError(p_line[0]);
        return; // <==
    }

    if (p_line[2] == '1') {
        sensors->setSwitch1(p_line[3] != '0');
    } else if (p_line[2] == '2') {
        sensors->setSwitch2(p_line[3] != '0');
    } else {
        SendLoraError(p_line[0]);
        return; // <==
    }

    std::string res;
    res += p_line[0];
    res += "D\n";
    loRa->send(res);
}

void CmdShow(std::string p_line)
{
    if (p_line.size() < 3) {
        SendLoraError(p_line[0]);
        return; // <==
    }

    if (p_line[2] == 'V') {
        loRa->send(p_line[0] + std::string(REVISION) + "\n");
        return; // <==
    }

    SendLoraError(p_line[0]);
}

void SensorsTimerCb()
{
    // Check sensors for change
    //
    double minDelta = 0.15; // 15%
    char buff[LORA_PACKET_MAX_SIZE + 1];
    if (forceUpdate <= 0 || sensors->getTemp() > (sensorsVal.temp * (1.0 + minDelta))
            || sensors->getTemp() < (sensorsVal.temp * (1.0 - minDelta))) {
        sensorsVal.temp = sensors->getTemp();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0PT%4.2f\n", sensorsVal.temp);
        loRa->send(buff);
    }
    if (forceUpdate <= 0 || sensors->getHumidity() > (sensorsVal.humidity * (1.0 + minDelta))
            || sensors->getHumidity() < (sensorsVal.humidity * (1.0 - minDelta))) {
        sensorsVal.humidity = sensors->getHumidity();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0PH%4.2f\n", sensorsVal.humidity);
        loRa->send(buff);
    }
    if (forceUpdate <= 0 || sensors->getLight() > (sensorsVal.light * (1.0 + minDelta))
            || sensors->getLight() < (sensorsVal.light * (1.0 - minDelta))) {
        sensorsVal.light = sensors->getLight();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0PL%4.2f\n", sensorsVal.light);
        loRa->send(buff);
    }
    if (forceUpdate <= 0 || sensorsVal.motion != sensors->getMotion()) {
        sensorsVal.motion = sensors->getMotion();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0PM%d\n", sensorsVal.motion);
        loRa->send(buff);
    }
    if (forceUpdate <= 0 || sensorsVal.switch1 != sensors->getSwitch1()) {
        sensorsVal.switch1 = sensors->getSwitch1();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0P1%d\n", sensorsVal.switch1);
        loRa->send(buff);
    }
    if (forceUpdate <= 0 || sensorsVal.switch2 != sensors->getSwitch2()) {
        sensorsVal.switch2 = sensors->getSwitch2();
        snprintf(buff, LORA_PACKET_MAX_SIZE, "0P2%d\n", sensorsVal.switch2);
       loRa->send(buff);
    }
    
    if (forceUpdate <= 0) {
        forceUpdate = ForceUpdatePeriod;
    } else {
        forceUpdate--;
    }
}

void ReadTimerCb()
{
    std::string data = loRa->receive();
    if (data == "") {
        return; // <==
    }

    size_t cursor = 0;
    while (true) {
        if (cursor == (size_t)-1) return;
        std::string line = mts::Text::getLine(data, cursor, cursor);
        if (line.size() == 0) return;
        if (line.size() < 2) {
            return;
        } else if (line[1] == 'G') {
            CmdGet(line);
        } else if (line[1] == 'S') {
            CmdSet(line);
        } else if (line[1] == 'H') {
            CmdShow(line);
        } else if (line[1] == 'I') {
            std::string res;
            res += line[0];
            res += "O\n";
            loRa->send(res);
        } else {
            std::string res;
            res += line[0];
            res += "E\n";
            loRa->send(res);
        }
    }
}

int main()
{
    printf("--STARTING-- %s\n", mDot::getInstance()->getId().c_str());
     
    sensors = new Sensors(sensorsConfig);
    
    forceUpdate = 1;

    // Get initial Sensors value
    sensorsVal.temp = sensors->getTemp();
    sensorsVal.humidity = sensors->getHumidity();
    sensorsVal.motion = sensors->getMotion();
    sensorsVal.light = sensors->getLight();
    sensorsVal.switch1 = sensors->getSwitch1();
    sensorsVal.switch2 = sensors->getSwitch2();
 
    loRa = new LoRa();
    loRa->init(LoRaConfig);
    loRa->connect();
    loRa->send("0S1\n");

    while (true) {
        loRa->PeriodicTimer();
        sensors->updateValues();
        ReadTimerCb();
        SensorsTimerCb();
        osDelay(1000);
    }
}
