/* mbed Library for FTDI FT813  Enbedded Video Engine "EVE"
 * based on Original Code Sample from FTDI
 * ported to mbed by Peter Drescher, DC2PD 2014
 * Released under the MIT License: http://mbed.org/license/mit */

#include "FT_Platform.h"

ft_void_t FT813::SendCmd(ft_uint32_t cmd)
{
    Transfer32(cmd);
}

ft_void_t FT813::SendStr(const ft_char8_t *s)
{
    TransferString(s);
}

ft_void_t FT813::StartFunc(ft_uint16_t count)
{
    _count = count;
    CheckCmdBuffer(_count);
    StartCmdTransfer(FT_GPU_WRITE, _count);
}

ft_void_t FT813::EndFunc()
{
    EndTransfer();
    Updatecmdfifo(_count);
}

// Start a new display list
// FT81x Series Programmers Guide Section 5.11
ft_void_t FT813::DLstart()
{
//    _address = 0;
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_DLSTART);
    EndFunc();
}

// Swap the current display list with the new display list
// FT81x Series Programmers Guide Section 5.12
ft_void_t FT813::Swap()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_SWAP);
    EndFunc();
}

// Set co-processor engine state to default values
// FT81x Series Programmers Guide Section 5.13
ft_void_t FT813::ColdStart()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_COLDSTART);
    EndFunc();
}

// Trigger interrupt INT_CMDFLAG
// FT81x Series Programmers Guide Section 5.14
ft_void_t FT813::Interrupt(ft_uint32_t ms)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_INTERRUPT);
    SendCmd(ms);
    EndFunc();
}

// Append more commands to current display list
// FT81x Series Programmers Guide Section 5.15
ft_void_t FT813::Append(ft_uint32_t ptr, ft_uint32_t num)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_APPEND);
    SendCmd(ptr);
    SendCmd(num);
    EndFunc();
}

// Read a register value 
// FT81x Series Programmers Guide Section 5.16
ft_void_t FT813::RegRead(ft_uint32_t ptr, ft_uint32_t result)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_REGREAD);
    SendCmd(ptr);
    SendCmd(0);
    EndFunc();
}

/* commands to operate on memory: */

// Write bytes into memory
// FT81x Series Programmers Guide Section 5.17
ft_void_t FT813::MemWrite(ft_uint32_t ptr, ft_uint32_t num)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_MEMWRITE);
    SendCmd(ptr);
    SendCmd(num);
    EndFunc();
}

// Decompress data into memory
// FT81x Series Programmers Guide Section 5.18
ft_void_t FT813::Inflate(ft_uint32_t ptr)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_INFLATE);
    SendCmd(ptr);
    EndFunc();
}

// Load a JPEG or PNG image
// FT81x Series Programmers Guide Section 5.19
ft_void_t FT813::LoadImage(ft_uint32_t ptr, ft_uint32_t options)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_LOADIMAGE);
    SendCmd(ptr);
    SendCmd(options);
    EndFunc();
}

// Set up a streaming media FIFO in RAM_G
// FT81x Series Programmers Guide Section 5.20
ft_void_t FT813::MediaFifo(ft_uint32_t ptr, ft_uint32_t size)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_MEDIAFIFO);
    SendCmd(ptr);
    SendCmd(size);
    EndFunc();
}

// Video playback
// FT81x Series Programmers Guide Section 5.21
ft_void_t FT813::PlayVideo(ft_uint32_t opts)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_PLAYVIDEO);
    SendCmd(opts);
    EndFunc();
}

// Initialize video frame decoder
// FT81x Series Programmers Guide Section 5.22
ft_void_t FT813::VideoStart()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_VIDEOSTART);
    EndFunc();
}

// Load the next frame of video
// FT81x Series Programmers Guide Section 5.23
ft_void_t FT813::VideoFrame(ft_uint32_t dst, ft_uint32_t ptr)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_VIDEOFRAME);
    SendCmd(dst);
    SendCmd(ptr);
    EndFunc();
}

// Compute a CRC-32 for memory
// FT81x Series Programmers Guide Section 5.24
ft_void_t FT813::MemCrc(ft_uint32_t ptr, ft_uint32_t num, ft_uint32_t result)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_MEMCRC);
    SendCmd(ptr);
    SendCmd(num);
    SendCmd(result);
    EndFunc();
}

// Write zero to a block of memory
// FT81x Series Programmers Guide Section 5.25
ft_void_t FT813::MemZero(ft_uint32_t ptr, ft_uint32_t num)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_MEMZERO);
    SendCmd(ptr);
    SendCmd(num);
    EndFunc();
}

// Fill memory with a byte value
// FT81x Series Programmers Guide Section 5.26
ft_void_t FT813::MemSet(ft_uint32_t ptr, ft_uint32_t value, ft_uint32_t num)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_MEMSET);
    SendCmd(ptr);
    SendCmd(value);
    SendCmd(num);
    EndFunc();
}

// Cmd_Memcpy - background copy a block of data
// FT81x Series Programmers Guide Section 5.27
ft_void_t FT813::Memcpy(ft_uint32_t dest, ft_uint32_t src, ft_uint32_t num)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_MEMCPY);
    SendCmd(dest);
    SendCmd(src);
    SendCmd(num);
    EndFunc();
}

/* commands to draw graphics objects: */
// ******************** Screen Object Creation CoProcessor Command Functions ******************************

// Draw a button
// FT81x Series Programmers Guide Section 5.28
ft_void_t FT813::Button(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_int16_t font, ft_uint16_t options, const ft_char8_t* s)
{
    StartFunc(FT_CMD_SIZE*4 + strlen(s) + 1);
    SendCmd(CMD_BUTTON);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd((((ft_uint32_t)options<<16)|font));       // patch from Ivano Pelicella to draw flat buttons
    SendStr(s);
    EndFunc();
}

// Draw an analog clock
// FT81x Series Programmers Guide Section 5.29
ft_void_t FT813::Clock(ft_int16_t x, ft_int16_t y, ft_int16_t r, ft_uint16_t options, ft_uint16_t h, ft_uint16_t m, ft_uint16_t s, ft_uint16_t ms)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_CLOCK);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)options<<16)|r));
    SendCmd((((ft_uint32_t)m<<16)|h));
    SendCmd((((ft_uint32_t)ms<<16)|s));
    EndFunc();
}

// Set the foreground color
// FT81x Series Programmers Guide Section 5.30
ft_void_t FT813::FgColor(ft_uint32_t c)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_FGCOLOR);
    SendCmd(c);
    EndFunc();
}

// Set the background color
// FT81x Series Programmers Guide Section 5.31
ft_void_t FT813::BgColor(ft_uint32_t c)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_BGCOLOR);
    SendCmd(c);
    EndFunc();
}

// Set the 3D button highlight color - Set Highlight Gradient Color
// FT81x Series Programmers Guide Section 5.32
ft_void_t FT813::GradColor(ft_uint32_t c)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_GRADCOLOR);
    SendCmd(c);
    EndFunc();
}

// Draw a gauge
// FT81x Series Programmers Guide Section 5.33
/* Error handling for val is not done, so better to always use range of 65535 in order that needle is drawn within display region */
ft_void_t FT813::Gauge(ft_int16_t x, ft_int16_t y, ft_int16_t r, ft_uint16_t options, ft_uint16_t major, ft_uint16_t minor, ft_uint16_t val, ft_uint16_t range)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_GAUGE);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)options<<16)|r));
    SendCmd((((ft_uint32_t)minor<<16)|major));
    SendCmd((((ft_uint32_t)range<<16)|val));
    EndFunc();
}

// Draw a smooth color gradient
// FT81x Series Programmers Guide Section 5.34
ft_void_t FT813::Gradient(ft_int16_t x0, ft_int16_t y0, ft_uint32_t rgb0, ft_int16_t x1, ft_int16_t y1, ft_uint32_t rgb1)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_GRADIENT);
    SendCmd((((ft_uint32_t)y0<<16)|(x0 & 0xffff)));
    SendCmd(rgb0);
    SendCmd((((ft_uint32_t)y1<<16)|(x1 & 0xffff)));
    SendCmd(rgb1);
    EndFunc();
}

// Draw a row of keys
// FT81x Series Programmers Guide Section 5.35
ft_void_t FT813::Keys(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_int16_t font, ft_uint16_t options, const ft_char8_t* s)
{
    StartFunc(FT_CMD_SIZE*4 + strlen(s) + 1);
    SendCmd(CMD_KEYS);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd((((ft_uint32_t)options<<16)|font));
    SendStr(s);
    EndFunc();
}

// Draw a progress bar
// FT81x Series Programmers Guide Section 5.36
ft_void_t FT813::Progress(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_uint16_t options, ft_uint16_t val, ft_uint16_t range)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_PROGRESS);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd((((ft_uint32_t)val<<16)|options));
    SendCmd(range);
    EndFunc();
}

// Draw a scroll bar
// FT81x Series Programmers Guide Section 5.37
ft_void_t FT813::Scrollbar(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_uint16_t options, ft_uint16_t val, ft_uint16_t size, ft_uint16_t range)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_SCROLLBAR);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd((((ft_uint32_t)val<<16)|options));
    SendCmd((((ft_uint32_t)range<<16)|size));
    EndFunc();
}

// Draw a slider
// FT81x Series Programmers Guide Section 5.38
ft_void_t FT813::Slider(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_uint16_t options, ft_uint16_t val, ft_uint16_t range)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_SLIDER);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd((((ft_uint32_t)val<<16)|options));
    SendCmd(range);
    EndFunc();
}

// Draw a rotary dial control
// FT81x Series Programmers Guide Section 5.39
// This is much like a Gauge except for the helpful range parameter.  For some reason, all dials are 65535 around.
ft_void_t FT813::Dial(ft_int16_t x, ft_int16_t y, ft_int16_t r, ft_uint16_t options, ft_uint16_t val)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_DIAL);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)options<<16)|r));
    SendCmd(val);
    EndFunc();
}

// Draw a toggle switch
// FT81x Series Programmers Guide Section 5.40
ft_void_t FT813::Toggle(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t font, ft_uint16_t options, ft_uint16_t state, const ft_char8_t* s)
{
    StartFunc(FT_CMD_SIZE*4 + strlen(s) + 1);
    SendCmd(CMD_TOGGLE);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)font<<16)|w));
    SendCmd((((ft_uint32_t)state<<16)|options));
    SendStr(s);
    EndFunc();
}

// Draw Text
// FT81x Series Programmers Guide Section 5.41
ft_void_t FT813::Text(ft_int16_t x, ft_int16_t y, ft_int16_t font, ft_uint16_t options, const ft_char8_t* s)
{
    StartFunc(FT_CMD_SIZE*3 + strlen(s) + 1);
    SendCmd(CMD_TEXT);
    //Copro_SendCmd((((ft_uint32_t)y<<16)|(ft_uint32_t)x));
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)options<<16)|(ft_uint32_t)font));
    SendStr(s);
    EndFunc();
}

// Set the base for number output
// FT81x Series Programmers Guide Section 5.42
ft_void_t FT813::SetBase(ft_uint32_t base)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_SETBASE);
    SendCmd(base);
    EndFunc();
}

// Draw number
// FT81x Series Programmers Guide Section 5.43
ft_void_t FT813::Number(ft_int16_t x, ft_int16_t y, ft_int16_t font, ft_uint16_t options, ft_int32_t n)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_NUMBER);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)options<<16)|font));
    SendCmd(n);
    EndFunc();
}

// Set the current matrix to the identity matrix
// FT81x Series Programmers Guide Section 5.44
ft_void_t FT813::LoadIdentity()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_LOADIDENTITY);
    EndFunc();
}

// Write the current matrix to the display list
// FT81x Series Programmers Guide Section 5.45
ft_void_t FT813::SetMatrix()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_SETMATRIX);
    EndFunc();
}

// Retrieves the current matrix coefficients
// FT81x Series Programmers Guide Section 5.46
ft_void_t FT813::GetMatrix(ft_int32_t a, ft_int32_t b, ft_int32_t c, ft_int32_t d, ft_int32_t e, ft_int32_t f)
{
    StartFunc(FT_CMD_SIZE*7);
    SendCmd(CMD_GETMATRIX);
    SendCmd(a);
    SendCmd(b);
    SendCmd(c);
    SendCmd(d);
    SendCmd(e);
    SendCmd(f);
    EndFunc();
}

// Get the end memory address of data inflated by CMD_INFLATE - Get the last used address from CoPro operation
// FT81x Series Programmers Guide Section 5.47
ft_void_t FT813::GetPtr(ft_uint32_t result)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_GETPTR);
    SendCmd(result);
    EndFunc();
}

// Get the image properties decompressed by CMD_LOADIMAGE
// FT81x Series Programmers Guide Section 5.48
// Jack
ft_void_t FT813::GetProps(ft_uint32_t ptr, ft_uint32_t w, ft_uint32_t h)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_GETPROPS);
    SendCmd(ptr);
    SendCmd(w);
    SendCmd(h);
    EndFunc();
}

// Apply a scale to the current matrix
// FT81x Series Programmers Guide Section 5.49
ft_void_t FT813::Scale(ft_int32_t sx, ft_int32_t sy)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_SCALE);
    SendCmd(sx);
    SendCmd(sy);
    EndFunc();
}

// Apply a rotation to the current matrix
// FT81x Series Programmers Guide Section 5.50
ft_void_t FT813::Rotate(ft_int32_t a)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_ROTATE);
    SendCmd(a);
    EndFunc();
}

// Apply a translation to the current matrix
// FT81x Series Programmers Guide Section 5.51
ft_void_t FT813::Translate(ft_int32_t tx, ft_int32_t ty)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_TRANSLATE);
    SendCmd(tx);
    SendCmd(ty);
    EndFunc();
}

// Execute the touch screen calibration routine
// FT81x Series Programmers Guide Section 5.52
// * This business about "result" in the manual really seems to be simply leftover cruft of no purpose - send zero
ft_void_t FT813::Calibrate(ft_uint32_t result)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_CALIBRATE);
    SendCmd(result);
    EndFunc();
    WaitCmdfifo_empty();
}

// Rotate the screen
// FT81x Series Programmers Guide Section 5.53
ft_void_t FT813::SetRotate(ft_uint32_t r)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_SETROTATE);
    SendCmd(r);
    EndFunc();
}

// Start an animated spinner
// FT81x Series Programmers Guide Section 5.54
ft_void_t FT813::Spinner(ft_int16_t x, ft_int16_t y, ft_uint16_t style, ft_uint16_t scale)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_SPINNER);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)scale<<16)|style));
    EndFunc();
}

// Start an animated screensaver
// FT81x Series Programmers Guide Section 5.55
ft_void_t FT813::ScreenSaver()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_SCREENSAVER);
    EndFunc();
}

// Start a continuous sketch update
// FT81x Series Programmers Guide Section 5.56
ft_void_t FT813::Sketch(ft_int16_t x, ft_int16_t y, ft_uint16_t w, ft_uint16_t h, ft_uint32_t ptr, ft_uint16_t format)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_SKETCH);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd(ptr);
    SendCmd(format);
    EndFunc();
}

// Stop any of spinner, screensaver or sketch
// FT81x Series Programmers Guide Section 5.57
ft_void_t FT813::Stop()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_STOP);
    EndFunc();
}

// Set up a custom font
// FT81x Series Programmers Guide Section 5.58
ft_void_t FT813::SetFont(ft_uint32_t font, ft_uint32_t ptr)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_SETFONT);
    SendCmd(font);
    SendCmd(ptr);
    EndFunc();
}

// Set up a custom font
// FT81x Series Programmers Guide Section 5.59
ft_void_t FT813::SetFont2(ft_uint32_t font, ft_uint32_t ptr, ft_uint32_t firstchar)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_SETFONT2);
    SendCmd(font);
    SendCmd(ptr);
    SendCmd(firstchar);
    EndFunc();
}

// Set the scratch bitmap for widget use
// FT81x Series Programmers Guide Section 5.60
ft_void_t FT813::SetScratch(ft_uint32_t handle)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_SETSCRATCH);
    SendCmd(handle);
    EndFunc();
}

// Load a ROM font into bitmap handle
// FT81x Series Programmers Guide Section 5.61
ft_void_t FT813::RomFont(ft_uint32_t rom_slot, ft_uint32_t font)
{
    StartFunc(FT_CMD_SIZE*3);
    SendCmd(CMD_ROMFONT);
    SendCmd(rom_slot);
    SendCmd(font);
    EndFunc();
}

// Track touches for a graphics object - Make Track (for a slider)
// FT81x Series Programmers Guide Section 5.62
// tag refers to the tag # previously assigned to the object that this track is tracking.
ft_void_t FT813::Track(ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h, ft_int16_t tag)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_TRACK);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|w));
    SendCmd(tag);
    EndFunc();
}

// Take a snapshot of the current screen
// FT81x Series Programmers Guide Section 5.63
ft_void_t FT813::Snapshot(ft_uint32_t ptr)
{
    StartFunc(FT_CMD_SIZE*2);
    SendCmd(CMD_SNAPSHOT);
    SendCmd(ptr);
    EndFunc();
}

// Take a snapshot of part of the current screen
// FT81x Series Programmers Guide Section 5.64
ft_void_t FT813::Snapshot2(ft_uint32_t fmt, ft_uint32_t ptr, ft_int16_t x, ft_int16_t y, ft_int16_t w, ft_int16_t h)
{
    StartFunc(FT_CMD_SIZE*5);
    SendCmd(CMD_SNAPSHOT2);
    SendCmd(fmt);
    SendCmd(ptr);
    SendCmd((((ft_uint32_t)y<<16)|(x & 0xffff)));
    SendCmd((((ft_uint32_t)h<<16)|(w & 0xffff)));
    EndFunc();
}

// Set up display list for bitmap - Cmd_SetBitmap - generate DL commands for bitmap parms
// FT81x Series Programmers Guide Section 5.65
ft_void_t FT813::SetBitmap(ft_int32_t addr, ft_int16_t fmt, ft_uint16_t width, ft_uint16_t height)
{
    StartFunc(FT_CMD_SIZE*4);
    SendCmd(CMD_SETBITMAP);
    SendCmd(addr);
    SendCmd((((ft_uint32_t)width<<16)|(fmt & 0xffff)));
    SendCmd((((ft_uint32_t)0<<16)|(height & 0xffff)));
    EndFunc();
}

// Play FTDI logo animation
// FT81x Series Programmers Guide Section 5.66
ft_void_t FT813::Logo()
{
    StartFunc(FT_CMD_SIZE);
    SendCmd(CMD_LOGO);
    EndFunc();
}

// ******************** Miscellaneous Operation CoProcessor Command Functions ******************************

// ?
// FT81x Series Programmers Guide Section ?
ft_void_t FT813::TouchTransform(ft_int32_t x0, ft_int32_t y0, ft_int32_t x1, ft_int32_t y1, ft_int32_t x2, ft_int32_t y2, ft_int32_t tx0, ft_int32_t ty0, ft_int32_t tx1, ft_int32_t ty1, ft_int32_t tx2, ft_int32_t ty2, ft_uint16_t result)
{
    StartFunc(FT_CMD_SIZE*6*2+FT_CMD_SIZE*2);
    SendCmd(CMD_TOUCH_TRANSFORM);
    SendCmd(x0);
    SendCmd(y0);
    SendCmd(x1);
    SendCmd(y1);
    SendCmd(x2);
    SendCmd(y2);
    SendCmd(tx0);
    SendCmd(ty0);
    SendCmd(tx1);
    SendCmd(ty1);
    SendCmd(tx2);
    SendCmd(ty2);
    SendCmd(result);
    EndFunc();
}

// ?
// FT81x Series Programmers Guide Section ?
ft_void_t FT813::BitmapTransform(ft_int32_t x0, ft_int32_t y0, ft_int32_t x1, ft_int32_t y1, ft_int32_t x2, ft_int32_t y2, ft_int32_t tx0, ft_int32_t ty0, ft_int32_t tx1, ft_int32_t ty1, ft_int32_t tx2, ft_int32_t ty2, ft_uint16_t result)
{
    StartFunc(FT_CMD_SIZE*6*2+FT_CMD_SIZE*2);
    SendCmd(CMD_BITMAP_TRANSFORM);
    SendCmd(x0);
    SendCmd(y0);
    SendCmd(x1);
    SendCmd(y1);
    SendCmd(x2);
    SendCmd(y2);
    SendCmd(tx0);
    SendCmd(ty0);
    SendCmd(tx1);
    SendCmd(ty1);
    SendCmd(tx2);
    SendCmd(ty2);
    SendCmd(result);
    EndFunc();
}

ft_void_t FT813::DL(ft_uint32_t cmd)
{
    WrCmd32(cmd);
    /* Increment the command index */
    CmdBuffer_Index += FT_CMD_SIZE;
}

ft_void_t FT813::WrDlCmd_Buffer(ft_uint32_t cmd)
{
    Wr32((RAM_DL+DlBuffer_Index), cmd);
    /* Increment the command index */
    DlBuffer_Index += FT_CMD_SIZE;
}

ft_void_t FT813::Flush_DL_Buffer()
{
    DlBuffer_Index = 0;
}

ft_void_t FT813::Flush_Co_Buffer()
{
    CmdBuffer_Index = 0;
}

/* API to check the status of previous DLSWAP and perform DLSWAP of new DL */
/* Check for the status of previous DLSWAP and if still not done wait for few ms and check again */
ft_void_t FT813::DLSwap(ft_uint8_t DL_Swap_Type)
{
    ft_uint8_t Swap_Type = DLSWAP_FRAME, Swap_Done = DLSWAP_FRAME;

    if(DL_Swap_Type == DLSWAP_LINE) {
        Swap_Type = DLSWAP_LINE;
    }

    /* Perform a new DL swap */
    Wr8(REG_DLSWAP, Swap_Type);

    /* Wait till the swap is done */
    while(Swap_Done) {
        Swap_Done = Rd8(REG_DLSWAP);

        if(DLSWAP_DONE != Swap_Done) {
            Sleep(10);//wait for 10ms
        }
    }
}

/* Nothing beyond this */
