/**
 * =============================================================================
 * Firmware updater (Version 0.0.2)
 * =============================================================================
 * Copyright (c) 2010 Shinichiro Nakamura (CuBeatSystems)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * =============================================================================
 */

#ifndef FIRMWARE_UPDATER_H
#define FIRMWARE_UPDATER_H

#include "mbed.h"
#include "HTTPClient.h"
#include "LocalFileSystem.h"
#include <string>

/**
 * @code
 * #include "mbed.h"
 * #include "FirmwareUpdater.h"
 * #include "EthernetNetIf.h"
 *
 * EthernetNetIf eth;
 * FirmwareUpdater fwup("http://mbed.org/media/uploads/shintamainjp/", "firm", true);
 *
 * // There are 2 files for the firmware.
 * //  1. firm.txt : firmware version file.
 * //  2. firm.bin : firmware binary file.
 *
 * int main() {
 *     eth.setup();
 *     if (fwup.exist() == 0) {
 *         printf("Found a new firmware.\n");
 *         if (fwup.execute() == 0) {
 *             printf("Update succeed.\n");
 *             printf("Resetting this system...\n\n\n\n\n");
 *             fwup.reset();
 *         } else {
 *             printf("Update failed!\n");
 *         }
 *     }
 * }
 * @endcode
 */
class FirmwareUpdater {
public:
    /**
     * Create.
     *
     * @param url URL for firmware. Do not include a target file name.
     * @param name An application name. Do not include a extention.
     * @param log True if logging.
     */
     const std::string name;
    FirmwareUpdater(std::string url, std::string name, bool log = false);

    /**
     * Dispose.
     */
    ~FirmwareUpdater();

    /**
     * Get a URL.
     *
     * @return URL.
     */
    const std::string getURL() const;

    /**
     * Get a name.
     *
     * @return name.
     */
    const std::string getName() const;

    /**
     * Checking a new firmware.
     * Compare versions of the software between local storage on mbed and on webserver.
     *
     * @return Return 0 if a new firmware exists.
     */
    int exist();
    int readVersionFromFile(const char *filename);
    /**
     * Execute update.
     *
     * @return Return 0 if it succeed.
     */
    int execute();

    /**
     * Reset system.
     */
    void reset();

private:
    static const int MAXNAMELEN = 8;
    static const std::string EXT_BIN;
    static const std::string EXT_BINTMP;
    static const std::string EXT_TXT;
    static const std::string EXT_TXTTMP;
    const std::string url;
    
    const bool log;
    LocalFileSystem local;
    HTTPClient client;

    /**
     * Fetch a file.
     *
     * @param src_url URL of a source file.
     * @param local_file Local file name.
     *
     * @return Return 0 if it succeed.
     */
    int fetch(std::string src_url, std::string local_file);

    /**
     * Copy a file.
     *
     * @param local_file1 Source file.
     * @param local_file2 Destination file.
     *
     * @return Return 0 if it succeed.
     */
    int copy(std::string local_file1, std::string local_file2);

    /**
     * Output a message to a log file.
     *
     * @param format ...
     */
    void LOG(const char* format, ...);

    /**
     * Cleanup all bin files.
     */
    int cleanupAllBinFiles(void);
    
    /**
     * Create a new binary file name.
     *
     * @param ver Version.
     * @param buf A pointer to a buffer.
     * @param siz A size of the buffer.
     *
     * @return Return 0 if it succeed.
     */
    int createNewBinName(const int ver, char *buf, size_t siz);

    /**
     * Read a version from a file.
     *
     * @param filename file name.
     * @return A version.
     */
    //int readVersionFromFile(const char *filename);

    /**
     * Read a version from a URL.
     *
     * @param url URL.
     * @return A version.
     */
    int readVersionFromURL(const char *url);
};

#endif
