/* WIZnet IoT Shield Cat.M1 Sample code for Arm MBED
 * Copyright (c) 2019 WIZnet Co., Ltd.
 * SPDX-License-Identifier: Apache-2.0
 */
 
 /* 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include <string>
#include "mbed.h"

#define RET_OK                      1
#define RET_NOK                     -1
#define DEBUG_ENABLE                1
#define DEBUG_DISABLE               0
#define ON                          1
#define OFF                         0

#define MAX_BUF_SIZE                1024

#define AMM592_APN_PROTOCOL_IPv4      1
#define AMM592_APN_PROTOCOL_IPv6      2
#define AMM592_DEFAULT_TIMEOUT        1000
#define AMM592_CONNECT_TIMEOUT        15000
#define AMM592_SEND_TIMEOUT           500
#define AMM592_RECV_TIMEOUT           500

#define AMM592_APN_PROTOCOL           AMM592_APN_PROTOCOL_IPv6
#define AMM592_DEFAULT_BAUD_RATE      115200
#define AMM592_PARSER_DELIMITER       "\r\n"

#define CATM1_APN_SKT               "lte-internet.sktelecom.com"

#define CATM1_DEVICE_NAME_AMM592      "AMM592"
#define DEVNAME                     CATM1_DEVICE_NAME_AMM592

#define devlog(f_, ...)             if(CATM1_DEVICE_DEBUG == DEBUG_ENABLE) { pc.printf("\r\n[%s] ", DEVNAME);  pc.printf((f_), ##__VA_ARGS__); }
#define myprintf(f_, ...)           {pc.printf("\r\n[MAIN] ");  pc.printf((f_), ##__VA_ARGS__);}

/* Pin configuraiton */
// Cat.M1
#define MBED_CONF_IOTSHIELD_CATM1_TX                D8
#define MBED_CONF_IOTSHIELD_CATM1_RX                D2
#define MBED_CONF_IOTSHIELD_CATM1_RESET             D7
#define MBED_CONF_IOTSHIELD_CATM1_PWRKEY            D9

// Sensors
#define MBED_CONF_IOTSHIELD_SENSOR_CDS              A0
#define MBED_CONF_IOTSHIELD_SENSOR_TEMP             A1

/* Debug message settings */
#define AMM592_PARSER_DEBUG           DEBUG_DISABLE
#define CATM1_DEVICE_DEBUG          DEBUG_ENABLE 

// Functions: Module Status
void waitCatM1Ready(void);
int8_t setEchoStatus_AMM592(int onoff);
int8_t getUsimStatus_AMM592(void);
int8_t getNetworkStatus_AMM592(void);
int8_t checknSetApn_AMM592(const char * apn);

// Functions: DNS
int8_t getIpAddressByName_AMM592(const char * name);

// Functions: PDP context
int8_t getIpAddress_AMM592(char * ipstr);

// Functions: TCP/UDP Socket service
int8_t sockOpenConnect_AMM592(int socket_id, int socket_type, int data_type, const char * addr, int port);
int8_t sockClose_AMM592(void);
int8_t sendData_AMM592(int socket_id, int data_type, char * data, int len);
int8_t recvData_AMM592(int socket_id, int data_type, int * len, char * data);

Serial pc(USBTX, USBRX); // tx, rx

UARTSerial *_serial;
ATCmdParser *_parser;

DigitalOut _RESET_AMM592(MBED_CONF_IOTSHIELD_CATM1_RESET);
DigitalOut _PWRKEY_AMM592(MBED_CONF_IOTSHIELD_CATM1_PWRKEY);

// Destination (Remote Host)
// IP address and Port number
char dest_ip[] = "222.xx.xxx.xxx;
int  dest_port = 5000;

void serialPcInit(void)
{
    pc.baud(115200);
    pc.format(8, Serial::None, 1);
}

void serialDeviceInit(PinName tx, PinName rx, int baudrate) 
{        
    _serial = new UARTSerial(tx, rx, baudrate);    
}

void serialAtParserInit(const char *delimiter, bool debug_en)
{
    _parser = new ATCmdParser(_serial);    
    _parser->debug_on(debug_en);
    _parser->set_delimiter(delimiter);    
    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);
}

void catm1DeviceInit(void)
{
    serialDeviceInit(   MBED_CONF_IOTSHIELD_CATM1_TX, 
                        MBED_CONF_IOTSHIELD_CATM1_RX, 
                        AMM592_DEFAULT_BAUD_RATE);
                        
    serialAtParserInit( AMM592_PARSER_DELIMITER, 
                        AMM592_PARSER_DEBUG);
}

void catm1DeviceReset_AMM592(void)
{
    _RESET_AMM592 = 1;
    _PWRKEY_AMM592 = 1;
    wait_ms(300);
    
    _RESET_AMM592 = 0;
    _PWRKEY_AMM592 = 0;
    wait_ms(400);
    
    _RESET_AMM592 = 1;    
    wait_ms(1000);
}

// ----------------------------------------------------------------
// Main routine
// ----------------------------------------------------------------

int main()
{
    serialPcInit();    
    catm1DeviceInit();
    
    myprintf("Waiting for Cat.M1 Module Ready...\r\n");
    
    catm1DeviceReset_AMM592();
    
    waitCatM1Ready();
    
    wait_ms(5000);
            
    myprintf("System Init Complete\r\n");
    
    myprintf("WIZnet IoT Shield for Arm MBED");
    myprintf("LTE Cat.M1 Version");
    myprintf("=================================================");    
    myprintf(">> Target Board: WIoT-AM01 (AM Telecom AMM592)");
    myprintf(">> Sample Code: TCP Client Send & Recv");
    myprintf("=================================================\r\n");
    
    setEchoStatus_AMM592(OFF);
   
    getUsimStatus_AMM592();
    
    getNetworkStatus_AMM592();
    
    checknSetApn_AMM592(CATM1_APN_SKT);

   
    // DNS Sample
    const char dest_domain[] = "www.google.com";
    
    if(getIpAddressByName_AMM592(dest_domain)) {
        myprintf("DNSQUERY successful\r\n");
    } else {
        myprintf("DNS: failed\r\n");
    }

    
    //Convert IP to HEX
    //64:ff9b:: is standard SK prefix
    int i=1;
    char * pch;
    char _dest_ip[20]="64:ff9b::";
    char temp[4];
    pch = strtok(dest_ip, ".");
    while(pch!=NULL) {
        if (i!=2) {
            sprintf(temp, "%x\0", atoi(pch));
            strcat(_dest_ip, temp);
            i++;
        }
        else {
            sprintf(temp, "%x:\0", atoi(pch));
            strcat(_dest_ip, temp);
            i++;
        }
        pch = strtok(NULL, ".");
    }
    
    // TCP Client: Send and Receive
    myprintf("TCP Client Start - Connect to Remote Host \r\n");
    
    if(sockOpenConnect_AMM592(0,0,0, _dest_ip, dest_port) == RET_OK) {
        myprintf("sockOpenConnect: success\r\n");
        
        char sendbuf[] = "Hello Cat.M1";         
        if(sendData_AMM592(0, 0, sendbuf, strlen(sendbuf))) {
            myprintf("dataSend [%d] %s\r\n", strlen(sendbuf), sendbuf);
        }
        
    } else {
        myprintf("sockOpen: failed\r\n");
        
        if(sockClose_AMM592() == RET_OK) {
            myprintf("sockClose: success\r\n");
        }
    }    
    
    _parser->debug_on(DEBUG_DISABLE);

    while(1)
    {
            char recvbuf[100] = {0, };
            int recvlen = 0;
            
            if(recvData_AMM592(0, 0, &recvlen, recvbuf) == RET_OK) {
                myprintf("dataRecv [%d] %s\r\n", recvlen, recvbuf);
                
                // Loopback: Resend received data to TCP server
                //sendData_AMM592(0, 0, recvbuf, recvlen);
                //myprintf("dataSend [%d] %s\r\n", recvlen, recvbuf);
                
                char * ptr = strstr(recvbuf, "exit");
                if(ptr != 0) break;
            }       
    }

    if(sockClose_AMM592() == RET_OK) {
        myprintf("sockClose: success\r\n");
    }
 
    
}

// ----------------------------------------------------------------
// Functions: Cat.M1 Status
// ----------------------------------------------------------------

void waitCatM1Ready(void)
{
    while(1) 
    {   
        if(_parser->recv("@NOTI:34,AMT_BOOT_ALERT")&& _parser->recv("@NETSTI:3")) 
        {
            myprintf("AMM592 ready\r\n");
            return ;
        }
        else if(_parser->send("AT") && _parser->recv("OK"))
        {
            myprintf("AMM592 already available\r\n");
            return ;
        }        
    }        
}

int8_t setEchoStatus_AMM592(int onoff)
{
    int8_t ret = RET_NOK;
    char _buf[10];        
    
    sprintf((char *)_buf, "ATE%d", onoff);    
    
    if(_parser->send(_buf) && _parser->recv("OK")) {        
        devlog("Turn Echo %s success\r\n", onoff?"ON":"OFF");
        ret = RET_OK;
    } else { 
        devlog("Turn Echo %s failed\r\n", onoff?"ON":"OFF");
    }    
    return ret;
}
 
int8_t getUsimStatus_AMM592(void)
{
    int8_t ret = RET_NOK;
    
    _parser->send("AT+CPIN?");    
    if(_parser->recv("+CPIN: READY") && _parser->recv("OK")) {
        devlog("USIM Status: READY\r\n");
        ret = RET_OK;
    } else { 
        devlog("Retrieving USIM Status failed\r\n");        
    }
    return ret;
}

int8_t getNetworkStatus_AMM592(void)
{
  int8_t ret = RET_NOK;    
    bool cereg = false, nsi = false;
    int n, stat;

    if(_parser->send("AT+CEREG?") && _parser->recv("+CEREG: %d, %d", &n, &stat) && _parser->recv("OK")){
        if ((n==0 || n==1) && stat==1){
            cereg = true;
        }
    }
    
    if(_parser->send("AT@NSI") && _parser->recv("@NSI:%d,\"IN SRV\"") && _parser->recv("OK")) {
        devlog("Network Status: attached\r\n");
        nsi = true;
    } else if (_parser->send("AT@NSI") && _parser->recv("@NSI: \"LIMITED\"") && _parser->recv("OK")) {
        devlog("Network Status: limited\r\n");   
    } else { 
        devlog("Network Status: Error\r\n");        
    }
    
    if (cereg && nsi){
        ret = RET_OK;
    }
    return ret;
}

int8_t checknSetApn_AMM592(const char * apn) // Configure Parameters of a TCP/IP Context
{       
   bool cgdccont = false, cgatt = false;
    int8_t ret = RET_NOK;
    
    devlog("Checking APN...\r\n");
    //APN setting
    if (_parser->send("AT+CGDCONT=1,\"IPV4V6\",\"%s\"", apn) && _parser->recv("OK")){
        cgdccont = true;
    }
    //Attach setting
    if (_parser->send("AT+CGATT?") && _parser->recv("+CGATT: 1") &&_parser->recv("OK")){
        cgatt = true;
    }

    if (cgdccont && cgatt){
        ret = RET_OK;
    }

    devlog("APN Check Done\r\n");
        
    return ret;
}

int8_t getIpAddressByName_AMM592(const char * name)
{
    int8_t ret = RET_NOK;

    if( _parser->send("AT@DNSQ6=%s", name)&& _parser->recv("OK"))
        ret = RET_OK;       
    
    return ret;           
}

// ----------------------------------------------------------------
// Functions: TCP/UDP socket service
// ----------------------------------------------------------------
int8_t sockOpenConnect_AMM592(int socket_id, int socket_type, int data_type, const char * addr, int port)
{
    int8_t ret = RET_NOK;  
    int err = 0;
    int id = 0;
    
    bool done = false;
    Timer t;
    
    _parser->set_timeout(AMM592_CONNECT_TIMEOUT);
    
    if((socket_type != 0) && (socket_type != 1)) {        
        return RET_NOK;
    }

    t.start();
    //devlog("AT@SOCKOPEN6=%d,%d,%d,%s,%d", socket_id, socket_type, data_type, addr, port);
    _parser->send("AT@SOCKOPEN6=%d,%d,%d,%s,%d", socket_id, socket_type, data_type, addr, port);
    do {        
        done = (_parser->recv("@SOCKOPEN6:%d,%d", &id, &err) && (err == 0)); 
    } while(!done && t.read_ms() < AMM592_CONNECT_TIMEOUT);
    
   if(done) ret = RET_OK;

    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);    
    _parser->flush();
    
    return ret;
}

int8_t sockClose_AMM592(void)
{
    int8_t ret = RET_NOK;
    int id = 0;
    
    _parser->set_timeout(AMM592_CONNECT_TIMEOUT);
    
    if(_parser->send("AT@SOCKCLOSE6=%d", id) && _parser->recv("OK")) {
        ret = RET_OK;        
    }
    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);
    
    return ret;
}
int8_t sendData_AMM592(int socket_id, int data_type, char * data, int len)
{
    int8_t ret = RET_NOK;
    int id = 0;
    bool done = false;
    int err = 0;
    int lencom=0;
    Timer t;
    
    _parser->set_timeout(AMM592_SEND_TIMEOUT);
    t.start();
    
    _parser->send("AT@SOCKSND6=%d,%d,\"%s\"", socket_id, data_type, data);
 
   do {        
        done = (_parser->recv("@SOCKSND6:%d,%d,%d", &id, &lencom, &err)  && (err == 0)); 
        } while(!done && t.read_ms() < AMM592_CONNECT_TIMEOUT);

    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);
    
    return ret;
}

int8_t recvData_AMM592(int socket_id, int data_type, int * len, char * data)
{
    int8_t ret = RET_NOK;
    int recvCount = 0;
    int socketid, datatype;
    
     _parser->set_timeout(AMM592_RECV_TIMEOUT);   
    
    if( _parser->recv("@SOCKRCV6=%d,%d,%d,",&socketid, &datatype, &recvCount) ) {
        if(recvCount > 0) {
            _parser->read(data, recvCount);
            ret = RET_OK;
         }        
    }
    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);    
    _parser->flush();
    
    *len = recvCount;
        
    return ret;
}