/* WIZnet IoT Shield Cat.M1 Sample code for Arm MBED
 * Copyright (c) 2019 WIZnet Co., Ltd.
 * SPDX-License-Identifier: Apache-2.0
 */
 
 /* 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#include <string>
#include "mbed.h"

#define RET_OK                      1
#define RET_NOK                     -1
#define DEBUG_ENABLE                1
#define DEBUG_DISABLE               0
#define ON                          1
#define OFF                         0

#define MAX_BUF_SIZE                1024
#define AMM592_DEFAULT_TIMEOUT        1000
#define AM592_APN_PROTOCOL_IPv4      1
#define AM592_APN_PROTOCOL_IPv6      2
#define AM592_DEFAULT_TIMEOUT        1000
#define AM592_CONNECT_TIMEOUT        15000
#define AM592_SEND_TIMEOUT           500
#define AM592_RECV_TIMEOUT           500

#define AM592_APN_PROTOCOL           AM592_APN_PROTOCOL_IPv6
#define AM592_DEFAULT_BAUD_RATE      115200
#define AM592_PARSER_DELIMITER       "\r\n"

#define CATM1_APN_SKT               "lte-internet.sktelecom.com"

#define CATM1_DEVICE_NAME_AM592      "AM592"
#define DEVNAME                     CATM1_DEVICE_NAME_AM592

#define devlog(f_, ...)             if(CATM1_DEVICE_DEBUG == DEBUG_ENABLE) { pc.printf("\r\n[%s] ", DEVNAME);  pc.printf((f_), ##__VA_ARGS__); }
#define myprintf(f_, ...)           {pc.printf("\r\n[MAIN] ");  pc.printf((f_), ##__VA_ARGS__);}

/* Pin configuraiton */
// Cat.M1
#define MBED_CONF_IOTSHIELD_CATM1_TX                D8
#define MBED_CONF_IOTSHIELD_CATM1_RX                D2
#define MBED_CONF_IOTSHIELD_CATM1_RESET             D7
#define MBED_CONF_IOTSHIELD_CATM1_PWRKEY            D9

// Sensors
#define MBED_CONF_IOTSHIELD_SENSOR_CDS              A0
#define MBED_CONF_IOTSHIELD_SENSOR_TEMP             A1

/* Debug message settings */
#define AM592_PARSER_DEBUG           DEBUG_DISABLE
#define CATM1_DEVICE_DEBUG          DEBUG_ENABLE 

// Functions: Module Status
void waitCatM1Ready(void);
int8_t setEchoStatus_AM592(int onoff);
int8_t getUsimStatus_AM592(void);
int8_t getNetworkStatus_AM592(void);
int8_t checknSetApn_AM592(const char * apn);

// Functions: Ping test
void printPingToHost_AM592(char * host, int pingnum);

// Example: Ping destinations
char ping_dest_1st[] = "2001:4860:4860::8888";
char ping_dest_2nd[] = "ipv6.google.com";

Serial pc(USBTX, USBRX);    // USB debug

UARTSerial *_serial;        // Cat.M1 module    
ATCmdParser *_parser;

DigitalOut _RESET_AM592(MBED_CONF_IOTSHIELD_CATM1_RESET);
DigitalOut _PWRKEY_AM592(MBED_CONF_IOTSHIELD_CATM1_PWRKEY);

void serialPcInit(void)
{
    pc.baud(115200);
    pc.format(8, Serial::None, 1);
}

void serialDeviceInit(PinName tx, PinName rx, int baudrate) 
{        
    _serial = new UARTSerial(tx, rx, baudrate);    
}

void serialAtParserInit(const char *delimiter, bool debug_en)
{
    _parser = new ATCmdParser(_serial);    
    _parser->debug_on(debug_en);
    _parser->set_delimiter(delimiter);    
    _parser->set_timeout(AM592_DEFAULT_TIMEOUT);
}

void catm1DeviceInit(void)
{
    serialDeviceInit(   MBED_CONF_IOTSHIELD_CATM1_TX, 
                        MBED_CONF_IOTSHIELD_CATM1_RX, 
                        AM592_DEFAULT_BAUD_RATE);
                        
    serialAtParserInit( AM592_PARSER_DELIMITER, 
                        AM592_PARSER_DEBUG);
}

void catm1DeviceReset_AM592(void)
{
    _RESET_AM592 = 1;
    _PWRKEY_AM592 = 1;
    wait_ms(300);
    
    _RESET_AM592 = 0;
    _PWRKEY_AM592 = 0;
    wait_ms(400);
    
    _RESET_AM592 = 1;    
    wait_ms(1000);
}

// ----------------------------------------------------------------
// Main routine
// ----------------------------------------------------------------

int main()
{
    serialPcInit();    
    catm1DeviceInit();
    
    myprintf("Waiting for Cat.M1 Module Ready...\r\n");
    
    catm1DeviceReset_AM592();
    
    wait_ms(5000);
    
    waitCatM1Ready();
            
    myprintf("System Init Complete\r\n");
        
    myprintf("WIZnet IoT Shield for Arm MBED");
    myprintf("LTE Cat.M1 Version");
    myprintf("=================================================");
    myprintf(">> Target Board: WIoT-AM01 (AM Telecom AMM592)");
    myprintf(">> Sample Code: Ping Test");
    myprintf("=================================================\r\n");
    
    setEchoStatus_AM592(0);
   
    getUsimStatus_AM592();
    
    getNetworkStatus_AM592();
    
    checknSetApn_AM592(CATM1_APN_SKT);

    
    myprintf("[Ping] Host: %s\r\n", ping_dest_1st);
    printPingToHost_AM592(ping_dest_1st, 4);    
    
    wait_ms(2000);
    
    myprintf("[Ping] Host: %s\r\n", ping_dest_2nd);
    printPingToHost_AM592(ping_dest_2nd, 4);

}

// ----------------------------------------------------------------
// Functions: Cat.M1 Status
// ----------------------------------------------------------------

void waitCatM1Ready(void)
{
    while(1) 
    {   
        if(_parser->recv("@NOTI:34,AMT_BOOT_ALERT") && _parser->recv("@NETSTI:3") ) 
        {
            myprintf("AM592 ready\r\n");
            return ;
        }
        else if(_parser->send("AT") && _parser->recv("OK"))
        {
            myprintf("AM592 already available\r\n");
            return ;
        }        
    }        
}

int8_t setEchoStatus_AM592(int onoff)
{
    int8_t ret = RET_NOK;
    char _buf[10];        
    
    sprintf((char *)_buf, "ATE%d", onoff);    
    
    if(_parser->send(_buf) && _parser->recv("OK")) {        
        devlog("Turn Echo %s success\r\n", onoff?"ON":"OFF");
        ret = RET_OK;
    } else { 
        devlog("Turn Echo %s failed\r\n", onoff?"ON":"OFF");
    }    
    return ret;
}
 
int8_t getUsimStatus_AM592(void)
{
    int8_t ret = RET_NOK;
    
    _parser->send("AT+CPIN?");    
    if(_parser->recv("+CPIN: READY") && _parser->recv("OK")) {
        devlog("USIM Status: READY\r\n");
        ret = RET_OK;
    } else { 
        devlog("Retrieving USIM Status failed\r\n");        
    }
    return ret;
}

int8_t getNetworkStatus_AM592(void)
{
    int8_t ret = RET_NOK;    
    bool cereg = false, nsi = false;
    int n, stat;

    if(_parser->send("AT+CEREG?") && _parser->recv("+CEREG: %d, %d", &n, &stat) && _parser->recv("OK")){
        if ((n==0 || n==1) && stat==1){
            cereg = true;
        }
    }
    
    if(_parser->send("AT@NSI") && _parser->recv("@NSI:%d,\"IN SRV\"") && _parser->recv("OK")) {
        devlog("Network Status: attached\r\n");
        nsi = true;
    } else if (_parser->send("AT@NSI") && _parser->recv("@NSI: \"LIMITED\"") && _parser->recv("OK")) {
        devlog("Network Status: limited\r\n");
        nsi = true;    
    } else { 
        devlog("Network Status: Error\r\n");        
    }
    
    if (cereg && nsi){
        ret = RET_OK;
    }
    return ret;
}

int8_t checknSetApn_AM592(const char * apn) // Configure Parameters of a TCP/IP Context
{       
    bool cgdccont = false, cgatt = false;
    int8_t ret = RET_NOK;
    
    devlog("Checking APN...\r\n");
    //APN setting
    if (_parser->send("AT+CGDCONT=1,\"IPV4V6\",\"%s\"", apn) && _parser->recv("OK")){
        cgdccont = true;
    }
    //Attach setting
    if (_parser->send("AT+CGATT?") && _parser->recv("+CGATT: 1") &&_parser->recv("OK")){
        cgatt = true;
    }

    if (cgdccont && cgatt){
        ret = RET_OK;
    }

    devlog("APN Check Done\r\n");
        
    return ret;
}

// ----------------------------------------------------------------
// Functions: Cat.M1 Ping test
// ----------------------------------------------------------------

void printPingToHost_AM592(char * host, int pingnum)
{   
    Timer t;
    
    if((pingnum < 1) || (pingnum > 10)) {
        devlog("The maximum number of sending Ping request range is 1-10, and the default value is 4\r\n");
        return;
    }
   
    _parser->set_timeout((1000 * pingnum) + 2000);
   
    if(_parser->send("AT@PING6=\"%s -c %d\"", host, pingnum) && _parser->recv("OK"))
    {
        t.start();
        while (t.read_ms()<((1000*pingnum)+2000))
        {
            pc.printf("%c", _parser->getc());
        }
        t.stop(); t.reset();   
    }
    
    _parser->set_timeout(AM592_DEFAULT_TIMEOUT);
    _parser->flush();
}