/* WIZnet IoT Shield Cat.M1 Sample code for Arm MBED
 * Copyright (c) 2019 WIZnet Co., Ltd.
 * SPDX-License-Identifier: Apache-2.0
 */
 
 /* 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */


#include <string>
#include "mbed.h"

#define RET_OK                      1
#define RET_NOK                     -1
#define DEBUG_ENABLE                1
#define DEBUG_DISABLE               0
#define ON                          1
#define OFF                         0

#define MAX_BUF_SIZE                1024

#define AMM592_APN_PROTOCOL_IPv4      1
#define AMM592_APN_PROTOCOL_IPv6      2
#define AMM592_DEFAULT_TIMEOUT        1000
#define AMM592_CONNECT_TIMEOUT        15000
#define AMM592_SEND_TIMEOUT           500
#define AMM592_RECV_TIMEOUT           500

#define AMM592_APN_PROTOCOL           AMM592_APN_PROTOCOL_IPv6
#define AMM592_DEFAULT_BAUD_RATE      115200
#define AMM592_PARSER_DELIMITER       "\r\n"

#define CATM1_APN_SKT               "lte-internet.sktelecom.com"

#define CATM1_DEVICE_NAME_AMM592      "AMM592"
#define DEVNAME                     CATM1_DEVICE_NAME_AMM592

#define devlog(f_, ...)             if(CATM1_DEVICE_DEBUG == DEBUG_ENABLE) { pc.printf("\r\n[%s] ", DEVNAME);  pc.printf((f_), ##__VA_ARGS__); }
#define myprintf(f_, ...)           {pc.printf("\r\n[MAIN] ");  pc.printf((f_), ##__VA_ARGS__);}

/* Pin configuraiton */
// Cat.M1
#define MBED_CONF_IOTSHIELD_CATM1_TX                D8
#define MBED_CONF_IOTSHIELD_CATM1_RX                D2
#define MBED_CONF_IOTSHIELD_CATM1_RESET             D7
#define MBED_CONF_IOTSHIELD_CATM1_PWRKEY            D9

// Sensors
#define MBED_CONF_IOTSHIELD_SENSOR_CDS              A0
#define MBED_CONF_IOTSHIELD_SENSOR_TEMP             A1

/* Debug message settings */
#define AMM592_PARSER_DEBUG           DEBUG_DISABLE
#define CATM1_DEVICE_DEBUG          DEBUG_ENABLE 


// Functions: Module Status
void waitCatM1Ready(void);
int8_t setEchoStatus_AMM592(int onoff);
int8_t getUsimStatus_AMM592(void);
int8_t getNetworkStatus_AMM592(void);
int8_t checknSetApn_AMM592(const char * apn);

// Functions: PSM (Power Saving Mode)
int8_t setPsmActivate_AMM592(int Requested_Periodic_TAU, int Requested_Active_Time);
int8_t setPsmDeactivate_AMM592(void);
int8_t getPsmSetting_AMM592(bool * enable, int * Requested_Periodic_TAU, int * Requested_Active_Time);
int8_t getDeactPsmSetting_AMM592(bool * enable);
// Functions: Network time
int8_t getNetworkTimeLocal_AMM592(char * timestr);
void setFlagGettime(void);
void clearFlagGettime(void);


Serial pc(USBTX, USBRX);    // USB debug

UARTSerial *_serial;        // Cat.M1 module    
ATCmdParser *_parser;

DigitalOut _RESET_AMM592(MBED_CONF_IOTSHIELD_CATM1_RESET);
DigitalOut _PWRKEY_AMM592(MBED_CONF_IOTSHIELD_CATM1_PWRKEY);

Ticker flip;
bool flag_gettime = false;

void serialPcInit(void)
{
    pc.baud(115200);
    pc.format(8, Serial::None, 1);
}

void serialDeviceInit(PinName tx, PinName rx, int baudrate) 
{        
    _serial = new UARTSerial(tx, rx, baudrate);    
}

void serialAtParserInit(const char *delimiter, bool debug_en)
{
    _parser = new ATCmdParser(_serial);    
    _parser->debug_on(debug_en);
    _parser->set_delimiter(delimiter);    
    _parser->set_timeout(AMM592_DEFAULT_TIMEOUT);
}

void catm1DeviceInit(void)
{
    serialDeviceInit(   MBED_CONF_IOTSHIELD_CATM1_TX, 
                        MBED_CONF_IOTSHIELD_CATM1_RX, 
                        AMM592_DEFAULT_BAUD_RATE);
                        
    serialAtParserInit( AMM592_PARSER_DELIMITER, 
                        AMM592_PARSER_DEBUG);
}

void catm1DeviceReset_AMM592(void)
{
    _RESET_AMM592 = 1;
    _PWRKEY_AMM592 = 1;
    wait_ms(300);
    
    _RESET_AMM592 = 0;
    _PWRKEY_AMM592 = 0;
    wait_ms(400);
    
    _RESET_AMM592 = 1;    
    wait_ms(1000);
}


// ----------------------------------------------------------------
// Main routine
// ----------------------------------------------------------------

int main()
{
    serialPcInit();    
    catm1DeviceInit();
    
    myprintf("Waiting for Cat.M1 Module Ready...\r\n");
    
    catm1DeviceReset_AMM592();
    
    waitCatM1Ready();
    
    wait_ms(5000);
            
    myprintf("System Init Complete\r\n");
        
    myprintf("WIZnet IoT Shield for Arm MBED");
    myprintf("LTE Cat.M1 Version");
    myprintf("=================================================");
    myprintf(">> Target Board: WIoT-AM01 (AM Telecom AMM592)");
    myprintf(">> Sample Code: PSM (Power Saving Mode)");
    myprintf("=================================================\r\n");
    
    setEchoStatus_AMM592(0);
   
    getUsimStatus_AMM592();
    
    getNetworkStatus_AMM592();
    
    checknSetApn_AMM592(CATM1_APN_SKT);
    
    Timer t;
    float elapsed_time_sec = 0;       
    bool psm_en = false;
    int psm_tau = 0;
    int psm_active = 0;
    
    // PSM enable
  
   setPsmDeactivate_AMM592();

    if(getDeactPsmSetting_AMM592(&psm_en) == RET_OK) {
        if(psm_en != true) {
   
            if(setPsmActivate_AMM592(60,300) == RET_OK) {
                myprintf("Cat.M1 PSM enable, Device reboot");
                
                // Cat.M1 reboot
                catm1DeviceReset_AMM592();
                waitCatM1Ready();
            } else {
                 myprintf("Cat.M1 PSM enable failed");
            }
        } 
    }

    getPsmSetting_AMM592(&psm_en, &psm_tau, &psm_active);
    myprintf("PSM Config: %s", psm_en?"Enabled":"Disabled");

    // Timer event callback       
    flip.attach(callback(&setFlagGettime), 1.0);

    while(1)
    {
        if(flag_gettime) {
            char nettime[30] = {0, };
            if(getNetworkTimeLocal_AMM592(nettime) == RET_OK) {
                if(elapsed_time_sec > 0) {
                    t.stop();
                    myprintf("Cat.M1 Active, Sleep time: %.2fsec", elapsed_time_sec);                    
                    elapsed_time_sec = 0; 
                    t.reset();  
                }                
                myprintf("%s", nettime);
            } else {
                if(elapsed_time_sec == 0) {                    
                    t.start();         
                    myprintf("%s", "Sleep Start");               
                }
                elapsed_time_sec = t.read();
                myprintf("Cat.M1 PSM, %.2f", elapsed_time_sec);               
            }
            clearFlagGettime();
        }
    }
}


// ----------------------------------------------------------------
// Functions: Cat.M1 Status
// ----------------------------------------------------------------

void waitCatM1Ready(void)
{
    while(1) 
    {   
        if(_parser->recv("@NOTI:34,AMT_BOOT_ALERT") && _parser->recv("@NETSTI:3")) 
        {
            myprintf("AMM592 ready\r\n");
            return ;
        }
        else if(_parser->send("AT") && _parser->recv("OK"))
        {
            myprintf("AMM592 already available\r\n");
            return ;
        }        
    }        
}

int8_t setEchoStatus_AMM592(int onoff)
{
    int8_t ret = RET_NOK;
    char _buf[10];        
    
    sprintf((char *)_buf, "ATE%d", onoff);    
    
    if(_parser->send(_buf) && _parser->recv("OK")) {        
        devlog("Turn Echo %s success\r\n", onoff?"ON":"OFF");
        ret = RET_OK;
    } else { 
        devlog("Turn Echo %s failed\r\n", onoff?"ON":"OFF");
    }    
    return ret;
}
 
int8_t getUsimStatus_AMM592(void)
{
    int8_t ret = RET_NOK;
    
    _parser->send("AT+CPIN?");    
    if(_parser->recv("+CPIN: READY") && _parser->recv("OK")) {
        devlog("USIM Status: READY\r\n");
        ret = RET_OK;
    } else { 
        devlog("Retrieving USIM Status failed\r\n");        
    }
    return ret;
}

int8_t getNetworkStatus_AMM592(void)
{
   int8_t ret = RET_NOK;    
    bool cereg = false, nsi = false;
    int n, stat;

    if(_parser->send("AT+CEREG?") && _parser->recv("+CEREG: %d, %d", &n, &stat) && _parser->recv("OK")){
        if ((n==0 || n==1) && stat==1){
            cereg = true;
        }
    }
    
    if(_parser->send("AT@NSI") && _parser->recv("@NSI:%d,\"IN SRV\"") && _parser->recv("OK")) {
        devlog("Network Status: attached\r\n");
        nsi = true;
    } else if (_parser->send("AT@NSI") && _parser->recv("@NSI: \"LIMITED\"") && _parser->recv("OK")) {
        devlog("Network Status: limited\r\n");   
    } else { 
        devlog("Network Status: Error\r\n");        
    }
    
    if (cereg && nsi){
        ret = RET_OK;
    }
    return ret;
}

int8_t checknSetApn_AMM592(const char * apn) // Configure Parameters of a TCP/IP Context
{       
  bool cgdccont = false, cgatt = false;
    int8_t ret = RET_NOK;
    
    devlog("Checking APN...\r\n");
    //APN setting
    if (_parser->send("AT+CGDCONT=1,\"IPV4V6\",\"%s\"", apn) && _parser->recv("OK")){
        cgdccont = true;
    }
    //Attach setting
    if (_parser->send("AT+CGATT?") && _parser->recv("+CGATT: 1") &&_parser->recv("OK")){
        cgatt = true;
    }

    if (cgdccont && cgatt){
        ret = RET_OK;
    }

    devlog("APN Check Done\r\n");
        
    return ret;
}

// ----------------------------------------------------------------
// Functions: Cat.M1 PSM activate / deactivate
// ----------------------------------------------------------------

int8_t setPsmActivate_AMM592(int Requested_Periodic_TAU, int Requested_Active_Time)
{
    int8_t ret = RET_NOK;
    
    if (_parser->send("AT@PSMSET=1,%d,%d", Requested_Periodic_TAU, Requested_Active_Time)
        && _parser->recv("OK")) 
    {
        devlog("PSM activate success\r\n");
        
        ret = RET_OK;
    }
    return ret;
}

int8_t setPsmDeactivate_AMM592(void)
{
    int8_t ret = RET_NOK;
    
    if (_parser->send("AT@PSMSET=0") && _parser->recv("OK")) {
        devlog("PSM deactivate success\r\n");
    }
    //Reboot
    catm1DeviceReset_AMM592();
    waitCatM1Ready();
    
    return ret;
}

int8_t getPsmSetting_AMM592(bool * enable, int * Requested_Periodic_TAU, int * Requested_Active_Time)
{
    int8_t ret = RET_NOK;
    int en = 0;
    
    if (_parser->send("AT@PSMSET?") // AMM592 only
        && _parser->recv("@PSMSET:%d,%d,%d", &en, Requested_Periodic_TAU, Requested_Active_Time)
        && _parser->recv("OK")) 
    {
        if(en != 0) 
           *enable = true;
        else
           *enable = false;
        
        devlog("Get PSM setting success\r\n");
    
        ret = RET_OK;
    }
    return ret;
}

int8_t getDeactPsmSetting_AMM592(bool * enable)
{
    int8_t ret = RET_NOK;
    int en = 0;
    
    if (_parser->send("AT@PSMSET?") // AMM592 only
        && _parser->recv("@PSMSET:%d", &en)
        && _parser->recv("OK")) 
    {
        if(en != 0) 
           *enable = true;
        else
           *enable = false;
        
        devlog("Get PSM setting success\r\n");
        ret = RET_OK;
    }
    return ret;
}

// ----------------------------------------------------------------
// Functions: Cat.M1 Network time
// ----------------------------------------------------------------

int8_t getNetworkTimeLocal_AMM592(char * timestr)
{
    int8_t ret = RET_NOK;
    if (_parser->send("AT+CCLK?") 
        && _parser->recv("+CCLK: \"%[^\"]\"", timestr)
        && _parser->recv("OK")) 
    {
        //devlog("Get current local time success\r\n");        
        ret = RET_OK;
    }
    return ret;
}

void setFlagGettime(void)
{
    flag_gettime = true;
}

void clearFlagGettime(void)
{
    flag_gettime = false;
}
