/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file ranging_driver.h
 *\brief Application-level functions used for configuration and operation during ranging.
 */

#ifndef RANGING_DRIVER
#define RANGING_DRIVER

#include "common_driver.h"

//-----------------------------------------------------------------------------
// constant definitions
//-----------------------------------------------------------------------------

#define ECE_FACTOR 		(85/100)	//0.85
#define ECE_FACTOR_M 	85			//ECE factor Molecular
#define ECE_FACTOR_D	100			//ECE factor Denominator

// register addresses
#define SYSRANGE_START                        0x18
#define SYSRANGE_THRESH_HIGH                  0x19
#define SYSRANGE_THRESH_LOW                   0x1A
#define SYSRANGE_INTERMEASUREMENT_PERIOD      0x1B
#define SYSRANGE_MAX_CONVERGENCE_TIME         0x1C
#define SYSRANGE_CROSSTALK_COMPENSATION_RATE  0x1E
#define SYSRANGE_CROSSTALK_COMPENSATION_RANGE 0x20
#define SYSRANGE_CROSSTALK_VALID_HEIGHT       0x21
#define SYSRANGE_EARLY_CONVERGENCE_ESTIMATE   0x22
#define SYSRANGE_PART_TO_PART_RANGE_OFFSET    0x24
#define SYSRANGE_RANGE_IGNORE_VALID_HEIGHT    0x25
#define SYSRANGE_RANGE_IGNORE_THRESHOLD       0x26
#define SYSRANGE_EMITTER_BLOCK_THRESHOLD      0x28
#define SYSRANGE_MAX_AMBIENT_LEVEL_THRESH     0x2A
#define SYSRANGE_MAX_AMBIENT_LEVEL_MULT       0x2C
#define SYSRANGE_RANGE_CHECK_ENABLES          0x2D
#define SYSRANGE_VHV_RECALIBRATE              0x2E
#define SYSRANGE_VHV_REPEAT_RATE              0x31

#define RESULT_RANGE_STATUS      0x4D
#define RESULT_RANGE_VAL         0x62
#define RESULT_RANGE_SIGNAL_RATE 0x66

// SYSRANGE_START
#define RANGE_START_SINGLESHOT 0x01 // bit 0 set / bit 1 clear
#define RANGE_START_CONTINUOUS 0x03 // bit 0 set / bit 1 set
#define RANGE_CONTINUOUS_MODE  0x02 // bit 0 clear / bit 1 set
#define RANGE_STOP             0x00 // bit 0 set / bit 1 don't care

// RESULT_RANGE_STATUS bit masks
#define RANGE_DEVICE_READY       0x01
#define RANGE_ERROR_CODE         0xF0 // covers bits [7:4]

// SYSRANGE_RANGE_CHECK_ENABLES bit masks
#define RANGE_EARLY_CONVERGENCE_ENABLE  0x01
#define RANGE_EARLY_CONVERGENCE_DISABLE 0xFE
#define RANGE_RANGE_IGNORE_ENABLE       0x02
#define RANGE_RANGE_IGNORE_DISABLE      0xFD
#define RANGE_MAX_AMBIENT_ENABLE        0x04
#define RANGE_MAX_AMBIENT_DISABLE       0xFB
#define RANGE_EMITTER_BLOCK_ENABLE      0x08
#define RANGE_EMITTER_BLOCK_DISABLE     0xF7
#define RANGE_SIGNAL_TO_NOISE_ENABLE    0x10
#define RANGE_SIGNAL_TO_NOISE_DISABLE   0xEF

#define USER_CONV_CTRL_RETURN_THRESHOLD_FINE 0xB8

#define VERNIER_MEASUREMENTS              0x0109
#define VERNIER_RIPPLE_AVE_SAMPLE_PERIOD 0x010A


/**
 * @brief This data type defines range measurement data.
 */
typedef struct
{
    int32_t range_mm;
    /**< range distance measurement (mm). */

    float_t signalRate_mcps;
    /**<  signal rate (MCPS), which is effectively a measure of target reflectance.*/

    uint32_t errorStatus;
    /**< Error status of the current measurement. \n
     * No Error := 0. \n
     * Refer to product sheets for other error codes. */
}sensor_RangeData;


/*!
 *\brief Device setup, for Ranging operations, with parameters that require the system mode to be STARTed before being applied.
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error range_set_dynamic_config(uint8_t device_base_address);

/*!
 *\brief Set Mode and Operation commands in the sysrange_start register.
 *
 * Possible combinations are : \n
 *
 * RANGE_START_SINGLESHOT 0x01 # bit 0 set / bit 1 clear  \n
 * RANGE_START_CONTINUOUS 0x03 # bit 0 set / bit 1 set \n
 * RANGE_STOP             0x01 # bit 0 set / bit 1 don't care \n
 *
 *\param[in] device_base_address
 *\param[in] mode Mode select/operation command to be written to the SYSRANGE_START register.
 *
 *\retval Boolean. Return True if a valid command is applied, otherwise False.
 */
sensor_error range_set_systemMode(uint8_t device_base_address, int32_t mode);

/*!
 *\brief Report status of Range mode-select and Stop/Start.
 *
 * Range_Start_Singleshot \n
 * Range_Start_Continuous \n
 * Range_Stop \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_START register.
 */
uint8_t range_get_systemMode(uint8_t device_base_address);

/*!
 *\brief Report basic result from last ranging operation.
 *
 * Accessing Result_Range_Val register \n
 *
 *\param[in] device_base_address
 *\retval 8-bit Integer. Returns the result of the last successfully completed ranging operation, by reading the RESULT_RANGE_VAL register.
 */
uint8_t range_get_result(uint8_t device_base_address);

/*!
 *\brief Report signal rate from last ranging operation.
 * Reads the RESULT_RANGE_SIGNAL_RATE register.
 *\param[in] device_base_address
 *\retval a uint32_t in 9.7 format.
 */
uint32_t range_get_signal_rate(uint8_t device_base_address);

/*!
 *\brief Report all results from last ranging operation.
 *
 * Report all results data associated with a ranging operation. \n\n
 *
 * Result_range_val \n
 * Result_range_stray \n
 * Result_range_raw \n
 * Result_range_return_rate \n
 * Result_range_reference_rate \n
 * Result_range_Return_VCSEL_count \n
 * Result_range_Reference_VCSEL_count \n
 * Result_range_Return_AMB_count \n
 * Result_range_Reference_AMB_count \n
 * Result_range_Return_Conv_time \n
 * Result_range_Reference_Conv_time \n
 *
 *\param[in] device_base_address
 *
 *\retval List. Returns a list of all the results data, from the last successfully completed ranging operation.
 */
sensor_error range_get_full_result(uint8_t device_base_address);

/*!
 *\brief Set min/max range thresholds (1 to 254mm) in SYSRANGE_THRESH_LOW & SYSRANGE_THRESH_HIGH registers.
 *
 *\param[in] device_base_address
 *\param[in] low_threshold Ranging low threshold to be written to the SYSRANGE_THRESH_LOW register (default = 0mm.)
 *\param[in] high_threshold Ranging high threshold to be written to the SYSRANGE_THRESH_HIGH register (default = 255mm.)
 *\retval sensor_error
 */
sensor_error Range_Set_Thresholds(uint8_t device_base_address, int32_t low_threshold, int32_t high_threshold);

/*!
 *\brief Set ranging high threshold in the SYSRANGE_THRESH_HIGH register.
 *
 * Range : 0-255 mm \n
 *
 *\param[in] device_base_address
 *\param[in] threshold Ranging high threshold to be written to the SYSRANGE_THRESH_HIGH register(0-255mm).
 *\retval sensor_error
 */
sensor_error range_set_high_threshold(uint8_t device_base_address, uint8_t threshold);

/*!
 *\brief Report ranging high threshold from the SYSRANGE_THRESH_HIGH register.
 *
 * Range : 0-255 mm \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_THRESH_HIGH register.
 */
uint8_t range_get_high_threshold(uint8_t device_base_address);

/*!
 *\brief Set ranging low threshold in the SYSRANGE_THRESH_LOW register.
 *
 * Range : 0-255 mm \n
 *
 *\param[in] device_base_address
 *\param[in] threshold Ranging low threshold to be written to the SYSRANGE_THRESH_LOW register.
 *\retval sensor_error
 */
sensor_error range_set_low_threshold(uint8_t device_base_address, uint8_t threshold);

/*!
 *\brief Report ranging low threshold from the SYSRANGE_THRESH_LOW register.
 *
 * Range : 0-255 mm \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_THRESH_LOW register.
 */
uint8_t range_get_low_threshold(uint8_t device_base_address);

/*!
 *\brief Set ranging intermeasurement period in the SYSRANGE_INTERMEASUREMENT_PERIOD register.
 *
 * Time delay between measurements in continuous ranging mode. Range 10ms - 2.55secs (1 code = 10ms. Code 0 = 10ms.).\n
 * Min (default) value stored in NVM.
 *
 *\param[in] device_base_address
 *\param[in] intermeasurement_period Time delay in ms between measurements in continuous-ranging mode. <0-2550ms>
 *\retval sensor_error
 */
sensor_error range_set_interMeasurement_period(uint8_t device_base_address, uint16_t intermeasurement_period);

/*!
 *\brief Report ranging intermeasurement period from the SYSRANGE_INTERMEASUREMENT_PERIOD register.
 *
 * Range 10ms-2.55s, 1 code = 10 ms, code 0 = 10ms. \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the range inter measurement period in ms.
 */
uint16_t range_get_interMeasurement_period(uint8_t device_base_address);

/*!
 *\brief Set ranging maximum convergence time in the SYSRANGE_MAX_CONVERGENCE_TIME register.
 *
 * Maximum time to run measurements in ranging modes. \n
 *
 * Range 0-50ms, 1 code = 1 ms \n
 *
 *\param[in] device_base_address
 *\param[in] max_convergence_time Maximum time to run measurements in ranging modes (max = 50ms).
 *\retval sensor_error
 */
sensor_error range_set_max_convergence_time(uint8_t device_base_address, int32_t max_convergence_time);

/*!
 *\brief Report ranging maximum convergence time from the SYSRANGE_MAX_CONVERGENCE_TIME register.
 *
 * Range 0-50ms, 1 code = 1 ms \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_MAX_CONVERGENCE_TIME register.
 */

uint8_t range_get_max_convergence_time(uint8_t device_base_address);

/*!
 *\brief Set ranging crosstalk compensation rate in the SYSRANGE_CROSSTALK_COMPENSATION_RATE register.
 *
 * User-controlled crosstalk compensation rate, in MHz. \n
 *
 *\param[in] device_base_address
 *\param[in] crosstalk_compensation_rate User-controlled crosstalk compensation rate, in ranging modes.
 *\retval sensor_error
 */
sensor_error range_set_crosstalk_compensation_rate(uint8_t device_base_address, int32_t crosstalk_compensation_rate);

/*!
 *\brief Report ranging crosstalk compensation rate (MHz) from the SYSRANGE_CROSSTALK_COMPENSATION_RATE register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_CROSSTALK_COMPENSATION_RATE register.
 */
int32_t range_get_crosstalk_compensation_rate(uint8_t device_base_address);

/*!
 *\brief Set ranging crosstalk compensation range in the SYSRANGE_CROSSTALK_COMPENSATION_RANGE register.
 *
 * User-controlled crosstalk compensation range, in mm. \n
 *
 *\param[in] device_base_address
 *\param[in] crosstalk_compensation_range User-controlled crosstalk compensation range, in ranging modes.
 *\retval sensor_error
 */
sensor_error range_set_crosstalk_compensation_range(uint8_t device_base_address, int32_t crosstalk_compensation_range);

/*!
 *\brief Report ranging crosstalk compensation range (mm) from the SYSRANGE_CROSSTALK_COMPENSATION_RANGE register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_CROSSTALK_COMPENSATION_RANGE register.
 */
uint8_t range_get_crosstalk_compensation_range(uint8_t device_base_address);

/*!
 *\brief Set ranging crosstalk valid height in the SYSRANGE_CROSSTALK_VALID_HEIGHT register.
 *
 * Minimum range at which to apply crosstalk compensation, in mm. \n
 *\param[in] device_base_address
 *\param[in] crosstalk_valid_height Minimum range at which to apply crosstalk compensation, in ranging modes.
 *\retval sensor_error
 */
sensor_error range_set_crosstalk_valid_height(uint8_t device_base_address, int32_t crosstalk_valid_height);

/*!
 *\brief Report ranging crosstalk valid height from the SYSRANGE_CROSSTALK_VALID_HEIGHT register.
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_CROSSTALK_VALID_HEIGHT register.
 */
uint8_t range_get_crosstalk_valid_height(uint8_t device_base_address);

/*!
 *\brief Set ranging ECE (early convergence estimate) Threshold in the SYSRANGE_EARLY_CONVERGENCE_ESTIMATE' register.
 *
 * User-set limit below which a ranging operation is aborted. \n
 * An estimate of convergence time is performed 1ms into a ranging operation. If the return rate is below this user-set limit, the ranging operation is aborted to save power. \n
 *
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error range_set_early_convergence_estimate_threshold(uint8_t device_base_address);

/*!
 *\brief Report ranging early convergence estimate from the SYSRANGE_EARLY_CONVERGENCE_ESTIMATE register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_EARLY_CONVERGENCE_ESTIMATE register.
 */
uint8_t range_get_early_convergence_estimate_threshold(uint8_t device_base_address);

uint32_t range_get_vernier_ave_total_time(uint8_t device_base_address);

/*!
 *\brief read convergence count threshold from USER_CONV_CTRL_RETURN_THRESHOLD_FINE register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the formated reading of the USER_CONV_CTRL_RETURN_THRESHOLD_FINE register.
 */
uint32_t range_get_converg_ctrl_rtn_thresh_fine(uint8_t device_base_address);

/*!
 *\brief Set Ranging Ignore Valid Height threshold in the SYSRANGE_RANGE_IGNORE_VALID_HEIGHT register.
 *
 * Default : 256 \n
 *
 *\param[in] device_base_address
 *\param[in] ignore_valid_height User-set limit
 *\retval sensor_error
 */
sensor_error range_set_ignore_valid_height(uint8_t device_base_address, int32_t ignore_valid_height);

/*!
 *\brief Report ranging ignore valid height threshold from the SYSRANGE_RANGE_IGNORE_VALID_HEIGHT register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_RANGE_IGNORE_VALID_HEIGHT register.
 */
uint8_t range_get_ignore_valid_height(uint8_t device_base_address);

/*!
 *\brief Set Range Ignore Threshold in the SYSRANGE_RANGE_IGNORE_THRESHOLD register.
 *
 * Minimum acceptable count rate of VCSEL counts on the return array. \n
 *
 * Default : 2. \n
 *
 *\param[in] device_base_address
 *\param[in] range_ignore_threshold Minimum acceptable count rate of VCSEL counts on the return array. in Mcps 9.7 format.
 *\retval sensor_error
 */
sensor_error range_set_range_ignore_threshold(uint8_t device_base_address, uint32_t range_ignore_threshold);

/*!
 *\brief Report Range Ignore Threshold from the SYSRANGE_RANGE_IGNORE_THRESHOLD register. in Mcps 9.7 format.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_RANGE_IGNORE_THRESHOLD register.
 */
uint32_t range_get_range_ignore_threshold(uint8_t device_base_address);

/*!
 *\brief Set Emitter Block Threshold in the SYSRANGE_EMITTER_BLOCK_THRESHOLD register.
 *
 * Maximum Reference Array VCSEL returns allowed before reporting that the emitter is blocked. \n
 * MHz, 9.7 format \n
 *
 *\param[in] device_base_address
 *\param[in] emitter_block_threshold Minimum acceptable count rate of VCSEL counts on the return array.
 *\retval sensor_error
 */
sensor_error range_set_emitter_block_threshold(uint8_t device_base_address, int32_t emitter_block_threshold);

/*!
 *\brief Report Emitter Block Threshold from the SYSRANGE_EMITTER_BLOCK_THRESHOLD register.
 *
 * MHz, 9.7 format \n
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_EMITTER_BLOCK_THRESHOLD register.
 */
uint8_t range_get_emitter_block_threshold(uint8_t device_base_address);

/**
 * @brief Function to set the SNR threshold.
 *
 *\param[in] device_base_address
 *\param[in] snrThresh
 *\retval sensor_error
 */
sensor_error range_set_snr_thresh(uint8_t device_base_address, float_t snrThresh);

/**
 * @brief Function to get the SNR threshold.
 *
 *\param[in] device_base_address
 *\retval Float. Returns the SNR threshold.
 */
float_t range_get_snr_thresh(uint8_t device_base_address);


/*!
 *\brief Set Range Check Enables in the SYSRANGE_RANGE_CHECK_ENABLES register.
 *
 *\param[in] device_base_address
 *\param[in] range_check_enables
 *\retval sensor_error
 */
sensor_error range_set_range_check_enables(uint8_t device_base_address, int32_t range_check_enables);

/*!
 *\brief Report Range Check Enables from the SYSRANGE_RANGE_CHECK_ENABLES register.
 *
 *\param[in] device_base_address
 *\retval Integer. Returns the contents of the SYSRANGE_RANGE_CHECK_ENABLES register.
 */
uint8_t range_get_range_check_enables(uint8_t device_base_address);

/*!
 *\brief Set Range Check Enables in the SYSRANGE_VHV_RECALIBRATE register.
 *
 *\param[in] device_base_address
 *\param[in] VHV_Recalibrate
 *\retval sensor_error
 */
sensor_error range_set_vhv_recalibrate(uint8_t device_base_address, int32_t VHV_Recalibrate);

/*!
 *\brief Report Range Check Enables from the SYSRANGE_VHV_RECALIBRATE register.
 *
 *\param[in] device_base_address
 *\retval Integer.
 */
uint8_t range_get_vhv_recalibrate(uint8_t device_base_address);

/*!
 *\brief Set VHV Repeat Rate.
 *
 * Repeat rate of autoVHV task \n
 * measurements, 0 = off, 255 = after every 255 measurements \n
 *
 *\param[in] device_base_address
 *\param[in] VHV_repeat_rate Repeat rate of autoVHV task
 *\retval sensor_error
 */
sensor_error range_set_vhv_repeat_rate(uint8_t device_base_address, int32_t VHV_repeat_rate);

/*!
 *\brief Report VHV Repeat Rate.
 *
 *\param[in] device_base_address
 *\retval Integer. Repeat rate of autoVHV task, in measurements. 0 = off, 255 = after every 255 measurements.
 */
uint8_t range_get_vhv_repeat_rate(uint8_t device_base_address);

/*!
 *\brief Report contents of the RESULT_RANGE_STATUS register.
 *
 *\param[in] device_base_address
 *\retval Integer
 */
int32_t range_get_result_status(uint8_t device_base_address);

/*!
 *\brief Report result_range_device_ready status in the RESULT_RANGE_STATUS register.
 *
 *\param[in] device_base_address
 *\retval Boolean. Return True if the result_range_device_ready bit in the RESULT_RANGE_STATUS register is set, otherwise False.
 */
bool_t range_get_device_ready(uint8_t device_base_address);

/*!
 *\brief Report result_range_error_codes status.
 *
 *\param[in] device_base_address
 *\retval Integer. Return True if the result_range_error_codes bit in the RESULT_RANGE_STATUS register is set, otherwise False.
 */
uint8_t range_get_result_error_codes(uint8_t device_base_address);

/*!
 *\brief Set the ECE Factor Molecular and Demoninator.
 *
 * (< 1.00 for maximum detection, > 1.00 for minimizing red glow)
 *
 *\param[in] device_base_address
 *\param[in] ECE_Factor_M
 *\param[in] ECE_Factor_D
 *\retval sensor_error
 */
sensor_error range_set_ece_factor(uint8_t device_base_address, uint32_t ECE_Factor_M, uint32_t ECE_Factor_D);

/*!
 *\brief Store the ECE Margin Molecular and Demoninator to provided variables.
 *
 * (< 1.00 for maximum detection, > 1.00 for minimizing red glow)
 *
 *\param[in] pece_factor_m pointer to ECE_Factor_M variable
 *\param[in] pece_factor_d pointer to ECE_Factor_D variable
 *\retval sensor_error.
 */
sensor_error range_get_ece_factor(uint32_t *pece_factor_m, uint32_t *pece_factor_d);


/*!
 *\brief Set System Interrupt Config GPIO for Ranging operations.

    Returns True if a command is successfully applied to the SYSTEM_INTERRUPT_CONFIG_GPIO register, otherwise False. \n
    Possible settings are : \n
      CONFIG_GPIO_INTERRUPT_DISABLED = 0x00 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_LOW = 0x01 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_HIGH = 0x02 \n
      CONFIG_GPIO_INTERRUPT_OUT_OF_WINDOW = 0x03 \n
      CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY = 0x04 \n
 *\param[in] device_base_address
 *\param[in] Ranging_GPIO_interrupt_config
 *\retval sensor_error
*/
sensor_error range_set_system_interrupt_config_gpio(uint8_t device_base_address, uint8_t Ranging_GPIO_interrupt_config);

/*!
 *\brief Report System Interrupt Config GPIO Ranging.

    Returns the ranging-only portion of the SYSTEM_INTERRUPT_CONFIG_GPIO register. \n
    Possible returns are : \n
      CONFIG_GPIO_INTERRUPT_DISABLED = 0x00 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_LOW = 0x01 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_HIGH = 0x02 \n
      CONFIG_GPIO_INTERRUPT_OUT_OF_WINDOW = 0x03 \n
      CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY = 0x04 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t range_get_system_interrupt_config_gpio(uint8_t device_base_address);

/*!
 *\brief Clear Ranging System Interrupt.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error range_set_system_interrupt_clear(uint8_t device_base_address);


sensor_error Get_Range_History_Buffer(uint8_t device_base_address);

/*!
 *\brief Report GPIO Interrupt Result Status for a Ranging operation.

    Returns the Ranging-only portion of the RESULT_INTERRUPT_STATUS_GPIO register. \n
    Possible returns are : \n
    0: No threshold events reported \n
    1: Level Low threshold event \n
    2: Level High threshold event \n
    3: Out Of Window threshold event \n
    4: New Sample Ready threshold event \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t range_get_result_interrupt_status_gpio(uint8_t device_base_address);

/*!
 *\brief Set history buffer to range mode and enable.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error range_set_history_buffer_mode_enable(uint8_t device_base_address);


/**
 * @brief Function to return the Maximum range reported by the
 * sensor.
 */
uint32_t range_get_upper_limit(uint8_t device_base_address);

/**
 * @brief Function to return the Minimum range reported by the
 * sensor.
 */
uint32_t range_get_lower_limit(uint8_t device_base_address);

#endif


