/*******************************************************************************
################################################################################
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/**
 * @file platform.h
 *
 * @brief All platform specific declarations to be used by the High Level API are defined here.
 * The developer is responsible for providing the implementation of the prototypes declared.
 *
 */

#ifndef _PLATFORM
#define _PLATFORM
#ifndef __KERNEL__
#include <stdbool.h>
#include <stdint.h>
#endif
/** @brief Typedef defining .\n
 * The developer should modify this to suit the platform being deployed.
 */
#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/** @brief Typedef defining 8 bit unsigned char type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef unsigned char bool_t;

#ifndef _STDINT_H

/** @brief Typedef defining 64 bit unsigned long type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef unsigned long long uint64_t;

/** @brief Typedef defining 64 bit long type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef long long long64_t;

/** @brief Typedef defining 32 bit unsigned int type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef unsigned int uint32_t;

/** @brief Typedef defining 32 bit int type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef int int32_t;

/** @brief Typedef defining 16 bit unsigned short type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef unsigned short uint16_t;

/** @brief Typedef defining 16 bit short type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef short int16_t;

/** @brief Typedef defining 8 bit unsigned char type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef unsigned char uint8_t;

/** @brief Typedef defining 8 bit char type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef signed char int8_t;

#endif //_STDINT_H

/** @brief Typedef defining 64 bit double type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef double double_t;

/** @brief Typedef defining 32 bit float type.\n
 * The developer should modify this to suit the platform being deployed.
 */
typedef float float_t;

#ifndef NULL
#define NULL 0
#endif

/**
 * @brief Method to print a given string to the program output.
 */
void debug1_print(char *pBuffer);

/**
 * @brief Method to print a given string to the program output.
 */
void debug2_print(char *pBuffer);

/**
 * Method to initialise a given I2C device for a given I2C
 * address.
 */
void i2c_initialise(uint32_t addr);

/**
 * Method to close device if opened.
 */
void i2c_close();

/**
 * Method to write a single byte to a given I2C reg index.
 * Throws WriteFail upon failure.
 */
void i2c_write_byte(uint32_t reg, uint8_t data, uint8_t baseAddr);

/**
 * Method to write a single word to a given I2C reg index.
 * Throws WriteFail upon failure.
 */
void i2c_write_word(uint32_t reg, uint16_t data, uint8_t baseAddr);

/**
 * Method to write an array of bytes to a given I2C reg index.
 * Throws WriteFail upon failure.
 */
void i2c_write(uint32_t reg, uint8_t *data, int32_t size, uint8_t baseAddr);

/**
 * Method to read a single byte from a given I2C reg index.
 * Throws ReadFail upon error.
 */
uint8_t i2c_read_byte(uint32_t reg, uint8_t baseAddr);

/**
 * Method to read a two byte word from a given I2C reg index.
 * Throws ReadFail upon error.
 */
uint16_t i2c_read_word(uint32_t reg, uint8_t baseAddr);

/**
 * Method to read an int32 from a given I2C reg index.
 * Throws ReadFail upon error.
 */
uint32_t i2c_read_uint32(uint32_t reg, uint8_t baseAddr);

/**
 * Method to read multiple bytes from a given I2C reg index.
 * Throws ReadFail upon error.
 */
void i2c_read(uint32_t reg, uint8_t* data, int32_t size, uint8_t baseAddr);

/**
 * Method to start the timer.
 */
void timer_start();

/**
 * Method reporting the number of seconds elapsed since the timer was started.
 */
double_t timer_elapsedTime();

/**
 * Method to stop the timer.
 */
void timer_stop();

/**
 * Method to wait a given number of ms.
 */
void timer_wait_ms(int32_t ms);

/**
 * Method to wait a given number of us
 */
void timer_wait_us(int32_t us);

/**
 * Method to report the current clock time in us.
 */
uint32_t timer_get_clock_time_usecs();

/**
 * Method to report the current clock time in ms.
 */
uint32_t timer_get_clock_time_msecs();

#endif  /* _PLATFORM */


