/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file common_driver.h
 *\brief Application-level methods used for generic, system or identification operations.
 */

#ifndef _RANGE_DRIVER
#define _RANGE_DRIVER

#include "definitions.h"
#include "platform.h"
//-----------------------------------------------------------------------------
// module imports
//-----------------------------------------------------------------------------

// "circle_types.h" included for following definition :
//#ifndef __cplusplus
//typedef enum {FALSE = 0, TRUE = !FALSE} bool;
//#endif


//-----------------------------------------------------------------------------
// constant definitions
//-----------------------------------------------------------------------------

// API Version Strings
#define API_MAJOR_VERSION                      01
#define API_MINOR_VERSION                      00
#define API_BUILD_NUMBER                       00
#define API_CODE_REVISION                      1590

// register addresses
#define SYSTEM_MODE_GPIO0                      0x10
#define SYSTEM_MODE_GPIO1                      0x11
#define SYSTEM_HISTORY_CTRL                    0x12
#define SYSTEM_INTERRUPT_CONFIG_GPIO           0x14
#define SYSTEM_INTERRUPT_CLEAR                 0x15
#define SYSTEM_FRESH_OUT_OF_RESET              0x16
#define SYSTEM_GROUPED_PARAMETER_HOLD          0x17

#define RESULT_INTERRUPT_STATUS_GPIO           0x4F

#define PAD_I2C_CONFIG                         0x13D

#define I2C_SLAVE_DEVICE_ADDRESS               0x212

#define SYSTEM_FATAL_ERROR_CODE                0x290
#define SYSTEM_FATAL_ERROR_STATUS              0x291



// PAD_I2C_CONFIG
#define I2C_1v2_PAD_VOLTAGE                    0x01
#define I2C_2v8_PAD_VOLTAGE                    0x02

// SYSTEM_INTERRUPT_CONFIG_GPIO
#define CONFIG_GPIO_INTERRUPT_DISABLED         0x00
#define CONFIG_GPIO_INTERRUPT_LEVEL_LOW        0x01
#define CONFIG_GPIO_INTERRUPT_LEVEL_HIGH       0x02
#define CONFIG_GPIO_INTERRUPT_OUT_OF_WINDOW    0x03
#define CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY 0x04

// SYSTEM_MODE_GPIOx
#define GPIOx_MODE_SELECT_RANGING              0x00
#define GPIOx_MODE_SELECT_ALS                  0x01

#define GPIOx_POLARITY_SELECT_OFF              0x00
#define GPIOx_POLARITY_SELECT_ON               0x01
#define GPIOx_POLARITY_SELECT_CLEARED          0xDF

#define GPIOx_MODE_SELECT                      0x01
#define GPIOx_POLARITY_SELECT                  0x20

#define GPIOx_SELECT_OFF                       0x00
#define GPIOx_SELECT_MEASURE_READY             0x01
#define GPIOx_SELECT_THRESHOLD_OUTPUT          0x02
#define GPIOx_SELECT_BLANK_IN                  0x03
#define GPIOx_SELECT_BLANK_OUT                 0x04
#define GPIOx_SELECT_START_STOP                0x05
#define GPIOx_SELECT_DISABLED                  0x06
#define GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT 0x07
#define GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT     0x08

// SYSTEM_HISTORY_CTRL
#define HISTORY_BUFFER_ENABLE                  0x01
#define HISTORY_BUFFER_DISABLE                 0x00

#define HISTORY_BUFFER_ENABLED                 0x01
#define HISTORY_BUFFER_MODE                    0x02
#define HISTORY_BUFFER_CLEARED                 0x04

#define HISTORY_BUFFER_RANGING_MODE            0x00
#define HISTORY_BUFFER_ALS_MODE                0x01

// SYSTEM_INTERRUPT_CLEAR
#define INTERRUPT_CLEAR_RANGING                0x01
#define INTERRUPT_CLEAR_ALS                    0x02
#define INTERRUPT_CLEAR_ERROR                  0x04

// RESULT_INTERRUPT_STATUS_GPIO
#define RES_INT_STAT_GPIO_NO_ERROR             0x00
#define RES_INT_STAT_GPIO_LOW_LEVEL_THRESHOLD  0x01
#define RES_INT_STAT_GPIO_HIGH_LEVEL_THRESHOLD 0x02
#define RES_INT_STAT_GPIO_OUT_OF_WINDOW        0x03
#define RES_INT_STAT_GPIO_NEW_SAMPLE_READY     0x04


//-----------------------------------------------------------------------------
// global variable declarations
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// method definitions
//-----------------------------------------------------------------------------

/*!
 *
 */
sensor_error common_initialise(uint8_t device_base_address);

/*!
 *\brief Device setup for Ranging and ALS operations.\n
 *
 * To apply these settings the operation bits (bit 0) in the SYSRANGE_START SYSALS_START Registers must be cleared. \n
 *
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error common_set_static_config(uint8_t device_base_address);

/*!
 *\brief Report device Identification Info.
 *
 * Return the contents of the Identification block registers, in the following order : \n
 *
 * IDENTIFICATION_MODEL_ID \n
 * IDENTIFICATION_MODEL_REV_MAJOR \n
 * IDENTIFICATION_MODEL_REV_MINOR \n
 * IDENTIFICATION_MODULE_REV_MAJOR \n
 * IDENTIFICATION_MODULE_REV_MINOR \n
 * IDENTIFICATION_NVM_REVISION_ID \n
 * IDENTIFICATION_MASK_REVISION_ID \n
 * IDENTIFICATION_month \n
 * IDENTIFICATION_year \n
 * IDENTIFICATION_phase \n
 * IDENTIFICATION_day \n
 * IDENTIFICATION_time \n
 * IDENTIFICATION_code \n
 * IDENTIFICATION_FIRMWARE_REVISION_ID\n
 *\param[in] device_base_address
 *\retval List
 */
sensor_error common_get_identification(uint8_t device_base_address);

/*!
 *\brief Set Device I2C Base Address.
 *
 * Uses I2C_Slave_Device_address register at 0x0212. Can be over-written by a value read from NVM. \n
 * Defaults to 0x52 (8-bit address)/0x29 (7-bit). Register reports a 7-bit value. \n
 *\param[in] device_base_address
 *\param[in] i2c_base_address 8-bit I2C Slave device base address.
 *\retval sensor_error
 */
sensor_error common_set_i2c_base_address(uint8_t device_base_address, uint32_t i2c_base_address);

/*!
 *\brief Set I2C Pad Voltage.
 *
 * I2C Pad External Bus Supply voltage (applies to both pads). Can be overridden by a value read from in NVM. \n
 *
 * Possible values are : \n
 *     I2C_1v8_PAD_VOLTAGE = 0x01 \n
 *     I2C_2v8_PAD_VOLTAGE = 0x02 \n
 *\param[in] device_base_address
 *\param[in] pad_voltage I2C Pad External Bus Supply voltage.
 *\retval sensor_error
 */
sensor_error common_set_i2c_pad_voltage(uint8_t device_base_address, uint8_t pad_voltage);

/*!
 *\brief Report I2C Pad Voltage.
 *
 * I2C Pad External Bus Supply voltage.
 * Possible values are : \n
 *     I2C_1v8_PAD_VOLTAGE = 0x01 \n
 *     I2C_2v8_PAD_VOLTAGE = 0x02 \n
 *\param[in] device_base_address
 *\retval Integer
 */
uint8_t common_get_i2c_pad_voltage(uint8_t device_base_address);

/*!
 *\brief Set GPIO0 Mode.
 *
 * Possible values for mode are : \n
 *   GPIOx_MODE_SELECT_RANGING = 0x00 \n
 *   GPIOx_MODE_SELECT_ALS     = 0x01 \n
 *
 * Possible settings for select are : \n
 *   GPIOx_SELECT_OFF = 0x00 \n
 *   GPIOx_SELECT_MEASURE_READY = 0x01 \n
 *   GPIOx_SELECT_DISABLED = 0x06 \n
 *   GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n
 *
 * Possible settings for polarity are : \n
 *   GPIOx_POLARITY_SELECT_OFF = 0x00 \n
 *   GPIOx_POLARITY_SELECT_ON = 0x01 \n
 *\param[in] device_base_address
 *\param[in] mode Operating mode to be selected.
 *\param[in] select
 *\param[in] polarity
 *\retval sensor_error
 */
sensor_error common_set_system_mode_gpio0(uint8_t device_base_address, uint8_t mode, uint8_t select, uint8_t polarity);

/*!
 *\brief Set GPIO0 Mode.
 *
 * Possible values for mode are : \n
 *     GPIOx_MODE_SELECT_RANGING = 0x00 \n
 *     GPIOx_MODE_SELECT_ALS     = 0x01 \n
 *\param[in] device_base_address
 *\param[in] mode: Operating mode to be selected.
 *\retval sensor_error
 */
sensor_error common_set_gpio0_mode(uint8_t device_base_address, uint8_t mode);

/*!
 *\brief Report GPIO0 Mode.

    Possible results are : \n
     GPIOx_MODE_SELECT_RANGING = 0x00 \n
     GPIOx_MODE_SELECT_ALS     = 0x01 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_gpio0_mode(uint8_t device_base_address);

/*!
 *\brief Set GPIO0 function configuration.

    Possible settings are : \n
      GPIOx_SELECT_OFF = 0x00 \n
      GPIOx_SELECT_MEASURE_READY = 0x01 \n
      GPIOx_SELECT_THRESHOLD_OUTPUT = 0x02 \n
      GPIOx_SELECT_BLANK_IN = 0x03 \n
      GPIOx_SELECT_BLANK_OUT = 0x04 \n
      GPIOx_SELECT_START_STOP = 0x05 \n
      GPIOx_SELECT_DISABLED = 0x06 \n
      GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT = 0x07 \n
      GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n
 *\param[in] device_base_address
 *\param[in] select:
 *\retval sensor_error
*/
sensor_error common_set_gpio0_select(uint8_t device_base_address, uint8_t select);

/*!
 *\brief Report GPIO0 functional configuration.

    Possible results are : \n
      GPIOx_SELECT_OFF = 0x00 \n
      GPIOx_SELECT_MEASURE_READY = 0x01 \n
      GPIOx_SELECT_THRESHOLD_OUTPUT = 0x02 \n
      GPIOx_SELECT_BLANK_IN = 0x03 \n
      GPIOx_SELECT_BLANK_OUT = 0x04 \n
      GPIOx_SELECT_START_STOP = 0x05 \n
      GPIOx_SELECT_DISABLED = 0x06 \n
      GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT = 0x07 \n
      GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_gpio0_select(uint8_t device_base_address);

/*!
 *\brief Set GPIO0 Polarity.

    Possible settings are : \n
      GPIOx_POLARITY_SELECT_OFF = 0x00 \n
      GPIOx_POLARITY_SELECT_ON = 0x01 \n
 *\param[in] device_base_address
 *\param[in] polarity:
 *\retval sensor_error
*/
sensor_error common_set_gpio0_polarity(uint8_t device_base_address, uint8_t polarity);

/*!
 *\brief Report GPIO0 Polarity.
 *
 * Report the status of the GPIO_0 Polarity select bit in SYSTEM_MODE_GPIO0 register. \n
 *\param[in] device_base_address
 *\retval Boolean
*/
bool_t common_get_gpio0_polarity(uint8_t device_base_address);

/*!
 *\brief Set GPIO1 Mode.

    Possible values for mode are : \n
      GPIOx_MODE_SELECT_RANGING = 0x00 \n
      GPIOx_MODE_SELECT_ALS     = 0x01 \n

    Possible settings for select are : \n
      GPIOx_SELECT_OFF = 0x00 \n
      GPIOx_SELECT_MEASURE_READY = 0x01 \n
      GPIOx_SELECT_DISABLED = 0x06 \n
      GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n

    Possible settings for polarity are : \n
      GPIOx_POLARITY_SELECT_OFF = 0x00 \n
      GPIOx_POLARITY_SELECT_ON = 0x01 \n
 *\param[in] device_base_address
 *\param[in] mode: Operating mode to be selected.
 *\param[in] select:
 *\param[in] polarity:
 *\retval sensor_error
*/
sensor_error common_set_system_mode_gpio1(uint8_t device_base_address, uint8_t mode, uint8_t select, uint8_t polarity);

/*!
 *\brief Set GPIO1 Mode.

  Possible mode settings are : \n
     GPIOx_MODE_SELECT_RANGING = 0x00 \n
     GPIOx_MODE_SELECT_ALS     = 0x01 \n
 *\param[in] device_base_address
 *\param[in] mode:
 *\retval sensor_error
*/
sensor_error common_set_gpio1_mode(uint8_t device_base_address, uint8_t mode);

/*!
 *\brief Report GPIO1 Mode.

    Possible results are : \n
      GPIOx_MODE_SELECT_RANGING = 0x00 \n
      GPIOx_MODE_SELECT_ALS     = 0x01 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_gpio1_mode(uint8_t device_base_address);

/*!
 *\brief Set GPIO1 functional configuration.

    Possible settings are : \n
      GPIOx_SELECT_OFF = 0x00 \n
      GPIOx_SELECT_MEASURE_READY = 0x01 \n
      GPIOx_SELECT_THRESHOLD_OUTPUT = 0x02 \n
      GPIOx_SELECT_BLANK_IN = 0x03 \n
      GPIOx_SELECT_BLANK_OUT = 0x04 \n
      GPIOx_SELECT_START_STOP = 0x05 \n
      GPIOx_SELECT_DISABLED = 0x06 \n
      GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT = 0x07 \n
      GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n
 *\param[in] device_base_address
 *\param[in] select:
 *\retval sensor_error
*/
sensor_error common_set_gpio1_select(uint8_t device_base_address, uint8_t select);

/*!
 *\brief Report GPIO1 functional configuration.

    Possible results are : \n
      GPIOx_SELECT_OFF = 0x00 \n
      GPIOx_SELECT_MEASURE_READY = 0x01 \n
      GPIOx_SELECT_THRESHOLD_OUTPUT = 0x02 \n
      GPIOx_SELECT_BLANK_IN = 0x03 \n
      GPIOx_SELECT_BLANK_OUT = 0x04 \n
      GPIOx_SELECT_START_STOP = 0x05 \n
      GPIOx_SELECT_DISABLED = 0x06 \n
      GPIOx_SELECT_COMBINED_THRESHOLD_OUTPUT = 0x07 \n
      GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT = 0x08 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_gpio1_select(uint8_t device_base_address);

/*!
 *\brief Set GPIO1 Polarity.

      GPIOx_POLARITY_SELECT_OFF = 0x00 \n
      GPIOx_POLARITY_SELECT_ON = 0x01 \n
 *\param[in] device_base_address
 *\param[in] polarity:
 *\retval sensor_error
*/
sensor_error common_set_gpio1_polarity(uint8_t device_base_address, uint8_t polarity);

/*!
 *\brief Report GPIO1 Polarity.

    Report the status of the GPIO_0 Polarity select bit in SYSTEM_MODE_GPIO0 register.
 *\param[in] device_base_address
 *\retval Boolean
*/
bool_t common_get_gpio1_polarity(uint8_t device_base_address);

/*!
 *\brief Set History Buffer Enable.

    Possible settings are : \n
      HISTORY_BUFFER_ENABLE = 0x01 \n
      HISTORY_BUFFER_DISABLE = 0x00 \n
 *\param[in] device_base_address
 *\param[in] history_buffer_enable:
 *\retval sensor_error
*/
sensor_error common_set_history_buffer_enable(uint8_t device_base_address, uint8_t history_buffer_enable);

/*!
 *\brief Report History Buffer Enable status.

    Returns True if the command completes successfully, otherwise False.\n
 *\param[in] device_base_address
 *\retval Boolean
*/
bool_t common_get_history_buffer_enable(uint8_t device_base_address);

/*!
 *\brief Set History Buffer Mode.

    Possible modes are : \n
      HISTORY_BUFFER_RANGING_MODE = 0x00 \n
      HISTORY_BUFFER_ALS_MODE = 0x01 \n
 *\param[in] device_base_address
 *\param[in] history_buffer_mode:
 *\retval sensor_error
*/
sensor_error common_set_history_buffer_mode(uint8_t device_base_address, uint8_t history_buffer_mode);

/*!
 *\brief Report History Buffer Mode.

    Returns the contents of the SYSTEM_HISTORY_CTRL register. \n
    Possible values are : \n
      HISTORY_BUFFER_RANGING_MODE = 0x00 \n
      HISTORY_BUFFER_ALS_MODE = 0x01 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_history_buffer_mode(uint8_t device_base_address);

/*!
 *\brief Clear history buffer (mode not cleared)
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error common_set_history_buffer_clear(uint8_t device_base_address);

/*!
 *\brief Report History Buffer Clear status.
 *
 * Returns True if the SYSTEM_HISTORY_CTRL register is cleared, otherwise False. \n
 *\param[in] device_base_address
 *\retval Boolean
*/
bool_t common_get_history_buffer_clear(uint8_t device_base_address);

/*!
 *\brief Clear 'Error' System Interrupt.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error common_set_system_interrupt_clear_error(uint8_t device_base_address);

/*!
 *\brief Report GPIO Interrupt Error Result Status.

    Returns the Error flag portion of the RESULT_INTERRUPT_STATUS_GPIO register. \n
    Possible returns are : \n
    0: No error reported \n
    1: Laser Safety Error \n
    2: PLL error (either PLL1 or PLL2) \n
    3: Measurement error \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_error_result_interrupt_status_gpio(uint8_t device_base_address);

/*!
 *\brief Clear System Fresh Out Of Reset flag.
 *
 * The fresh out of reset bit defaults to 1 at boot-up. The user can set this to 0 after initial boot and then use this flag to check for reset conditions.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error common_clear_system_fresh_out_of_reset(uint8_t device_base_address);

/*!
 *\brief Report System Fresh Out Of Reset status.
 *
 * Returns True if the SYSTEM_FRESH_OUT_OF_RESET register status is set, otherwise False. \n
 *\param[in] device_base_address
 *\retval Boolean
*/
bool_t common_get_system_fresh_out_of_reset(uint8_t device_base_address);

/*!
 *\brief Set System Group Parameter Hold flag.

    The grouped_parameter_hold flag is set to indicate that device data is being updated. \n
    0: Data is stable - user is safe to copy. \n
    1: Data being updated - user not safe to copy. \n
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error common_set_system_group_parameter_hold(uint8_t device_base_address);

/*!
 *\brief Clear System Group Parameter Hold flag.

    The grouped_parameter_hold flag is set to indicate that device data is being updated. \n
    0: Data is stable - user is safe to copy. \n
    1: Data being updated - user not safe to copy. \n
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error common_clear_system_group_parameter_hold(uint8_t device_base_address);

/*!
 *\brief Report System Fatal Error Code flag status.
 *
 * Returns the contents of the SYSTEM_FATAL_ERROR_CODE register, which is written to when a fatal error occurs.
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_system_fatal_error_code(uint8_t device_base_address);

/*!
 *\brief Report System Fatal Error Status.

    Returns the contents of the SYSTEM_FATAL_ERROR_STATUS register, which is written to when a fatal error occurs.\n

    Read Only HW status bit, set high when a 1 is written into FATAL_ERROR_CODE. \n
    Checked by FW on bootup to ascertain device status. Can only be rest by a main Go2 reset. \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t common_get_system_fatal_error_status(uint8_t device_base_address);

/*!
 *\brief Report System Health.
 *
 * A combination of fresh_from_reset, fatal_error and laser safety checks
 *\param[in] device_base_address
 *\retval Integer. A combined system health flag.
*/
uint8_t common_get_system_health_check(uint8_t device_base_address);

#endif


