/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file als_driver.h
 *\brief Application-level methods used by VL6180X for ALS operations.
 */

#ifndef _ALS_DRIVER
#define _ALS_DRIVER

#include "definitions.h"
#include "common_driver.h"

//-----------------------------------------------------------------------------
// constant definitions
//----------------------------------------------------------------------------

// registers addresses
#define IDENTIFICATION_MODEL_ID             0x00
#define IDENTIFICATION_FIRMWARE_REVISION_ID 0x0B
#define IDENTIFICATION_MODULE_REV_MAJOR     0x03
#define IDENTIFICATION_MODULE_REV_MINOR     0x04

#define SYSALS_START                        0x38
#define SYSALS_THRESH_HIGH                  0x3A
#define SYSALS_THRESH_LOW                   0x3C
#define SYSALS_INTERMEASUREMENT_PERIOD      0x3E
#define SYSALS_ANALOGUE_GAIN                0x3F
#define SYSALS_INTEGRATION_PERIOD           0x40

#define RESULT_ALS_STATUS                   0x4E
#define RESULT_ALS_VAL                      0x50

#define INTERLEAVED_MODE_ENABLE             0x2A3


// SYSALS_START
#define ALS_START_SINGLESHOT                0x01 // bit 0 set / bit 1 clear
#define ALS_START_CONTINUOUS                0x03 // bit 0 set / bit 1 set
#define ALS_STOP                            0x00 // bit 0 set / bit 1 don't care

// RESULT_ALS_STATUS
#define ALS_DEVICE_READY                    0x01
#define ALS_ERROR_CODE                      0xF0 // covers bits [7:4]
#define FW_ALS_RESULT_SCALER                   0x120


#define DEFAULT_INTEGRATION_PERIOD          0x64
#define DEFAULT_LUX_RESOLUTION              0.56f
#define DEFAULT_ALS_SCALER                  1
#define DEFAULT_ALS_GAIN                    20
/**
 * @brief This data type defines range measurment data.
 */
typedef struct
{
    float_t lux;
    /**< Light measurement (Lux) */

    uint32_t errorStatus;
    /**< Error status of the current measurement. \n
     * No Error := 0. \n
     * Refer to product sheets for other error codes. */
}sensor_AlsData;


/*!
 *
 *\brief Device setup for ALS parameters. These settings can be applied at any time. The status of operation bit (bit 0) of the SYSALS_START is not important.
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error als_set_dynamic_config(uint8_t device_base_address);

/*!
 *\brief Set Mode and Operation commands in the SYSALS_START register.
 *
 * Possible combinations are : \n
 *   ALS_START_SINGLESHOT 0x01 (bit 0 set / bit 1 clear) \n
 *   ALS_START_CONTINUOUS 0x03 (bit 0 set / bit 1 set) \n
 *   ALS_STOP             0x01 (bit 0 set / bit 1 don't care) \n
 *\param[in] device_base_address
 *\param[in] mode Mode select/operation command to be written to the SYSALS_START register.
 *\retval sensor_error
 */
sensor_error als_set_systemMode(uint8_t device_base_address, uint8_t mode);

/*!
 *\brief Report status of ALS mode-select and Stop/Start.
 *
 * Returns a reading of the SYSALS_START register. \n
 * Possible results are : \n
 *   ALS_START_SINGLESHOT 0x01 (bit 0 set / bit 1 clear) \n
 *   ALS_START_CONTINUOUS 0x03 (bit 0 set / bit 1 set) \n
 *   ALS_STOP             0x01 (bit 0 set / bit 1 don't care) \n
 *\param[in] device_base_address
 *\retval uint8_t (unsigned, byte-wide integer)
 */
uint8_t als_get_systemMode(uint8_t device_base_address);

/*!
 *\brief Report 16-bit result from last ALS operation.
 *
 * Accesses the RESULT_ALS_VAL register, to report the latest ALS raw measurement.
 *\param[in] device_base_address
 *\retval uint16_t (unsigned, word-wide integer)
 */
uint16_t als_get_result(uint8_t device_base_address);

/*!
 *\brief Report 16-bit result from last ALS operation.
 *
 * Reads the latest ALS measurement and reports to the calling operation in LUX.
 *\param[in] device_base_address
 *\retval uint16_t (unsigned, word-wide integer)
 */
uint16_t als_get_lux(uint8_t device_base_address);

/*!
 *\brief Set min/max ALS thresholds (units?) in SYSALS_THRESH_LOW & SYSALS_THRESH_HIGH registers.
 *\param[in] device_base_address
 *\param[in] low_threshold Byte-wide, integer, ALS low threshold to be written to the SYSALS_THRESH_LOW register.
 *\param[in] high_threshold Byte-wide, integer, ALS high threshold to be written to the SYSALS_THRESH_HIGH register.
 *\retval sensor_error
 */
sensor_error als_set_thresholds(uint8_t device_base_address, uint16_t low_threshold, uint16_t high_threshold);

/*!
 *\brief Set ALS high threshold the SYSALS_THRESH_HIGH register.
 *\param[in] device_base_address
 *\param[in] threshold Word-wide, integer, ALS high threshold to be written to the SYSALS_THRESH_HIGH register.
 *\retval sensor_error
 */
sensor_error als_set_high_threshold(uint8_t device_base_address, uint16_t threshold);

/*!
 *\brief Report ALS high threshold from the SYSALS_THRESH_HIGH register.
 *\param[in] device_base_address
 *\retval uint16_t (unsigned, word-wide integer)
 */
uint16_t als_get_high_threshold(uint8_t device_base_address);

/*!
 *\brief Set ALS low threshold the SYSALS_THRESH_LOW register.
 *\param[in] device_base_address
 *\param[in] threshold ALS low threshold to be written to the SYSALS_THRESH_LOW register.
 *\retval sensor_error
 */
sensor_error als_set_low_threshold(uint8_t device_base_address, uint16_t threshold);

/*!
 *\brief Report ALS low threshold from the SYSALS_THRESH_LOW register.
 *\param[in] device_base_address
 *\retval uint16_t (unsigned, word-wide integer)
 */
uint16_t als_get_low_threshold(uint8_t device_base_address);

/*!
 *\brief Set ALS intermeasurement period in the SYSALS_INTERMEASUREMENT_PERIOD register.
 *
 * Range 10ms-2.55s, 1 code = 10 ms, code 0 = 10ms
 *\param[in] device_base_address
 *\param[in] intermeasurement_period: Time delay in ms between measurements in continuous-ALS mode.
 *\retval sensor_error
 */
sensor_error als_set_interMeasurement_period(uint8_t device_base_address, uint16_t intermeasurement_period);

/*!
 *\brief Report ALS intermeasurement period from the SYSALS_INTERMEASUREMENT_PERIOD register.
 *
 * Range 0-2.55s, 1 code = 10 ms
 *\param[in] device_base_address
 *\retval an 16-bit integer as time in ms.
 */
uint16_t als_get_interMeasurement_period(uint8_t device_base_address);

/*!
 *\brief Set dark/light ALS analogue gains in the SYSALS_ANALOGUE_GAIN register.
 *
 * Light gain will be clipped to a maximum value of 7. \n
 * Dark Gain is frozen at 4. \n
 *
 * [2:0] sysals_analogue_gain_light: ALS analogue gain (light channel) \n
 * 0: ALS Gain = 20 \n
 * 1: ALS Gain = 10 \n
 * 2: ALS Gain = 5.0 \n
 * 3: ALS Gain = 2.5 \n
 * 4: ALS Gain = 1.67 \n
 * 5: ALS Gain = 1.25 \n
 * 6: ALS Gain = 1.0 \n
 * 7: ALS Gain = 40 (testmode) \n
 *\param[in] device_base_address
 *\param[in] light_analogue_gain light gain
 *\retval sensor_error
 */
sensor_error als_set_analogue_gain(uint8_t device_base_address, uint8_t light_analogue_gain);

/*!
 *\brief Report ALS analogue gain from the SYSALS_ANALOGUE_GAIN register.
 *\param[in] device_base_address
 *\retval an 8-bit integer.
 */
uint8_t als_get_analogue_gain(uint8_t device_base_address);

/*!
 *\brief Set ALS Integration Period in the SYSALS_INTEGRATION_PERIOD register.
 *\param[in] device_base_address
 *\param[in] integration_period Integration period for ALS mode, in ms. 1 code = 1ms [codes (0:464) == (1ms:465ms)]
 *\retval sensor_error
 */
sensor_error als_set_integration_period(uint8_t device_base_address, uint16_t integration_period);

/*!
 *\brief Report ALS Integration Period from the SYSALS_INTEGRATION_PERIOD register.
 *\param[in] device_base_address
 *\retval a 16-bit integer.
 */
uint16_t als_get_integration_period(uint8_t device_base_address);

/*!
 *\brief Report ALS Result Status from RESULT_ALS_STATUS register.
 *\param[in] device_base_address
 *\retval an 8-bit integer.
 */
uint8_t als_get_result_status(uint8_t device_base_address);

/*!
 *\brief Report result_ALS_device_ready status in the ALS_RANGE_STATUS register.
 *\param[in] device_base_address
 *\retval True if the result_ALS_device_ready bit in the RESULT_ALS_STATUS register is set, otherwise False.
 */
bool_t als_get_device_ready(uint8_t device_base_address);

/*!
 *\brief Report ALS Result Error Codes from RESULT_ALS_STATUS register.
 *\param[in] device_base_address
 *\retval an 8-bit integer.
 */
uint8_t als_get_result_error_codes(uint8_t device_base_address);

/*!
 *\brief Enable Interleaved Mode.
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error als_set_interleaved_mode(uint8_t device_base_address);

/*!
 *\brief Disable Interleaved Mode.
 *\param[in] device_base_address
 *\retval sensor_error
 */
sensor_error als_clear_interleaved_mode(uint8_t device_base_address);

/*!
 *\brief Report Interleaved Mode.
 *\param[in] device_base_address
 *\retval an 8-bit integer.
 */
uint8_t als_get_interleaved_mode(uint8_t device_base_address);

/*!
 *\brief Set System Interrupt Config GPIO for ALS operations.

    Returns True a valid command was completed successfully, otherwise False.\n
    Possible settings are :\n
      CONFIG_GPIO_INTERRUPT_DISABLED = 0x00 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_LOW = 0x01 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_HIGH = 0x02 \n
      CONFIG_GPIO_INTERRUPT_OUT_OF_WINDOW = 0x03 \n
      CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY = 0x04 \n
 *\param[in] device_base_address
 *\param[in] ALS_GPIO_interrupt_config:
 *\retval sensor_error
*/
sensor_error als_set_system_interrupt_config_gpio(uint8_t device_base_address, uint8_t ALS_GPIO_interrupt_config);

/*!
 *\brief Report System Interrupt Config GPIO ALS.

    Returns the ALS-only portion of the SYSTEM_INTERRUPT_CONFIG_GPIO register.\n
    Possible returns are : \n
      CONFIG_GPIO_INTERRUPT_DISABLED = 0x00 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_LOW = 0x01 \n
      CONFIG_GPIO_INTERRUPT_LEVEL_HIGH = 0x02 \n
      CONFIG_GPIO_INTERRUPT_OUT_OF_WINDOW = 0x03 \n
      CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY = 0x04 \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t als_get_system_interrupt_config_gpio(uint8_t device_base_address);

/*!
 *\brief Report GPIO Interrupt Result Status for an ALS operation.

    Returns the ALS-only portion of the RESULT_INTERRUPT_STATUS_GPIO register.\n
    Possible returns are : \n
    0: No threshold events reported \n
    1: Level Low threshold event \n
    2: Level High threshold event \n
    3: Out Of Window threshold event \n
    4: New Sample Ready threshold event \n
 *\param[in] device_base_address
 *\retval Integer
*/
uint8_t als_get_result_interrupt_status_gpio(uint8_t device_base_address);


/*!
 *\brief Clear ALS System Interrupt.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error als_set_system_interrupt_clear(uint8_t device_base_address);

/*!
 *\brief Set history buffer to ALS mode and enable.
 *\param[in] device_base_address
 *\retval sensor_error
*/
sensor_error als_set_history_buffer_mode_enable(uint8_t device_base_address);

sensor_error Get_ALS_History_Buffer(uint8_t device_base_address);

sensor_error als_set_scaler(uint8_t device_base_address, uint8_t scaler);

uint32_t als_get_scaler(uint8_t device_base_address);

#endif


