/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/


/**
 * @file high_level_api.cpp
 *
 * Copyright (C) 2014 ST MicroElectronics
 *
 * @brief High level interface definition for device, providing methods for basic
 * ranging.
 *
 */

#include "vl6180x_high_level_api.h"
#include "definitions.h"
#include "common_driver.h"
#include "ranging_driver.h"
#include "als_driver.h"
#include "range_upscaling_driver.h"
#include "debug.h"
#include "utilities.h"

static uint8_t _device_i2c_addr = 0;

sensor_error get_vendor(uint8_t *pVendorStr)
{
    LOG_FUNCTION_START((void*)pVendorStr);

    LOG_FUNCTION_END(NULL);
    return SENSOR_ERROR_NONE;
}

sensor_error get_version(int32_t id, uint8_t *pVersionStr)
{
    LOG_FUNCTION_START((void*)&id,(void*)pVersionStr);

    LOG_FUNCTION_END(NULL);
    return SENSOR_ERROR_NONE;
}

uint32_t get_max_range()
{
    uint32_t maxRange = 0;

    maxRange = range_get_upper_limit(_device_i2c_addr);

    return maxRange;
}

uint32_t get_min_range()
{
    return range_get_lower_limit(_device_i2c_addr);;
}

sensor_error initialise(uint8_t device_base_address)
{

    LOG_FUNCTION_START((void*)&device_base_address);

    _device_i2c_addr = device_base_address;
    common_initialise(_device_i2c_addr);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error start_ranging(int32_t id)
{
    LOG_FUNCTION_START((void*)&id);

    // load settings
    common_set_static_config(_device_i2c_addr);

    // ranging setup
    range_set_low_threshold(_device_i2c_addr, 10); // threshold defaults are 0mm and 255mm
    range_set_high_threshold(_device_i2c_addr, 200);//200mm
    range_set_max_convergence_time(_device_i2c_addr, 50); // set max convergence time to 50ms;

    //set ranging in InterruptMode
    common_set_gpio1_polarity(_device_i2c_addr, GPIOx_POLARITY_SELECT_OFF);
    common_set_gpio1_select(_device_i2c_addr, GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT);
    common_set_gpio1_mode(_device_i2c_addr, GPIOx_MODE_SELECT_RANGING);
    range_set_system_interrupt_config_gpio(_device_i2c_addr, CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY);

    //range_set_systemMode(_device_i2c_addr, RANGE_START_SINGLESHOT);
    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error start_extended_ranging(int32_t id)
{
    LOG_FUNCTION_START((void*)&id);
    // load settings onto
    er_set_static_config(_device_i2c_addr);

    // ranging setup
    // range_set_low_threshold(_device_i2c_addr, 10); // threshold defaults are 0mm and 255mm
    // range_set_high_threshold(_device_i2c_addr, 200);//200mm
    range_set_max_convergence_time(_device_i2c_addr, 63); // set max convergence time to 50ms;
    range_set_crosstalk_compensation_rate(_device_i2c_addr, 0);

    //set ranging in InterruptMode
    common_set_gpio1_polarity(_device_i2c_addr, GPIOx_POLARITY_SELECT_OFF);
    common_set_gpio1_select(_device_i2c_addr, GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT);
    common_set_gpio1_mode(_device_i2c_addr, GPIOx_MODE_SELECT_RANGING);
    range_set_system_interrupt_config_gpio(_device_i2c_addr, CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error get_range_measurement(int32_t id, sensor_RangeData *pRangeData)
{
    const int cErrorDataShift = 4;

    LOG_FUNCTION_START((void*)&id,(void*)pRangeData);
    // start single range measurement
    range_set_systemMode(_device_i2c_addr, RANGE_START_SINGLESHOT);

    // poll the till new sample ready
    while (range_get_result_interrupt_status_gpio(_device_i2c_addr) != RES_INT_STAT_GPIO_NEW_SAMPLE_READY)
    {
        timer_wait_us(100);
    }

    pRangeData->range_mm        = range_get_result(_device_i2c_addr);
    pRangeData->signalRate_mcps = decodeFrom9_7_Format(range_get_signal_rate(_device_i2c_addr));
    pRangeData->errorStatus     = (range_get_result_status(_device_i2c_addr)) >> cErrorDataShift;

    // clear the interrupt on
    range_set_system_interrupt_clear(_device_i2c_addr);
    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error start_als(int32_t id)
{
    LOG_FUNCTION_START((void*)&id);
    common_set_static_config(_device_i2c_addr);

    als_set_integration_period(_device_i2c_addr, 100); //100ms
    als_set_interMeasurement_period(_device_i2c_addr, 200); //200ms
    als_set_analogue_gain(_device_i2c_addr, 0);

    als_set_low_threshold(_device_i2c_addr, 0);
    als_set_high_threshold(_device_i2c_addr, 0x0FFFF);

    common_set_gpio1_polarity(_device_i2c_addr, GPIOx_POLARITY_SELECT_OFF);
    common_set_gpio1_select(_device_i2c_addr, GPIOx_SELECT_GPIO_INTERRUPT_OUTPUT);
    common_set_gpio1_mode(_device_i2c_addr, GPIOx_MODE_SELECT_ALS);
    als_set_system_interrupt_config_gpio(_device_i2c_addr, CONFIG_GPIO_INTERRUPT_NEW_SAMPLE_READY);

//    als_set_systemMode(_device_i2c_addr, ALS_START_SINGLESHOT);
    LOG_FUNCTION_END(NULL);
	
    return SENSOR_ERROR_NONE;
}

sensor_error get_als_measurement(int32_t id, sensor_AlsData *pAlsData)
{
    const int cErrorDataShift = 4;
	
    LOG_FUNCTION_START((void*)&id,(void*)pAlsData);
    als_set_systemMode(_device_i2c_addr, ALS_START_SINGLESHOT);

    while (als_get_result_interrupt_status_gpio(_device_i2c_addr) != RES_INT_STAT_GPIO_NEW_SAMPLE_READY)
    {
        timer_wait_us(100);
    }
    pAlsData->lux         = als_get_lux(_device_i2c_addr);
    pAlsData->errorStatus = (als_get_result_status(_device_i2c_addr)) >> cErrorDataShift;

    als_set_system_interrupt_clear(_device_i2c_addr);
    LOG_FUNCTION_END(NULL);
		
    return SENSOR_ERROR_NONE;
}

sensor_error get_minimum_delay()
{

    return SENSOR_ERROR_NONE;
}

