/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*!
 *\file i2c_log.h
 *\brief I2C logging functionality.
 */

#include "platform.h"

/*!
 * \brief		Function to Start logging all I2C activities
 *
 * All I2C access activity will be logged once this function is called.
 * This function shall be called when debug level is set to I2C_LOG_LEVEL.
 * \retval		none
 */
void i2c_log_start();

/*!
 * \brief		Function to Stop logging all I2C activities
 *
 * All I2C access activity will stop to be logged once this function is called.
 * This function shall be called when debug level is set to I2C_LOG_LEVEL.
 * \retval		none
 */
void i2c_log_end();

/*!
 *\brief Function to output an I2C read transaction for a single byte to log file.
 *
 *\param[in] value data value read.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputReadByteMsg(uint8_t value, uint32_t regOffset);

/*!
 *\brief Function to output an I2C read transaction for a single word to log file.
 *
 *\param[in] value data value read.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputReadWordMsg(uint16_t value, uint32_t regOffset);

/*!
 *\brief Function to output an I2C read transaction for a single int to log file.
 *
 *\param[in] value data value read.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputReadIntMsg(uint32_t value, uint32_t regOffset);

/*!
 *\brief Function to output an I2C read transaction for an array of bytes to log file.
 *
 *\param[in] data data values read.
 *\param[in] regOffset I2C register identifier.
 *\param[in] size size of the data buffer.
 *\retval none
 */
void i2c_log_outputReadMsg(uint8_t *data, uint32_t regOffset, int32_t size);

/*!
 *\brief Function to output an I2C write transaction for a single byte to log file.
 *
 *\param[in] data data value written.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputWriteByteMsg(uint8_t data, uint32_t regOffset);

/*!
 *\brief Function to output an I2C write transaction for a single byte to log file.
 *
 *\param[in] data data value written.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputWriteWordMsg(uint16_t data, uint32_t regOffset);

/*!
 *\brief Function to output an I2C write transaction for a single byte to log file.
 *
 *\param[in] data data value written.
 *\param[in] regOffset I2C register identifier.
 *\retval none
 */
void i2c_log_outputWriteintMsg(uint32_t data, uint32_t regOffset);

/*!
 *\brief Function to output an I2C write transaction for an array of bytes to log file.
 *
 *\param[in] value data values written.
 *\param[in] regOffset I2C register identifier.
 *\param[in] size size of the data buffer.
 *\retval none
 */
void i2c_log_outputWriteMsg(uint8_t *value, uint32_t regOffset, int32_t size);


