/*******************************************************************************
################################################################################
#                             (C) STMicroelectronics 2014
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License version 2 and only version 2 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
#------------------------------------------------------------------------------
#                             Imaging Division
################################################################################
********************************************************************************/

/*
'''
Application-level functions used for configuration and operation during extended
ranging.
'''
*/

//-----------------------------------------------------------------------------
// module imports
//-----------------------------------------------------------------------------

// ST libraries

#include "range_upscaling_driver.h"

#include "common_driver.h"
#include "ranging_driver.h"
#include "debug.h"
#include "platform.h"


//-----------------------------------------------------------------------------
// constant definitions
//-----------------------------------------------------------------------------
#define IDENTIFICATION_MODEL_ID 0x00
#define IDENTIFICATION_MODULE_REV_MAJOR 0x03
#define IDENTIFICATION_MODULE_REV_MINOR 0x04

#define RESULT_RANGE_VAL 0x62

#define RANGE_SCALER 0x96

//-----------------------------------------------------------------------------
// global variable declarations
//-----------------------------------------------------------------------------
static ExtRangeScaler _rangeScaler = Scale2X;

//-----------------------------------------------------------------------------
// method definitions
//-----------------------------------------------------------------------------

static void er_set_register_tuning(uint8_t device_base_address);


static void er_set_register_tuning(uint8_t device_base_address)
{
    // apply REGISTER_TUNING_ER02_100614_CustomerView.tx

    // Mandatory : private registers
    i2c_write_byte(0x0207, 0x01, device_base_address);
    i2c_write_byte(0x0208, 0x01, device_base_address);
    i2c_write_byte(0x0096, 0x00, device_base_address);
    i2c_write_byte(0x0097, 0x54, device_base_address);
    i2c_write_byte(0x00e3, 0x00, device_base_address);
    i2c_write_byte(0x00e4, 0x04, device_base_address);
    i2c_write_byte(0x00e5, 0x02, device_base_address);
    i2c_write_byte(0x00e6, 0x01, device_base_address);
    i2c_write_byte(0x00e7, 0x03, device_base_address);
    i2c_write_byte(0x00f5, 0x02, device_base_address);
    i2c_write_byte(0x00d9, 0x05, device_base_address);
    i2c_write_byte(0x00db, 0xce, device_base_address);
    i2c_write_byte(0x00dc, 0x03, device_base_address);
    i2c_write_byte(0x00dd, 0xf8, device_base_address);
    i2c_write_byte(0x009f, 0x00, device_base_address);
    i2c_write_byte(0x00a3, 0x28, device_base_address);
    i2c_write_byte(0x00b7, 0x00, device_base_address);
    i2c_write_byte(0x00bb, 0x28, device_base_address);
    i2c_write_byte(0x00b2, 0x09, device_base_address);
    i2c_write_byte(0x00ca, 0x09, device_base_address);
    i2c_write_byte(0x0198, 0x01, device_base_address);
    i2c_write_byte(0x01b0, 0x17, device_base_address);
    i2c_write_byte(0x01ad, 0x00, device_base_address);
    i2c_write_byte(0x00ff, 0x05, device_base_address);
    i2c_write_byte(0x0100, 0x05, device_base_address);
    i2c_write_byte(0x0199, 0x05, device_base_address);
    i2c_write_byte(0x01a6, 0x1b, device_base_address);
    i2c_write_byte(0x01ac, 0x3e, device_base_address);
    i2c_write_byte(0x01a7, 0x1f, device_base_address);
    i2c_write_byte(0x0030, 0x00, device_base_address);

    // Recommended : Public registers - See data sheet for more detail
    i2c_write_byte(0x0011, 0x10, device_base_address); // Enables polling for �New Sample ready� when measurement completes
    i2c_write_byte(0x010a, 0x30, device_base_address); // Set the averaging sample period (compromise between lower noise and increased execution time)
    i2c_write_byte(0x003f, 0x46, device_base_address); // Sets the light and dark gain (upper nibble). Dark gain should not be changed.
    i2c_write_byte(0x0031, 0xFF, device_base_address); // sets the # of range measurements after which auto calibration of system is performed
    i2c_write_byte(0x0040, 0x63, device_base_address); // Set ALS integration time to 100ms
    i2c_write_byte(0x002e, 0x01, device_base_address); // perform a single temperature calibration of the ranging sensor
    i2c_write_byte(0x002c, 0xff, device_base_address); // set SNR limit to 0.06

    // Optional: Public registers - See data sheet for more detail
    i2c_write_byte(0x001b, 0x09, device_base_address); // Set default ranging inter-measurement period to 100ms
    i2c_write_byte(0x003e, 0x31, device_base_address); // Set default ALS inter-measurement period to 500ms
    i2c_write_byte(0x0014, 0x24, device_base_address); // Configures interrupt on �New sample ready�
}

sensor_error er_set_static_config(uint8_t device_base_address)
{
    unsigned char module_id, rev_maj, rev_min;
    bool_t module;
    int32_t programmedOffset;

    LOG_FUNCTION_START((void*)&device_base_address);

    module_id = i2c_read_byte(IDENTIFICATION_MODEL_ID, device_base_address);
    // for use with NVM-programmed parts
    rev_maj = i2c_read_byte(IDENTIFICATION_MODULE_REV_MAJOR, device_base_address);
    rev_min = i2c_read_byte(IDENTIFICATION_MODULE_REV_MINOR, device_base_address);

    module = FALSE;
    if ((module_id == 0xb4) && (rev_maj == 0x01) && (rev_min == 0x02))
        module = TRUE;

    if (module)
        er_set_register_tuning(device_base_address);

    // VHV automatically run on parts that are NVM-programmed, ie customer parts!

    if (common_get_system_fresh_out_of_reset(device_base_address))
    {
        // must write the scaler at least once to the device to ensure the scaler is in a known state.
        er_set_scaler(_rangeScaler, device_base_address);

        // for upscaled ranging, recalibrate the part-2-part offset
        programmedOffset = er_range_get_part2Part_range_offset(device_base_address);
        er_set_part2Part_range_offset(device_base_address, (uint8_t)(programmedOffset/_rangeScaler));

        // *** only if using glass ****
        //    xtalk_height = ranging.Range_Get_Crosstalk_Valid_Height(device)
        //    ranging.Range_Set_Crosstalk_Valid_Height(device, int(xtalk_height/_rangeScaler))

        common_clear_system_fresh_out_of_reset(device_base_address);
    }
    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

sensor_error er_set_part2Part_range_offset(uint8_t device_base_address, uint8_t part_to_part_range_offset)
{
    LOG_FUNCTION_START((void*)&device_base_address,(void*)&part_to_part_range_offset);

    i2c_write_byte(SYSRANGE_PART_TO_PART_RANGE_OFFSET, part_to_part_range_offset, device_base_address);

    LOG_FUNCTION_END(NULL);

    return SENSOR_ERROR_NONE;
}

uint8_t er_range_get_part2Part_range_offset(uint8_t device_base_address)
{
    uint8_t ret=0;

    LOG_FUNCTION_START((void*)&device_base_address);
    ret = i2c_read_byte(SYSRANGE_PART_TO_PART_RANGE_OFFSET, device_base_address);
    LOG_FUNCTION_END(ret);

    return ret;
}

uint32_t er_get_result(uint8_t device_base_address)
{
    uint32_t range = 0;

    LOG_FUNCTION_START((void*)&device_base_address);

    range = (uint32_t)(i2c_read_byte(RESULT_RANGE_VAL, device_base_address) * _rangeScaler);

    LOG_FUNCTION_END(range);

    return range;
}


sensor_error er_set_scaler(uint8_t scaler, uint8_t device_base_address)
{
    //const uint16_t cMask        = 0x01ff;
    const uint16_t cScalerVal1X = 253;
    const uint16_t cScalerVal2X = 127;
    const uint16_t cScalerVal3X = 84;
    uint32_t ret=0;
    uint16_t scalerVal = cScalerVal2X;

    LOG_FUNCTION_START((void*)&device_base_address);

    if(scaler == Scale1X)
    {
        _rangeScaler = Scale1X;
        scalerVal    = cScalerVal1X;
    }
    else if(scaler == Scale2X)
    {
        _rangeScaler = Scale2X;
        scalerVal    = cScalerVal2X;
    }
    else
    {
        _rangeScaler = Scale3X;
        scalerVal    = cScalerVal3X;
    }

    i2c_write_word(RANGE_SCALER, scalerVal, device_base_address);

    LOG_FUNCTION_END(ret);

    return ret;
}

uint8_t er_get_scaler(uint8_t device_base_address)
{
    uint8_t ret = _rangeScaler;

    //LOG_FUNCTION_START((void*)&device_base_address);

    //LOG_FUNCTION_END(ret);

    return ret;
}

uint32_t er_get_upper_limit(uint8_t device_base_address)
{
    const uint32_t cScalerMax1X = 255;
    const uint32_t cScalerMax2X = 510;
    const uint32_t cScalerMax3X = 765;
    uint32_t upperLim = cScalerMax1X;

    LOG_FUNCTION_START((void*)&device_base_address);

    if(_rangeScaler == Scale2X)
    {
        upperLim = cScalerMax2X;
    }
    else if(_rangeScaler == Scale3X)
    {
        upperLim = cScalerMax3X;
    }

    LOG_FUNCTION_END(upperLim);

    return upperLim;
}


uint32_t er_get_lower_limit(uint8_t device_base_address)
{
    uint32_t lowerLimit = 0;
    LOG_FUNCTION_START((void*)&device_base_address);

    LOG_FUNCTION_END(lowerLimit);

    return lowerLimit;
}


